\name{selection}
\alias{selection}
\title{selection}
\usage{
  selection(parm, index, model, data = NULL, alpha = 0.05,
  type = c("canonical","mean-value"), method = c("eigen","1d"))
}

\description{
  Finds the envelope eigenspace or dimension that is favored 
  using AIC, BIC, or the LRT at a specified size.
}

\arguments{
  

  \item{parm}{The MLE of the parameter of interest.}

  \item{index}{The indices denoting which components of the 
    canonical parameter vector are parameters of interest. }

  \item{model}{An aster model object. }  

  \item{data}{An asterdata object. }

  \item{alpha}{The desired size of the LRT. }

  \item{type}{The parameterization of the aster model in which 
    envelope methods are being applied. }

  \item{method}{The procedure used to obtain envelope estimators. }    
}

\value{
  \item{aic}{The eigenspace or envelope model dimension favored 
    using AIC. }

  \item{bic}{The eigenspace or envelope model dimension favored 
    using BIC. }

  \item{LRT}{The eigenspace or envelope model dimension favored 
    using the LRT of size \code{alpha}. }

  \item{out}{The output table of all model selection criteria for
    all envelope estimators considered. }
}

\details{
  This function provides the user with the envelope model dimension 
  or indices of the eigenspace favored by AIC, BIC, and the likelihood 
  ratio test of size \code{alpha}. There are four possible combinations 
  of outputs. They are: 
  \enumerate{
  \item[1.] The specification of \code{method = "eigen"} and 
    \code{type = "mean-value"} provides the user with the indices of 
    the eigenspace of estimated Fisher information used to construct an
    envelope estimator for \eqn{\tau}  favored by AIC, BIC, and the LRT of 
    size \code{alpha}.
  \item[2.] The specification of \code{method = "eigen"} and 
    \code{type = "canonical"} provides the user with the indices of 
    the eigenspace of estimated Fisher information used to construct an
    envelope estimator for \eqn{\beta} favored by AIC, BIC, and the LRT of 
    size \code{alpha}.
  \item[3.] The specification of \code{method = "1d"} and 
    \code{type = "mean-value"} provides the user with the envelope model 
    dimension used to construct an envelope estimator for \eqn{\tau} favored 
    by AIC, BIC, and the LRT of size \code{alpha}.
  \item[4.] The specification of \code{method = "1d"} and 
    \code{type = "canonical"} provides the user with the envelope model 
    dimension used to construct an envelope estimator for \eqn{\beta} favored 
    by AIC, BIC, and the LRT of size \code{alpha}.
  }

  When one is interested in envelope model dimensions or eigenspaces with 
  respect to \eqn{\beta}, then an \code{asterdata} object does not need to 
  be specified. On the other hand, an \code{asterdata} is needed in order to 
  map the estimated \eqn{\tau} to its corresponding \eqn{\beta} value. This 
  is necessary because of the interface (or lack thereof) between current 
  \code{aster} and \code{aster2} software. The way in which aster model 
  log likelihoods are evaluated is incorporated in \code{aster} software 
  and changing parameterizations is carried out using \code{aster2} software.
}

\references{
  Cook, R.D. and Zhang, X. (2014).
  Foundations for Envelope Models and Methods.
  \emph{JASA}, In Press.

  Cook, R.D. and Zhang, X. (2015).
  Algorithms for Envelope Estimation.
  \emph{Journal of Computational and Graphical Statistics}, 
  Published online. \doi{10.1080/10618600.2015.1029577}.
  
  Eck, D. J., Geyer, C. J., and Cook, R. D. (2015).
  Enveloping the aster model.
  \emph{in preparation}.
}

\examples{
\dontrun{
  set.seed(13)
  library(envlpaster)
  library(aster2)
  data(generateddata)
  m.null <- aster(resp ~ 0 + varb, fam = fam, pred = pred, 
    varvar = varb, idvar = id, root = root, data = redata)
  m1 <- aster(resp ~ 0 + varb + mass + timing, 
    fam = fam, pred = pred, varvar = varb, idvar = id,
    root = root, data = redata)
  m2 <- aster(resp ~ 0 + varb + mass + timing + 
    I(mass^2) + I(timing^2) + I(mass*timing), 
    fam = fam, pred = pred, varvar = varb, idvar = id,
    root = root, data = redata)
  anova.table <- anova(m.null,m1,m2); anova.table
  beta <- m1$coef
  a <- grepl( "offsp", names(beta)) 
  a <- a + grepl( "surviv", names(beta)) 
  b <- which(a == 1)
  target <- c(1:length(beta))[-b]
  nnode <- ncol(m1$x)
  data.aster <- asterdata(data, vars, pred, rep(0,nnode),
    fam, families = list("bernoulli", "poisson", 
    fam.zero.truncated.poisson()))
  selection(parm  = beta, index = target, model = m1, 
    data = data.aster, alpha = 0.05, type = "canonical", 
    method = "eigen")
  }
}
