% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimatr_difference_in_means.R
\name{difference_in_means}
\alias{difference_in_means}
\title{Design-based difference-in-means estimator}
\usage{
difference_in_means(formula, data, blocks, clusters, weights, subset,
  condition1 = NULL, condition2 = NULL, ci = TRUE, alpha = 0.05)
}
\arguments{
\item{formula}{an object of class formula, as in \code{\link{lm}}, such as
\code{Y ~ Z} with only one variable on the right-hand side, the treatment.}

\item{data}{A \code{data.frame}.}

\item{blocks}{An optional bare (unquoted) name of the block variable. Use
for blocked designs only.}

\item{clusters}{An optional bare (unquoted) name of the variable that
corresponds to the clusters in the data; used for cluster randomized
designs. For blocked designs, clusters must nest within blocks.}

\item{weights}{the bare (unquoted) names of the weights variable in the
supplied data.}

\item{subset}{An optional bare (unquoted) expression specifying a subset of
observations to be used.}

\item{condition1}{value in the treatment vector of the condition
to be the control. Effects are
estimated with \code{condition1} as the control and \code{condition2} as the
treatment. If unspecified, \code{condition1} is the "first" condition and
\code{condition2} is the "second" according to levels if the treatment is a
factor or according to a sortif it is a numeric or character variable (i.e
if unspecified and the treatment is 0s and 1s, \code{condition1} will by
default be 0 and \code{condition2} will be 1). See the examples for more.}

\item{condition2}{value in the treatment vector of the condition to be the
treatment. See \code{condition1}.}

\item{ci}{logical. Whether to compute and return p-values and
confidence intervals, TRUE by default.}

\item{alpha}{The significance level, 0.05 by default.}
}
\value{
Returns an object of class \code{"difference_in_means"}.

The post-estimation commands functions \code{summary} and \code{\link{tidy}}
return results in a \code{data.frame}. To get useful data out of the return,
you can use these data frames, you can use the resulting list directly, or
you can use the generic accessor functions \code{coef} and
\code{confint}.

An object of class \code{"difference_in_means"} is a list containing at
least the following components:
  \item{coefficients}{the estimated coefficients}
  \item{se}{the estimated standard errors}
  \item{df}{the estimated degrees of freedom}
  \item{p}{the p-values from a two-sided t-test using \code{coefficients}, \code{se}, and \code{df}}
  \item{ci_lower}{the lower bound of the \code{1 - alpha} percent confidence interval}
  \item{ci_upper}{the upper bound of the \code{1 - alpha} percent confidence interval}
  \item{coefficient_name}{a character vector of coefficient names}
  \item{alpha}{the significance level specified by the user}
  \item{N}{the number of observations used}
  \item{outcome}{the name of the outcome variable}
  \item{design}{the name of the design learned from the arguments passed}
}
\description{
Difference-in-means estimators that selects the appropriate
point estimate, standard errors, and degrees of freedom for a variety of
designs: unit randomized, cluster randomized, block randomized,
block-cluster randomized, matched-pairs, and matched-pair cluster
randomized designs
}
\details{
This function implements a difference-in-means estimator, with
support for blocked, clustered, matched-pairs, block-clustered, and
matched-pair clustered designs. One specifies their design by passing
the blocks and clusters in their data and this function chooses which
estimator is most appropriate.

If you pass only \code{blocks}, if all blocks are of size two, we will
infer that the design is a matched-pairs design. If they are all size four
or larger, we will infer that it is a regular blocked design. If you pass
both \code{blocks} and \code{clusters}, we will similarly
infer whether it is a matched-pairs clustered design or a block-clustered
design the number of clusters per block. If the user passes only
\code{clusters}, we will infer that the design was cluster-randomized. If
the user specifies neither the \code{blocks} nor the \code{clusters},
a regular Welch's t-test will be performed.

Importantly, if the user specifies weights, the estimation is handed off
to \code{\link{lm_robust}} with the appropriate robust standard errors
as weighted difference-in-means estimators are not implemented here.
More details of the about each of the estimators can be found in the
\href{estimatr.declaredesign.org/articles/technical-notes.html}{technical notes}.
}
\examples{

library(fabricatr)
library(randomizr)
# Get appropriate standard errors for unit-randomized designs

# ----------
# 1. Unit randomized
# ----------
dat <- fabricate(
  N = 100,
  Y = rnorm(100),
  Z_comp = complete_ra(N, prob = 0.4),
)

table(dat$Z_comp)
difference_in_means(Y ~ Z_comp, data = dat)

# ----------
# 2. Cluster randomized
# ----------
# Accurates estimates and standard errors for clustered designs
dat$clust <- sample(20, size = nrow(dat), replace = TRUE)
dat$Z_clust <- cluster_ra(dat$clust, prob = 0.6)

table(dat$Z_clust, dat$clust)
summary(difference_in_means(Y ~ Z_clust, clusters = clust, data = dat))

# ----------
# 3. Block randomized
# ----------
dat$block <- rep(1:10, each = 10)
dat$Z_block <- block_ra(dat$block, prob = 0.5)

table(dat$Z_block, dat$block)
difference_in_means(Y ~ Z_block, blocks = block, data = dat)

# ----------
# 4. Block cluster randomized
# ----------
# Learns this design if there are two clusters per block
dat$small_clust <- rep(1:50, each = 2)
dat$big_blocks <- rep(1:5, each = 10)

dat$Z_blcl <- block_and_cluster_ra(
  blocks = dat$big_blocks,
  clusters = dat$small_clust
 )

difference_in_means(
  Y ~ Z_blcl,
  blocks = big_blocks,
  clusters = small_clust,
  data = dat
 )

# ----------
# 5. Matched-pairs
# ----------
# Matched-pair estimates and standard errors are also accurate
# Specified same as blocked design, function learns that
# it is matched pair from size of blocks!
dat$pairs <- rep(1:50, each = 2)
dat$Z_pairs <- block_ra(dat$pairs, prob = 0.5)

table(dat$pairs, dat$Z_pairs)
difference_in_means(Y ~ Z_pairs, blocks = pairs, data = dat)

# ----------
# 6. Matched-pair cluster randomized
# ----------
# Learns this design if there are two clusters per block
dat$small_clust <- rep(1:50, each = 2)
dat$cluster_pairs <- rep(1:25, each = 4)
table(dat$cluster_pairs, dat$small_clust)

dat$Z_mpcl <- block_and_cluster_ra(
  blocks = dat$cluster_pairs,
  clusters = dat$small_clust
 )

difference_in_means(
  Y ~ Z_mpcl,
  blocks = cluster_pairs,
  clusters = small_clust,
  data = dat
 )

# ----------
# Other examples
# ----------

# Also works with multi-valued treatments if users specify
# comparison of interest
dat$Z_multi <- simple_ra(
  nrow(dat),
  condition_names = c("Treatment 2", "Treatment 1", "Control"),
  prob_each = c(0.4, 0.4, 0.2)
)

# Only need to specify which condition is treated `condition2` and
# which is control `condition1`
difference_in_means(
  Y ~ Z_multi,
  condition1 = "Treatment 2",
  condition2 = "Control",
  data = dat
)
difference_in_means(
  Y ~ Z_multi,
  condition1 = "Treatment 1",
  condition2 = "Control",
  data = dat
)

# Specifying weights will result in estimation via lm_robust()
dat$w <- runif(nrow(dat))
difference_in_means(Y ~ Z_comp, weights = w, data = dat)
lm_robust(Y ~ Z_comp, weights = w, data = dat)

}
\references{
Gerber, Alan S, and Donald P Green. 2012. Field Experiments: Design, Analysis, and Interpretation. New York: W.W. Norton.

Imai, Kosuke, Gary King, Clayton Nall. 2009. "The Essential Role of Pair Matching in Cluster-Randomized Experiments, with Application to the Mexican Universal Health Insurance Evaluation." Statistical Science 24 (1). Institute of Mathematical Statistics: 29-53. \url{https://doi.org/10.1214/08-STS274}.
}
\seealso{
\code{\link{lm_lin}}
}
