% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/va.R
\name{va}
\alias{va}
\title{Visual acuity notation conversion}
\usage{
va(
  x,
  to = "logmar",
  type = NULL,
  from_logmar = TRUE,
  logmarstep = FALSE,
  mixed = FALSE
)
}
\arguments{
\item{x}{Vector with visual acuity entries. Must be atomic.
Snellen fractions need to be entered with "/"}

\item{to}{To which class to convert. "etdrs", "logmar" or "snellen" -
any case allowed}

\item{type}{To which Snellen notation to convert: "m", "dec" or "ft"}

\item{from_logmar}{chose logmar when guessing between two notations (logmar
vs. snellen decimal or logmar vs. etdrs)}

\item{logmarstep}{how +/- entries are evaluated. FALSE:
increase/decrease Snellen fractions by lines. TRUE: plus/minus
entries equivalent to 0.02 logmar or 1 ETDRS letter}

\item{mixed}{TRUE Elements will be converted one by one.
Most plausibility checks will be overruled!}
}
\value{
vector of \code{va} class. See also "VA classes"
}
\description{
Cleans and converts visual acuity notations (classes)
between Snellen (decimal, meter and feet), ETDRS, and logMAR.
\code{va} detects the VA class and will convert to logMAR as default.
}
\details{
Each class can be converted from one to another, and va()
converts to logMAR by default. In case of ambiguous detection,
logMAR is selected as default, or the other alternative is selected with
\code{from_logmar = FALSE}.
}
\section{VA conversion}{

\itemize{
\item \strong{logMAR to ETDRS}: logMAR rounded to the first digit and converted with
the chart.
\item \strong{Snellen to logMAR}: logMAR = -1 * log10(snellen_frac)
\item \strong{Snellen to ETDRS}: ETDRS = 85 + 50 * log10(snellen_frac)
\href{https://doi.org/10.1097/iae.0b013e3181d87e04}{Gregori et al.}.
\item \strong{ETDRS to logMAR}: logMAR = -0.02 * etdrs + 1.7
\href{https://doi.org/10.1016/s0002-9394(02)01825-1}{Beck et al.}
\item \strong{Hand movements and counting fingers} are converted following
\href{https://doi.org/10.1167/iovs.05-0981}{Schulze-Bonsel et al.}
\item \strong{(No) light perception} are converted following the suggestions by
\href{https://michaelbach.de/sci/acuity.html}{Michael Bach}
\item \strong{To Snellen}:
Although there seems to be no good statistical reason to convert
back to Snellen, it is a very natural thing to eye specialists to think
in Snellen. A conversion to snellen gives a good gauge of how the visual
acuity for the patients are. However, back-conversion should not be
considered an exact science and any attempt to use formulas will result
in very weird Snellen values that have no correspondence to common charts.
Therefore, Snellen matching the nearest ETDRS and logMAR value in
the \link{va_chart} are used.
}
}

\section{Accepted VA formats}{

\itemize{
\item Snellen fractions (meter/ feet) need to be entered as fraction with
"/".
\item \strong{when converting to ETDRS or logMAR}: any fraction is allowed ,
e.g. 3/60 and 2/200 will also be recognized.
\item \strong{When converting between Snellen fractions}:
\emph{has to be either 6/ or 20/}. Other fractions will not be recognized -
see \strong{"Examples"}
\item ETDRS must be integer-equivalent between 0 and 100 (integer equivalent
means, it can also be a character vector)
\item logMAR must be between -0.3 and 3.0
\item Qualitative must be either of PL, LP, NLP, NPL, HM, CF (any case allowed)
\item Any element which is not recognized will be converted to NA
\item Vectors containing several notations ("mixed") are guessed and converted
element by element with \link{which_va_dissect} and \link{va_dissect}
}
}

\section{VA detection}{

\itemize{
\item Internally done with \code{\link[=which_va]{which_va()}} based on the following rules
\item if x integer and 3 < x <= 100: \code{etdrs}
\item if x integer and 0 <= x <= 3: \code{logmar}, but you can choose \code{etdrs}
\item if x numeric and -0.3 <= x <= 3: \code{logmar}
\item if x numeric and all x in intersection(va_chart$logMAR, va_chart$snellen_dec):
\code{logmar}, but you can choose \code{snellen}
\item \emph{non-mixed class}: if all x in va_chart$snellen_dec: \code{snellen}
\item \emph{mixed class} (\link{which_va_dissect}): snellen_dec not supported.
\item if character and format x/y: \code{snellen} (fraction)
\item if one of "CF", "HM", "LP", "PL", "NLP", or "NPL": \code{quali}
\item if numeric x beyond the ranges from above: \code{NA}
\item Any other string or NA: NA
}

Detection and conversion is on a vector as a whole by \code{\link[=which_va]{which_va()}}. If a
"mixed" VA notation
is found, \code{\link[=which_va_dissect]{which_va_dissect()}} and \code{\link[=va_dissect]{va_dissect()}} will be called instead
for each VA vector element individually.
}

\section{Problematic cases}{

There can be ambiguous cases for detection (detection defaults to logmar):
x is one of 0,1,2,3 - This can be ETDRS and logMAR.
x is one of c(1.5, 1, 0.8, 0.5, 0.4, 0.3, 0.2, 0.1, 0) - This can be
snellen decimal or logMAR.

\strong{snellen decimals} are a particular challenge and \code{va} may wrongly
assign  logMAR - this could happen if there are unusual snellen decimal
values in the data which are not part of \link{va_chart}. E.g., check the
values with \code{unique(x)}.
}

\section{Snellen "+/-" entries}{

By default, plus/minus entries are evaluated as intended by the
test design: Snellen fractions increase/decrease only by lines.\preformatted{- if entry -1 to +3 : take same Snellen value
- if <= -2 : take Snellen value one line below
- if >+3 (unlikely, but unfortunately not impossible):
}

If logmarstep = TRUE, each snellen optotype will be considered
equivalent to 0.02 logmar or 1 ETDRS letter (assuming 5 letters
in a row in a chart)
}

\section{VA cleaning}{

For more details see \code{\link[=clean_va]{clean_va()}}
\enumerate{
\item \code{NA} is assigned to strings such as "." or "", "n/a" or "   "
\item notation for qualitative entries is simplified.
}
}

\section{VA classes}{

convert_VA returns a vector of three classes:
\enumerate{
\item \code{va}
\item One of \code{snellen}, \code{logmar}, \code{etdrs} or \code{quali}.
\item Either of \code{character} (for Snellen and qualitative),
\code{numeric} (for logMAR), or \code{integer} (for ETDRS).
}
}

\examples{
## will automatically detect VA class and convert to logMAR by default
## ETDRS letters
x <- c(23, 56, 74, 58)
va(x)

## ... or convert to snellen
va(x, to = "snellen")

## snellen, mixed with categories. Also dealing with those "plus/minus" entries
va(c("NLP", "NPL", "PL", "LP", "HM", "CF", "6/60", "20/200", "6/9",
 "20/40", "20/40+3", "20/50-2"))

## A mix of notations is also possible
x <- c("NLP", "0.8", "34", "3/60", "2/200", "20/40+3", "20/50-2")
va(x)

## Any fraction is possible, and empty values
x <- c("CF", "3/60", "2/200", "", "20/40+3", ".", "      ")
va(x)

## but this not any fraction when converting from one class to the other
x <- c("3/60", "2/200", "6/60", "20/200", "6/9")
va(x, to="snellen", type = "m")
}
\seealso{
Other VA converter: 
\code{\link{snellen_steps}},
\code{\link{va_dissect}()},
\code{\link{va_methods}},
\code{\link{which_va}()}

Other VA cleaner: 
\code{\link{clean_va}()}
}
\concept{Ophthalmic functions}
\concept{VA cleaner}
\concept{VA converter}
