% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spotrate-class.R
\docType{class}
\name{SpotRate-class}
\alias{SpotRate-class}
\title{SpotRate class}
\description{
The \code{SpotRate} class abstracts the interst rate and has methods
to handle many calculations on it.
}
\details{
The \code{SpotRate} class fully specifies spot rates.
It has:
\itemize{
\item the spot rate values which are numeric values representing the rate.
\item the compounding regime that specifies how to compound the spot
rate. This is a \code{Compounding} object.
\item the daycount rule to compute the compounding periods right
adjusted to the spot rate frequency.
\item the calendar according to which the number of days are counted.
}

The \code{SpotRate} class is a \code{numeric}, that represents the
interest rate and that has the slots: \code{compounding}, \code{daycount}
and \code{calendar}.

For example, an annual simple interest rate of 6\%, that compounds in
calendar days, is defined as follows:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{sr_simple <- spotrate(0.06, "simple", "actual/360", "actual")
sr_simple
#> [1] "0.06 simple actual/360 actual"
}\if{html}{\out{</div>}}

\code{actual/360} is the daycount rule and \code{actual} is the calendar.

Differently, an annual compound interest rate of 10\%, that compounds
in business days according to calendar \code{Brazil/ANBIMA} is

\if{html}{\out{<div class="sourceCode r">}}\preformatted{sr_disc <- spotrate(0.1, "discrete", "business/252", "Brazil/ANBIMA")
sr_disc
#> [1] "0.1 discrete business/252 Brazil/ANBIMA"
}\if{html}{\out{</div>}}

The \code{calendar} slot is a \code{bizdays} calendar.

An $100,000 investment in an instrument that pays that interst rate for 5
years has the future value.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{100000 * compound(sr_disc, term(5, "years"))
#> [1] 161051
}\if{html}{\out{</div>}}

for the simple interest rate we have

\if{html}{\out{<div class="sourceCode r">}}\preformatted{100000 * compound(sr_simple, term(5, "years"))
#> [1] 130000
}\if{html}{\out{</div>}}

SpotRate objects can be created with vectors

\if{html}{\out{<div class="sourceCode r">}}\preformatted{rates <- c(1.69, 0.16, 0.07, 0.72, 0.10, 1.60, 0.18, 1.56, 0.60, 1.69)
sr_vec <- spotrate(rates, "discrete", "business/252", "Brazil/ANBIMA")
sr_vec
#>  [1] "1.69 discrete business/252 Brazil/ANBIMA"
#>  [2] "0.16 discrete business/252 Brazil/ANBIMA"
#>  [3] "0.07 discrete business/252 Brazil/ANBIMA"
#>  [4] "0.72 discrete business/252 Brazil/ANBIMA"
#>  [5] "0.10 discrete business/252 Brazil/ANBIMA"
#>  [6] "1.60 discrete business/252 Brazil/ANBIMA"
#>  [7] "0.18 discrete business/252 Brazil/ANBIMA"
#>  [8] "1.56 discrete business/252 Brazil/ANBIMA"
#>  [9] "0.60 discrete business/252 Brazil/ANBIMA"
#> [10] "1.69 discrete business/252 Brazil/ANBIMA"
}\if{html}{\out{</div>}}

and can be put into a \code{data.frame}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{data.frame(spot_rate = sr_vec)
#>                                   spot_rate
#> 1  1.69 discrete business/252 Brazil/ANBIMA
#> 2  0.16 discrete business/252 Brazil/ANBIMA
#> 3  0.07 discrete business/252 Brazil/ANBIMA
#> 4  0.72 discrete business/252 Brazil/ANBIMA
#> 5  0.10 discrete business/252 Brazil/ANBIMA
#> 6  1.60 discrete business/252 Brazil/ANBIMA
#> 7  0.18 discrete business/252 Brazil/ANBIMA
#> 8  1.56 discrete business/252 Brazil/ANBIMA
#> 9  0.60 discrete business/252 Brazil/ANBIMA
#> 10 1.69 discrete business/252 Brazil/ANBIMA
}\if{html}{\out{</div>}}

once in a \code{data.frame}, dplyr verbs can be used to manipulate it.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{require(dplyr, warn.conflicts = FALSE)
#> Loading required package: dplyr

data.frame(spot_rate = sr_vec) |>
   mutate(comp = compound(spot_rate, term(5, "months")))
#>                                   spot_rate     comp
#> 1  1.69 discrete business/252 Brazil/ANBIMA 1.510301
#> 2  0.16 discrete business/252 Brazil/ANBIMA 1.063794
#> 3  0.07 discrete business/252 Brazil/ANBIMA 1.028592
#> 4  0.72 discrete business/252 Brazil/ANBIMA 1.253536
#> 5  0.10 discrete business/252 Brazil/ANBIMA 1.040512
#> 6  1.60 discrete business/252 Brazil/ANBIMA 1.489037
#> 7  0.18 discrete business/252 Brazil/ANBIMA 1.071398
#> 8  1.56 discrete business/252 Brazil/ANBIMA 1.479449
#> 9  0.60 discrete business/252 Brazil/ANBIMA 1.216326
#> 10 1.69 discrete business/252 Brazil/ANBIMA 1.510301
}\if{html}{\out{</div>}}

SpotRate is \code{numeric}, so it executes arithmetic and comparison operations
with \code{numeric} objects.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{data.frame(spot_rate = sr_vec) |>
   mutate(
     new_spot_rate = spot_rate + 0.02,
     check_gt_1pp = spot_rate > 0.01,
     check_gt_nsr = spot_rate > new_spot_rate
   )
#>                                   spot_rate
#> 1  1.69 discrete business/252 Brazil/ANBIMA
#> 2  0.16 discrete business/252 Brazil/ANBIMA
#> 3  0.07 discrete business/252 Brazil/ANBIMA
#> 4  0.72 discrete business/252 Brazil/ANBIMA
#> 5  0.10 discrete business/252 Brazil/ANBIMA
#> 6  1.60 discrete business/252 Brazil/ANBIMA
#> 7  0.18 discrete business/252 Brazil/ANBIMA
#> 8  1.56 discrete business/252 Brazil/ANBIMA
#> 9  0.60 discrete business/252 Brazil/ANBIMA
#> 10 1.69 discrete business/252 Brazil/ANBIMA
#>                               new_spot_rate check_gt_1pp check_gt_nsr
#> 1  1.71 discrete business/252 Brazil/ANBIMA         TRUE        FALSE
#> 2  0.18 discrete business/252 Brazil/ANBIMA         TRUE        FALSE
#> 3  0.09 discrete business/252 Brazil/ANBIMA         TRUE        FALSE
#> 4  0.74 discrete business/252 Brazil/ANBIMA         TRUE        FALSE
#> 5  0.12 discrete business/252 Brazil/ANBIMA         TRUE        FALSE
#> 6  1.62 discrete business/252 Brazil/ANBIMA         TRUE        FALSE
#> 7  0.20 discrete business/252 Brazil/ANBIMA         TRUE        FALSE
#> 8  1.58 discrete business/252 Brazil/ANBIMA         TRUE        FALSE
#> 9  0.62 discrete business/252 Brazil/ANBIMA         TRUE        FALSE
#> 10 1.71 discrete business/252 Brazil/ANBIMA         TRUE        FALSE
}\if{html}{\out{</div>}}

SpotRate vectors also are created with the concatenation function \code{c}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{c(sr_disc, 0.1, 0.13, 0.14, 0.15)
#> [1] "0.10 discrete business/252 Brazil/ANBIMA"
#> [2] "0.10 discrete business/252 Brazil/ANBIMA"
#> [3] "0.13 discrete business/252 Brazil/ANBIMA"
#> [4] "0.14 discrete business/252 Brazil/ANBIMA"
#> [5] "0.15 discrete business/252 Brazil/ANBIMA"
}\if{html}{\out{</div>}}

Furtherly, all indexing operations of numeric objects are supported by
SpotRate objects.
\subsection{Invalid Operations}{

Operations involving SpotRate objects with different \code{compounding},
\code{daycount} or \code{calendar}, raise errors.

This happens with the following operations:
\itemize{
\item Compare: >, <, <=, >=
\item Arithmetic: +, -, *, /
\item Concatenation: \code{c}
}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{try(sr_simple + sr_disc)
#> Error in stop_if_spotrate_slots_differ(e1, e2, "SpotRate objects have different slots") : 
#>   SpotRate objects have different slots
try(sr_simple > sr_disc)
#> Error in stop_if_spotrate_slots_differ(e1, e2, "SpotRate objects have different slots") : 
#>   SpotRate objects have different slots
try(c(sr_simple, sr_disc))
#> Error in stop_if_spotrate_slots_differ(x, values_, "SpotRate objects have different slots") : 
#>   SpotRate objects have different slots
}\if{html}{\out{</div>}}
}
}
\note{
The \code{SpotRate} objects are annual rates.
}
