% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/checkPairwise.R
\name{checkPairwise}
\alias{checkPairwise}
\alias{plotCP}
\title{Check pedigree data for relationship errors}
\usage{
checkPairwise(
  x,
  ids = typedMembers(x),
  includeInbred = FALSE,
  acrossComps = TRUE,
  plotType = c("base", "ggplot2", "plotly", "none"),
  GLRthreshold = 1000,
  pvalThreshold = NULL,
  nsim = 0,
  seed = NULL,
  plot = TRUE,
  verbose = TRUE,
  excludeInbred = NULL,
  ...
)

plotCP(
  cpRes = NULL,
  plotType = c("base", "ggplot2", "plotly"),
  labels = FALSE,
  errtxt = "Potential error",
  seed = NULL,
  ...
)
}
\arguments{
\item{x}{A \code{ped} object or a list of such.}

\item{ids}{A vector of ID labels; the individuals to include in the check.
Default: All typed members of \code{x}.}

\item{includeInbred}{A logical, by default FALSE, indicating if inbred
individuals should be excluded from the analysis.}

\item{acrossComps}{A logical indicating if pairs of individuals in different
components should be considered. Default: TRUE.}

\item{plotType}{Either "base" (default), "ggplot2", "plotly" or "none".
Abbreviations are allowed.}

\item{GLRthreshold}{A positive number, by default 1000. Threshold for the
generalised likelihood ratio (see Details). Scores exceeding this are
flagged as potential errors in the output table and encircled in the plot.}

\item{pvalThreshold}{A positive number, or NULL (default). If given, this is
used instead of \code{GLRthreshold} to identify potential errors. Ignored if
\code{nsim = 0}.}

\item{nsim}{A nonnegative number; the number of simulations used to estimate
p-values. If 0 (default), this step is skipped.}

\item{seed}{An integer seed for the random number generator (optional, and
only relevant if \code{nsim > 0}).}

\item{plot}{Deprecated. To suppress the triangle plot, use \code{plotType = "none"}.}

\item{verbose}{A logical.}

\item{excludeInbred}{Deprecated; renamed to ´includeInbred´.}

\item{...}{Further parameters passed on to \code{\link[ribd:ibdTriangle]{ribd::ibdTriangle()}}.}

\item{cpRes}{A data frame: the output from \code{checkPairwise()}.}

\item{labels}{A logical (default: FALSE). If TRUE, labels are included in the
IBD triangle plot.}

\item{errtxt}{A character string to use for the error legend.}
}
\value{
If \code{plotType} is "none" or "base": A data frame containing both the
estimated and pedigree-based IBD coefficients for each pair of typed
individuals. The last columns (\code{GLR}, \code{pval} and \code{err}) contain test
results using the GLR scores to identify potential pedigree errors.

If \code{plotType} is "ggplot2" or "plotly", the plot objects are returned.
}
\description{
The \code{checkPairwise()} function provides a convenient way to check for
pedigree errors, given the available marker data. The function calls
\code{\link[=ibdEstimate]{ibdEstimate()}} to estimate IBD coefficients for all pairs of typed pedigree
members, and uses the estimates to test for potential errors. By default, the
results are shown in a colour-coded plot (based on \code{\link[ribd:ibdTriangle]{ribd::ibdTriangle()}})
where unlikely relationships are easy to spot.
}
\details{
To identify potential pedigree errors, the function calculates the
\emph{generalised likelihood ratio} (GLR) of each pairwise relationship.
This compares the likelihood of the estimated coefficients with that of the
coefficients implied by the pedigree. By default, relationships whose GLR
exceed 1000 are flagged as errors and shown with a circle in the plot.
Alternatively, if arguments \code{nsim} and \code{pvalThreshold} are supplied, the
p-value of each score is estimated by simulation, and used as threshold for
calling errors.

By default, inbred individuals are excluded from the analysis, since pairwise
relationships involving inbred individuals have undefined kappa coefficients
(and therefore no position in the triangle). In some cases it may still be
informative to include their estimates; set \code{includeInbred = TRUE} to
enforce this.
}
\examples{

### Example with realistic data

x = avuncularPed() |>
  profileSim(markers = NorwegianFrequencies, seed = 1729)

checkPairwise(x)

### Create an error: sample swap 1 <-> 3
als = getAlleles(x)
als[c(1,3), ] = als[c(3,1), ]
y = setAlleles(x, alleles = als)

checkPairwise(y)

# Using p-values instead of GLR
nsim = 10 # increase!
checkPairwise(y, nsim = nsim, pvalThreshold = 0.05)

# Plot can be done separately
res = checkPairwise(y, nsim = nsim, pvalThreshold = 0.05, plotType = "none")
plotCP(res, plotType = "base", errtxt = "Not good!")

\donttest{
# Combined plot of pedigree and check results
dev.new(height = 5, width = 8, noRStudioGD = TRUE)
layout(rbind(1:2), widths = 2:3)
plot(y, margins = 2, title = "Swapped 1 - 3")
plotCP(res, labels = TRUE)
}

}
\seealso{
\code{\link[=ibdEstimate]{ibdEstimate()}}.
}
