% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predict.emfrail.R
\name{predict.emfrail}
\alias{predict.emfrail}
\title{Predicted hazard and survival curves from an \code{emfrail} object}
\usage{
\method{predict}{emfrail}(object, newdata = NULL, lp = NULL,
  quantity = c("cumhaz", "survival"), type = c("conditional", "marginal"),
  conf_int = c("regular", "adjusted"), individual = FALSE, ...)
}
\arguments{
\item{object}{An \code{emfrail} fit object}

\item{newdata}{A data frame with the same variable names as those that appear in the \code{emfrail} formula, used to calculate the \code{lp} (optional).}

\item{lp}{A vector of linear predictor values at which to calculate the curves. Default is 0 (baseline).}

\item{quantity}{Can be \code{"cumhaz"} and/or \code{"survival"}. The quantity to be calculated for the values of \code{lp}.}

\item{type}{Can be \code{"conditional"} and/or \code{"marginal"}. The type of the quantity to be calculated.}

\item{conf_int}{Can be \code{"regular"} and/or \code{"adjusted"}. The type of confidence interval to be calculated.}

\item{individual}{Logical. Are the observations in \code{newdata} from the same individual? See details.}

\item{...}{Ignored}
}
\value{
The return value is a single data frame (if \code{lp} has length 1,
\code{newdata} has 1 row or \code{individual == TRUE}) or a list of data frames corresponding to each value of
\code{lp} or each row of \code{newdata} otherwise.
The names of the columns in the returned data frames are as follows: \code{time} represents the unique event time points
from the data set, \code{lp} is the value of the linear predictor (as specified in the input or as calculated from the lines of \code{newdata}).
By default, for each \code{lp} a data frame will contain the following columns: \code{cumhaz}, \code{survival},
\code{cumhaz_m}, \code{survival_m} for the cumulative hazard and survival, conditional and marginal, with corresponding confidence
bands. The naming of the columns is explained more in the Details section.
}
\description{
Predicted hazard and survival curves from an \code{emfrail} object
}
\details{
The function calculates predicted cumulative hazard and survival curves for given covariate
or linear predictor values; for the first, \code{newdata} must be specified and for the latter
\code{lp} must be specified. Each row of \code{newdata} or element of \code{lp} is consiered to be
a different subject, and the desired predictions are produced for each of them separately.

In \code{newdata} two columns may be specified with the names \code{tstart} and \code{tstop}.
In this case, each subject is assumed to be at risk only during the times specified by these two values.
If the two are not specified, the predicted curves are produced for a subject that is at risk for the
whole follow-up time.

A slightly different behaviour is observed if \code{individual == TRUE}. In this case, all the rows of
\code{newdata} are assumed to come from the same individual, and \code{tstart} and \code{tstop} must
be specified, and must not overlap. This may be used for describing subjects that
are not at risk during certain periods or subjects with time-dependent covariate values.

The two "quantities" that can be returned are
named \code{cumhaz} and \code{survival}. If we denote each quantity with \code{q}, then the columns with the marginal estimates
are named \code{q_m}. The confidence intervals contain the name of the quantity (conditional or marginal) followed by \code{_l} or \code{_r} for
the lower and upper bound. The bounds calculated with the adjusted standard errors have the name of the regular bounds followed by
\code{_a}. For example, the adjusted lower bound for the marginal survival is in the column named \code{survival_m_l_a}.

The \code{emfrail} only gives the Breslow estimates of the  baseline hazard \eqn{\lambda_0(t)} at the
event time points, conditional on the frailty. Let \eqn{\lambda(t)} be the baseline hazard for a linear predictor of interest.
The estimated conditional cumulative hazard is then
\eqn{\Lambda(t) = \sum_{s= 0}^t \lambda(s)}. The variance of \eqn{\Lambda(t)} can be calculated from the (maybe adjusted)
variance-covariance matrix.

The conditional survival is obtained by the usual expression \eqn{S(t) = \exp(-\Lambda(t))}. The marginal survival
is given by
\deqn{\bar S(t) = E \left[\exp(-\Lambda(t)) \right] = \mathcal{L}(\Lambda(t)),}
i.e. the Laplace transform of the frailty distribution calculated in \eqn{\Lambda(t)}.

The marginal hazard is obtained as \deqn{\bar \Lambda(t) = - \log \bar S(t).}

The only standard errors that are available from \code{emfrail} are those for \eqn{\lambda_0(t)}. From this,
standard errors of \eqn{\log \Lambda(t)} may be calculated. On this scale, the symmetric confidence intervals are built, and then
moved to the desired scale.
}
\note{
The linear predictor is taken as fixed, so the variability in the estimation of the regression coefficient is not taken into account.
Does not support left truncation (at the moment). That is because, if \code{individual == TRUE} and \code{tstart} and \code{tstop} are
specified, for the marginal estimates the distribution of the frailty is used to calculate the integral, and not
the distribution of the frailty given the truncation.
}
\examples{
kidney$sex <- ifelse(kidney$sex == 1, "male", "female")
m1 <- emfrail(formula = Surv(time, status) ~  sex + age  + cluster(id),
              data =  kidney)

# get all the possible prediction for the value 0 of the linear predictor
predict(m1, lp = 0)

# get the cumulative hazards for two different values of the linear predictor
predict(m1, lp = c(0, 1), quantity = "cumhaz", conf_int = NULL)

# get the cumulative hazards for a female and for a male, both aged 30
newdata1 <- data.frame(sex = c("female", "male"),
                       age = c(30, 30))

predict(m1, newdata = newdata1, quantity = "cumhaz", conf_int = NULL)

# get the cumulative hazards for an individual that changes
# sex from female to male at time 40.
newdata2 <- data.frame(sex = c("female", "male"),
                      age = c(30, 30),
                      tstart = c(0, 40),
                      tstop = c(40, Inf))

predict(m1, newdata = newdata2,
        individual = TRUE,
        quantity = "cumhaz", conf_int = NULL)

}
\seealso{
\code{\link{plot.emfrail}}, \code{\link{autoplot.emfrail}}
}
