% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/batchtools_torque.R
\name{batchtools_torque}
\alias{batchtools_torque}
\title{A batchtools TORQUE backend resolves futures in parallel via a TORQUE/PBS job scheduler}
\usage{
batchtools_torque(
  ...,
  template = "torque",
  scheduler.latency = 1,
  fs.latency = 65,
  resources = list(),
  delete = getOption("future.batchtools.delete", "on-success"),
  workers = getOption("future.batchtools.workers", default = 100L)
)
}
\arguments{
\item{template}{(optional) Name of job-script template to be searched
for by \code{\link[batchtools:findTemplateFile]{batchtools::findTemplateFile()}}. If not found, it defaults to
the \code{templates/torque.tmpl} part of this package (see below).}

\item{scheduler.latency}{[\code{numeric(1)}]\cr
Time to sleep after important interactions with the scheduler to ensure a sane state.
Currently only triggered after calling \code{\link[batchtools]{submitJobs}}.}

\item{fs.latency}{[\code{numeric(1)}]\cr
Expected maximum latency of the file system, in seconds.
Set to a positive number for network file systems like NFS which enables more robust (but also more expensive) mechanisms to
access files and directories.
Usually safe to set to \code{0} to disable the heuristic, e.g. if you are working on a local file system.}

\item{resources}{(optional) A named list passed to the \pkg{batchtools}
job-script template as variable \code{resources}. This is based on how
\code{\link[batchtools:submitJobs]{batchtools::submitJobs()}} works, with the exception for specially
reserved names defined by the \pkg{future.batchtools} package;
\itemize{
\item \code{resources[["asis"]]} is a character vector that are passed as-is to
the job script and are injected as job resource declarations.
\item \code{resources[["modules"]]} is character vector of Linux environment
modules to be loaded.
\item \code{resources[["startup"]]} and \code{resources[["shutdown"]]} are character
vectors of shell code to be injected to the job script as-is.
\item \code{resources[["details"]]}, if TRUE, results in the job script outputting
job details and job summaries at the beginning and at the end.
\item All remaining \code{resources} named elements are injected as named resource
specification for the scheduler.
}}

\item{delete}{Controls if and when the batchtools job registry folder is
deleted.
If \code{"on-success"} (default), it is deleted if the future was resolved
successfully \emph{and} the expression did not produce an error.
If \code{"never"}, then it is never deleted.
If \code{"always"}, then it is always deleted.}

\item{workers}{The maximum number of workers the batchtools backend may
use at any time, which for HPC schedulers corresponds to the maximum number
of queued jobs. The default is
\code{getOption("\link{future.batchtools.workers}", 100)}.}

\item{\ldots}{Not used.}
}
\description{
A batchtools TORQUE backend resolves futures in parallel via a TORQUE/PBS job scheduler
}
\details{
Batchtools TORQUE/PBS futures use \pkg{batchtools} cluster functions
created by \code{\link[batchtools:makeClusterFunctionsTORQUE]{batchtools::makeClusterFunctionsTORQUE()}}, which are used
to interact with the TORQUE/PBS job scheduler. This requires that
TORQUE/PBS commands \code{qsub}, \code{qselect}, and \code{qdel} are available on
the current machine.

The default template script \code{templates/torque.tmpl} can be found in:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{system.file("templates", "torque.tmpl", package = "future.batchtools")
}\if{html}{\out{</div>}}

and comprise:

\if{html}{\out{<div class="sourceCode bash">}}\preformatted{#!/bin/bash
######################################################################
# A batchtools launch script template for TORQUE/PBS
#
# Author: Henrik Bengtsson
######################################################################

## Job name
#PBS -N <\%= job.name \%>

## Direct streams to logfile
#PBS -o <\%= log.file \%>

## Merge standard error and output
#PBS -j oe

## Resources needed:
<\%
  ## Should scheduler "details" be seen?
  details <- isTRUE(resources[["details"]])
  resources[["details"]] <- NULL

  ## Shell "startup" code to evaluate
  startup <- resources[["startup"]]
  resources[["startup"]] <- NULL
     
  ## Shell "shutdown" code to evaluate
  shutdown <- resources[["shutdown"]]
  resources[["shutdown"]] <- NULL
     
  ## Environment modules specifications
  modules <- resources[["modules"]]
  resources[["modules"]] <- NULL
     
  ## As-is resource specifications
  job_declarations <- resources[["asis"]]
  resources[["asis"]] <- NULL
    
  ## Remaining resources are assumed to be of type '-l <key>=<value>'
  opts <- unlist(resources, use.names = TRUE)
  opts <- sprintf("-l \%s=\%s", names(opts), opts)
  job_declarations <- sprintf("#PBS \%s", c(job_declarations, opts))
  writeLines(job_declarations)
\%>

## Bash settings
set -e          # exit on error
set -u          # error on unset variables
set -o pipefail # fail a pipeline if any command fails
trap 'echo "ERROR: future.batchtools job script failed on line $LINENO" >&2; exit 1' ERR

<\% if (length(job_declarations) > 0) \{
  writeLines(c(
    "echo 'Job submission declarations:'",
    sprintf("echo '\%s'", job_declarations),
    "echo"
  ))
\} \%>

<\% if (details) \{ \%>
if command -v qstat > /dev/null; then
  echo "Job information:"
  qstat -f "$\{PBS_JOBID\}"
  echo
fi
<\% \} \%>

<\% if (length(startup) > 0) \{
  writeLines(startup)
\} \%>

<\% if (length(modules) > 0) \{
  writeLines(c(
    "echo 'Load environment modules:'",
    sprintf("echo '- modules: \%s'", paste(modules, collapse = ", ")),
    sprintf("module load \%s", paste(modules, collapse = " ")),
    "module list"
  ))
\} \%>

echo "Session information:"
echo "- timestamp: $(date +"\%Y-\%m-\%d \%H:\%M:\%S\%z")"
echo "- hostname: $(hostname)"
echo "- Rscript path: $(which Rscript)"
echo "- Rscript version: $(Rscript --version)"
echo "- Rscript library paths: $(Rscript -e "cat(shQuote(.libPaths()), sep = ' ')")"
echo

## Launch R and evaluate the batchtools R job
echo "Rscript -e 'batchtools::doJobCollection()' ..."
echo "- job name: '<\%= job.name \%>'"
echo "- job log file: '<\%= log.file \%>'"
echo "- job uri: '<\%= uri \%>'"
Rscript -e 'batchtools::doJobCollection("<\%= uri \%>")'
res=$?
echo " - exit code: $\{res\}"
echo "Rscript -e 'batchtools::doJobCollection()' ... done"
echo

<\% if (details) \{ \%>
if command -v qstat > /dev/null; then
  echo "Job summary:"
  qstat -f "$\{PBS_JOBID\}"
fi    
<\% \} \%>

<\% if (length(shutdown) > 0) \{
  writeLines(shutdown)
\} \%>

echo "End time: $(date +"\%Y-\%m-\%d \%H:\%M:\%S\%z")"

## Relay the exit code from Rscript
exit "$\{res\}"
}\if{html}{\out{</div>}}
}
\examples{
\dontshow{if (interactive()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(future)

# Limit runtime to 10 minutes and total memory to 400 MiB per future,
# request a parallel environment with four slots on a single host.
# Submit to the 'freecycle' queue. Load environment modules 'r' and
# 'jags'. Report on job details at startup and at the end of the job.
plan(future.batchtools::batchtools_torque, resources = list(
  walltime = "00:10:00", mem = "100mb",  ## memory is per process
  asis = c("-l nodes=1:ppn=4", "-q freecycle"),
  modules = c("r", "jags"),
  details = TRUE
))

f <- future({
  data.frame(
    hostname = Sys.info()[["nodename"]],
          os = Sys.info()[["sysname"]],
       cores = unname(parallelly::availableCores())
  )
})
info <- value(f)
print(info)
\dontshow{\}) # examplesIf}
}
\references{
\itemize{
\item \url{https://en.wikipedia.org/wiki/TORQUE}
}
}
