% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gdalalg.R
\name{GDALAlg-class}
\alias{GDALAlg-class}
\alias{Rcpp_GDALAlg}
\alias{Rcpp_GDALAlg-class}
\alias{GDALAlg}
\title{Class encapsulating a GDAL CLI algorithm}
\arguments{
\item{cmd}{A character string or character vector containing the path to the
algorithm, e.g., \code{"raster reproject"} or \code{c("raster", "reproject")}.}

\item{args}{Either a character vector or a named list containing input
arguments of the algorithm (see section \dQuote{Algorithm argument syntax}).}
}
\value{
An object of class \code{GDALAlg}, which contains a pointer to the
algorithm instance. Class methods are described in Details, along with a set
of writable fields for per-object settings. Values may be assigned to the
class fields by regular \code{<-} or \code{=} assignment.
}
\description{
\code{GDALAlg} provides bindings to \code{GDALAlgorithm} and related classes
that implement the "gdal" command line interface (CLI) in the GDAL API.
An object of class \code{GDALAlg} represents an instance of a CLI algorithm with
methods to obtain algorithm information and argument information, run the
algorithm, and access its output.

\strong{Requires GDAL >= 3.11.3}.

\strong{Experimental} (see "Development status")

\code{GDALAlg} is a C++ class exposed directly to \R (via \code{RCPP_EXPOSED_CLASS}).
Fields and methods of the class are accessed using the \code{$} operator. Note
that all arguments to class methods are required and must be given in the
order documented (naming optional).
}
\section{Usage (see Details)}{

\preformatted{
## Constructors
alg <- new(GDALAlg, cmd)
# or, with arguments
alg <- new(GDALAlg, cmd, args)

## Read/write fields (per-object settings)
alg$setVectorArgsFromObject
alg$outputLayerNameForOpen
alg$quiet

## Methods
alg$info()
alg$argInfo(arg_name)
alg$usage()
alg$usageAsJSON()

alg$parseCommandLineArgs()
alg$run()
alg$output()
alg$outputs()

alg$close()
alg$release()
}
}

\section{Details}{

\subsection{Constructors}{

\code{new(GDALAlg, cmd)}\cr
Instantiate an algorithm without specifying input arguments.

\code{new(GDALAlg, cmd, args)}\cr
Instantiate an algorithm giving input arguments as a character vector or
named list. See section \dQuote{Algorithm argument syntax} for details.
}

\subsection{Read/write fields (per-object settings)}{

\code{$setVectorArgsFromObject}\cr
A logical value, \code{TRUE} (the default) to set algorithm arguments
automatically when the \code{"input"} argument or the \code{"like"} argument is an
object of class \code{GDALVector}. Argument values specified explicitly will
override the automatic setting (as long as they result in a parsable set of
arguments). Automatically setting arguments from \code{GDALVector} input can
be disabled by setting this field to \code{FALSE}.
When enabled and the \code{"input"} or \code{"like"} argument for an algorithm is a
\code{GDALVector} object, the following arguments will be set automatically based
on properties of the object (when the argument is available to the
algorithm):
\itemize{
\item \code{"input-format"}: set to the object's driver short name
\item \code{"input-layer"}: set to the object's layer name if it is not a SQL layer
\item \code{"sql"}: set to the SQL statement if the object's layer is defined by one
\item \code{"dialect"}: set to the SQL dialect if one is specified for a SQL layer
\item \code{"like-layer"}: set to the \code{GDALVector} layer name if it is not a SQL
layer
\item \code{"like-sql"}: set to the SQL statement if the \code{GDALVector} layer is
defined by one
}

\code{$outputLayerNameForOpen}\cr
A character string specifying a layer name to open when obtaining algorithm
output as an object of class \code{GDALVector}. See method \code{$output()} below.
The default value is empty string (\code{""}) in which case the first layer by
index is opened. Ignored if output is not a vector dataset.

\code{$quiet}\cr
A logical value, \code{FALSE} by default. Set to \code{TRUE} to suppress progress
reporting along with various messages and warnings.
}

\subsection{Methods}{

\code{$info()}\cr
Returns a named list of algorithm properties with the following elements:
\itemize{
\item \code{name}: character string, the algorithm name
\item \code{description}: character string, the algorithm (short) description
\item \code{long_description}: character string, the algorithm longer description
\item \code{URL}: character string, the algorithm help URL
\item \code{has_subalgorithms}: logical, \code{TRUE} if the algorithm has sub-algorithms
\item \code{subalgorithm_names}: character vector of sub-algorithm names (may be
empty)
\item \code{arg_names}: character vector of available argument names
}

\code{$argInfo(arg_name)}\cr
Returns a named list of properties for an algorithm argument given as a
character string, with the following elements:
\itemize{
\item \code{name}: character string, the name of the argument
\item \code{type}: character string, the argument type as one of \code{"BOOLEAN"},
\code{"STRING"}, \code{"INTEGER"}, \code{"REAL"}, \code{"DATASET"}, \code{"STRING_LIST"},
\code{"INTEGER_LIST"}, \code{"REAL_LIST"}, \code{"DATASET_LIST"}
\item \code{description}: character string, the argument description
\item \code{short_name}: character string, the short name or empty string if there
is none
\item \code{aliases}: character vector of aliases (empty if none)
\item \code{meta_var}: character string, the "meta-var" hint (by default, the
meta-var value is the long name of the argument in upper case)
\item \code{category}: character string, the argument category
\item \code{is_positional}: logical, \code{TRUE} if the argument is a positional one
\item \code{is_required}: logical, \code{TRUE} if the argument is required
\item \code{min_count}: integer, the minimum number of values for the argument (only
applies to list type of arguments)
\item \code{max_count}: integer, the maximum number of values for the argument (only
applies to list type of arguments)
\item \code{packed_values_allowed}: logical, \code{TRUE} if, for list type of arguments,
several comma-separated values may be specified (i.e., \code{"--foo=bar,baz"})
\item \code{repeated_arg_allowed}: logical, \code{TRUE} if, for list type of arguments,
the argument may be repeated (i.e., \code{c("--foo=bar", "--foo=baz")})
\item \code{choices}: character vector of allowed values for the argument (may be
empty and only applies for argument types \code{"STRING"} and \code{"STRING_LIST"})
\item \code{is_explicitly_set}: logical, \code{TRUE} if the argument value has been
explicitly set
\item \code{has_default_value}: logical, \code{TRUE} if the argument has a declared
default value
\item \code{default_value}: character or integer or numeric, the default value if the
argument has one, otherwise \code{NULL}
\item \code{is_hidden_for_api}: logical, \code{TRUE} if the argument is only for CLI usage
(e.g., "--help") (GDAL >= 3.12 only, see \code{is_only_for_cli})
\item \code{is_hidden_for_cli}: logical, \code{TRUE} if the argument must not be
mentioned in CLI usage (e.g., "output-value" for "gdal raster info", which
is only meant when the algorithm is used from a non-CLI context such as
programmatically from \R)
\item \code{is_only_for_cli}: logical, \code{TRUE} if the argument is only for CLI usage
(e.g., "--help") (deprecated at GDAL 3.12, see \code{is_hidden_for_api})
\item \code{is_input}: logical, \code{TRUE} if the value of the argument is read-only
during the execution of the algorithm
\item \code{is_output}: logical, \code{TRUE} if (at least part of) the value of the
argument is set during the execution of the algorithm
\item \code{dataset_type_flags}: character vector containing strings \code{"RASTER"},
\code{"VECTOR"}, \code{"MULTIDIM_RASTER"}, possibly with \code{"UPDATE"} (\code{NULL} if
the argument is not a dataset type)
\item \code{dataset_input_flags}: character vector indicating if a dataset argument
supports specifying only the dataset name (\code{"NAME"}), only the dataset
object (\code{"OBJECT"}), or both (\verb{"NAME", "OBJECT"}) when it is used as an
input (\code{NULL} if the argument is not a dataset type)
\item \code{dataset_output_flags}: character vector indicating if a dataset argument
supports specifying only the dataset name (\code{"NAME"}), only the dataset
object (\code{"OBJECT"}), or both (\verb{"NAME", "OBJECT"}) when it is used as an
output (\code{NULL} if the argument is not a dataset type)
\item \code{mutual_exclusion_group}: character string, the name of the mutual
exclusion group to which this argument belongs
}

\code{$usage()}\cr
Print a help message for the algorithm to the console. No return value.

\code{$usageAsJSON()}\cr
Returns the usage of the algorithm as a JSON-serialized string.

\code{$parseCommandLineArgs()}\cr
Sets the value of arguments previously specified in the class constructor,
and instantiates the actual algorithm that will be run (but without running
it). Returns a logical value, \code{TRUE} indicating success or \code{FALSE} if an
error occurs.

\code{$run()}\cr
Executes the algorithm, first parsing arguments if
\code{$parseCommandLineArgs()} has not already been called explicitly.
Returns a logical value, \code{TRUE} indicating success or \code{FALSE} if an error
occurs.

\code{$output()}\cr
Returns the single output value of the algorithm, after it has been run.
If there are multiple output values, this method will raise an error, and
the \code{$outputs()} (plural) method should be called instead. The type of
the return value corresponds to the type of the single output argument value
(see method \code{$argInfo()} above).
If the output argument has type \code{"DATASET"}, an object of class \code{GDALRaster}
will be returned if the dataset is raster, or an object of class
\code{GDALVector} if the dataset is vector. In the latter case, by default the
\code{GDALVector} object will be opened on the first layer by index, but a
specific layer name may be specified by setting the value of the field
\code{$outputLayerNameForOpen} before calling the \code{$output()} method
(see above).
Note that currently, if the output dataset is multidimensional raster, only
the dataset name will be returned as a character string.

\code{$outputs()}\cr
Returns the output value(s) of the algorithm as a named list, after it has
been run. Most algorithms only return a single output, in which case the
\code{$output()} method (singular) is preferable for easier use. The element
names in the returned list are the names of the arguments that have outputs
(with any dash characters replaced by underscore), and the values are the
argument values which may include \code{GDALRaster} or \code{GDALVector} objects.

\code{$close()}\cr
Completes any pending actions, and returns the final status as a logical
value (\code{TRUE} if no errors occur during the underlying call to
\code{GDALAlgorithmFinalize()}). This is typically useful for algorithms that
generate an output dataset. It closes datasets and gets back potential error
status resulting from that, e.g., if an error occurs during flushing to disk
of the output dataset after successful \code{$run()} execution.

\code{$release()}\cr
Release memory associated with the algorithm, potentially after attempting
to finalize. No return value, called for side-effects.
}
}

\section{Algorithm argument syntax}{

Arguments are given in \R as a character vector or named list, but
otherwise syntax basically matches the GDAL specification for arguments as
they are given on the command line. Those specifications are listed here
along with some amendments regarding the character vector and named list
formats. Programmatic usage also allows passing and receiving datasets as
objects (i.e., \code{GDALRaster} or \code{GDALVector}), in addition to dataset names
(e.g., filename, URL, database connection string).
\itemize{
\item Commands accept one or several positional arguments, typically for dataset
names (or, in \R, as \code{GDALRaster} or \code{GDALVector} datasets). The order is
input(s) first, output last. Positional arguments can also be specified as
named arguments, if preferred to avoid any ambiguity.
\item Named arguments have:
\itemize{
\item at least one "long" name, preceded by two dash characters when specified
on the command line,
\item optionally, auxiliary long names (i.e., aliases),
\item and optionally a one-letter short name, preceded by a single dash
character on the command line, e.g., \verb{-f, --of, --format, --output-format <OUTPUT-FORMAT>}
}
\item Boolean arguments are specified by just specifying the argument name. In
\R list format, the named element must be assigned a value of logical \code{TRUE}.
\item Arguments that require a value are specified like:
\itemize{
\item \verb{-f VALUE} for one-letter short names.
\item \verb{--format VALUE} or \code{--format=VALUE} for long names.
\item in a named list, this would look like:\cr
\code{arg_list$format <- VALUE}
}
\item Some arguments can be multi-valued. Some of them require all values to be
packed together and separated with comma. This is, e.g., the case of:
\verb{--bbox <BBOX> Clipping bounding box as xmin,ymin,xmax,ymax}, e.g.,
\verb{--bbox=2.1,49.1,2.9,49.9}.
\item Others accept each value to be preceded by a new mention of the argument
name, e.g., \verb{--co COMPRESS=LZW --co TILED=YES}. For that one, if the value
of the argument does not contain commas, the packed form is also accepted:
\verb{--co COMPRESS=LZW,TILED=YES}. Note that repeated mentions of an argument
are possible in the character vector format for argument input, whereas
arguments given in named list format must used argument long names as the
list element names, and the packed format for the values (which can be a
character vector or numeric vector of values).
\item Named arguments can be placed before or after positional arguments.
}
}

\section{Development status}{

The GDAL Command Line Interface Modernization was first introduced in the
\href{https://github.com/OSGeo/gdal/releases/tag/v3.11.0}{GDAL 3.11.0 release}
(2025-05-09). The GDAL project states that the new CLI \dQuote{is
provisionally provided as an alternative interface to GDAL and OGR command
line utilities. The project reserves the right to modify, rename,
reorganize, and change the behavior until it is officially frozen via PSC
vote in a future major GDAL release.... Your usage of it should have no
expectation of compatibility until that time.}
(\url{https://gdal.org/en/latest/programs/#gdal-application})

Initial bindings to enable programmatic use of the new CLI algorithms from \R
were added in \pkg{gdalraster} 2.2.0, and will evolve over future releases.
The bindings are considered experimental until the upstream API is declared
stable. Breaking changes in minor version releases are possible until then.
}

\examples{
\dontshow{if (gdal_version_num() >= gdalraster:::.GDALALG_MIN_GDAL) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
f <- system.file("extdata/storml_elev.tif", package="gdalraster")

## raster info
gdal_usage("raster info")

# arguments given as character vector
args <- c("--format=text", "--no-md", f)
(alg <- new(GDALAlg, "raster info", args))

alg$run()

alg$output() |> writeLines()

alg$release()

## raster clip
gdal_usage("raster clip")

# arguments in list format
args <- list()
args$input <- f
f_clip <- file.path(tempdir(), "elev_clip.tif")
args$output <- f_clip
args$overwrite <- TRUE
args$bbox <- c(323776.1, 5102172.0,  327466.1, 5104782.0)

(alg <- new(GDALAlg, "raster clip", args))

alg$run()

# get output as a GDALRaster object
(ds_clip <- alg$output())

alg$release()

ds_clip$dim()
ds_clip$bbox()

## raster hillshade
gdal_usage("raster hillshade")

# input as a GDALRaster object and output to an in-memory raster
args <- list(input = ds_clip,
             output_format = "MEM",
             output = "")

(alg <- new(GDALAlg, "raster hillshade", args))

alg$run()

(ds_hillshade <- alg$output())

alg$release()

plot_raster(ds_hillshade)

ds_clip$close()
ds_hillshade$close()
deleteDataset(f_clip)

## vector convert shapefile to GeoPackage
gdal_usage("vector convert")

f_shp <- system.file("extdata/poly_multipoly.shp", package="gdalraster")
f_gpkg <- file.path(tempdir(), "polygons_test.gpkg")
args <- c("--input", f_shp, "--output", f_gpkg, "--overwrite")

(alg <- new(GDALAlg, "vector convert", args))

alg$run()

# output as a GDALVector object
(lyr <- alg$output())

alg$release()

lyr$info()

lyr$close()
deleteDataset(f_gpkg)

## raster reproject (usage and info)
# no arguments given, only for retrieving usage and properties via API
# cf. gdal_usage("raster reproject")

(alg <- new(GDALAlg, "raster reproject"))

# print usage to the console, no return value
alg$usage()

# or, get usage as a JSON string
# json <- alg$usageAsJSON()
# writeLines(json)

# list of algorithm properties
alg$info()

# list of properties for an algorithm argument
alg$argInfo("resampling")

alg$release()
\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link[=gdal_alg]{gdal_alg()}}, \code{\link[=gdal_commands]{gdal_commands()}}, \code{\link[=gdal_run]{gdal_run()}}, \code{\link[=gdal_usage]{gdal_usage()}}

GDAL RFC 104: Adding a \dQuote{gdal} front-end command line interface:\cr
\href{https://gdal.org/en/stable/development/rfc/rfc104_gdal_cli.html#implementation-details}{Implementation details}

\href{https://usdaforestservice.github.io/gdalraster/articles/use-gdal-cli-from-r.html}{Using \dQuote{gdal} CLI algorithms from R}
}
