% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ogr_proc.R
\name{ogr_proc}
\alias{ogr_proc}
\title{GDAL OGR facilities for vector geoprocessing}
\usage{
ogr_proc(
  mode,
  input_lyr,
  method_lyr,
  out_dsn,
  out_lyr_name = NULL,
  out_geom_type = NULL,
  out_fmt = NULL,
  dsco = NULL,
  lco = NULL,
  mode_opt = NULL,
  overwrite = FALSE,
  quiet = FALSE,
  return_obj = TRUE
)
}
\arguments{
\item{mode}{Character string specifying the operation to perform. One of
\code{Intersection}, \code{Union}, \code{SymDifference}, \code{Identity}, \code{Update}, \code{Clip} or
\code{Erase} (see Details).}

\item{input_lyr}{An object of class \code{\link{GDALVector}} to use as the
input layer. For overlay operations, this is the first layer in the
operation.}

\item{method_lyr}{An object of class \code{\link{GDALVector}} to use as the
method layer. This is the conditional layer supplied to an operation (e.g.,
Clip, Erase, Update), or the second layer in overlay operations (e.g., Union,
Intersection, SymDifference).}

\item{out_dsn}{The destination vector filename or database connection string
to which the output layer will be written.}

\item{out_lyr_name}{Layer name where the output vector will be written. May
be \code{NULL} (e.g., shapefile), but typically must be specified.}

\item{out_geom_type}{Character string specifying the geometry type of the
output layer. One of NONE, GEOMETRY, POINT, LINESTRING, POLYGON,
GEOMETRYCOLLECTION, MULTIPOINT, MULTIPOLYGON, GEOMETRY25D, POINT25D,
LINESTRING25D, POLYGON25D, GEOMETRYCOLLECTION25D, MULTIPOINT25D,
MULTIPOLYGON25D. Defaults to UNKNOWN if not specified.}

\item{out_fmt}{GDAL short name of the output vector format. If unspecified,
the function will attempt to guess the format from the value of \code{out_dsn}.}

\item{dsco}{Optional character vector of format-specific creation options
for \code{out_dsn} (\code{"NAME=VALUE"} pairs).}

\item{lco}{Optional character vector of format-specific creation options
for \code{out_layer} (\code{"NAME=VALUE"} pairs).}

\item{mode_opt}{Optional character vector of \code{"NAME=VALUE"} pairs that
specify processing options. Available options depend on the value of \code{mode}
(see Details).}

\item{overwrite}{Logical value. \code{TRUE} to overwrite the output layer if it
already exists. Defaults to \code{FALSE}.}

\item{quiet}{Logical value. If \code{TRUE}, a progress bar will not be
displayed. Defaults to \code{FALSE}.}

\item{return_obj}{Logical value. If \code{TRUE} (the default), an object of
class \code{\link{GDALVector}} opened on the output layer will be returned,
otherwise the function returns a logical value.}
}
\value{
Upon successful completion, an object of class
\code{\link{GDALVector}} is returned by default (\code{return_obj = TRUE}), or
logical \code{TRUE} is returned if \code{return_obj = FALSE}.
Logical \code{FALSE} is returned if an error occurs during processing.
}
\description{
\code{ogr_proc()} performs GIS overlay operations on vector layers
(\url{https://en.wikipedia.org/wiki/Vector_overlay}). It provides
an interface to the GDAL API methods for these operations
(\code{OGRLayer::Intersection()}, \code{OGRLayer::Union()}, etc).
Inputs are given as objects of class \code{\link{GDALVector}}, which
may have spatial and/or attribute filters applied.
The output layer will be created if it does not exist, but output can also
be appended to an existing layer, or written to an existing empty layer that
has a custom schema defined.
\code{ogr_proc()} is basically a port of the
\href{https://gdal.org/en/stable/programs/ogr_layer_algebra.html#ogr-layer-algebra}{\code{ogr_layer_algebra}}
utility in the GDAL Python bindings.
}
\details{
Seven processing modes are available:
\itemize{
\item \code{Intersection}: The output layer contains features whose geometries
represent areas that are common between features in the input layer and in
the method layer. The features in the output layer have attributes from
both input and method layers.
\item \code{Union}: The output layer contains features whose geometries represent
areas that are either in the input layer, in the method layer, or in
both. The features in the output layer have attributes from both input and
method layers. For features which represent areas that are only in the
input layer or only in the method layer the respective attributes have
undefined values.
\item \code{SymDifference}: The output layer contains features whose geometries
represent areas that are in either in the input layer or in the method layer
but not in both. The features in the output layer have attributes from both
input and method layers. For features which represent areas that are only in
the input or only in the method layer the respective attributes have
undefined values.
\item \code{Identity}: Identifies the features of the input layer with the ones from
the method layer. The output layer contains features whose geometries
represent areas that are in the input layer. The features in the output layer
have attributes from both the input and method layers.
\item \code{Update}: The update method creates a layer which adds features into the
input layer from the method layer, possibly cutting features in the input
layer. The features in the output layer have areas of the features of the
method layer or those areas of the features of the input layer that are not
covered by the method layer. The features of the output layer get their
attributes from the input layer.
\item \code{Clip}: The clip method creates a layer which has features from the input
layer clipped to the areas of the features in the method layer. By default
the output layer has attributes of the input layer.
\item \code{Erase}: The erase method creates a layer which has features from the input
layer whose areas are erased by the features in the method layer. By default,
the output layer has attributes of the input layer.
}

By default, \code{ogr_proc()} will create the output layer with an empty schema.
It will be initialized by GDAL to contain all fields in the input layer, or
depending on the operation, all fields in both the input and method layers.
In the latter case, the prefixes \code{"input_"} and \code{"method_"} will be added to
the output field names by default. The default prefixes can be overridden in
the \code{mode_opt} argument as described below.

Alternatively, the functions in the \code{\link{ogr_manage}} interface
could be used to create an empty layer with user-defined schema (e.g.,
\code{\link[=ogr_ds_create]{ogr_ds_create()}}, \code{\link[=ogr_layer_create]{ogr_layer_create()}} and \code{\link[=ogr_field_create]{ogr_field_create()}}). If the
schema of the output layer is set by the user and contains fields that have
the same name as a field in both the input and method layers, then the
attribute for an output feature will get the value from the feature of the
method layer.

Options that affect processing can be set as NAME=VALUE pairs passed in the
\code{mode_opt} argument. Some options are specific to certain processing modes
as noted below:
\itemize{
\item SKIP_FAILURES=YES/NO. Set it to YES to go on, even when a feature could
not be inserted or a GEOS call failed.
\item PROMOTE_TO_MULTI=YES/NO. Set to YES to convert Polygons into MultiPolygons,
LineStrings to MultiLineStrings or Points to MultiPoints (only since GDAL
3.9.2 for the latter).
\item INPUT_PREFIX=string. Set a prefix for the field names that will be created
from the fields of the input layer.
\item METHOD_PREFIX=string. Set a prefix for the field names that will be created
from the fields of the method layer.
\item USE_PREPARED_GEOMETRIES=YES/NO. Set to NO to not use prepared geometries to
pretest intersection of features of method layer with features of input
layer. Applies to \code{Intersection}, \code{Union}, \code{Identity}.
\item PRETEST_CONTAINMENT=YES/NO. Set to YES to pretest the containment of
features of method layer within the features of input layer. This will speed
up the operation significantly in some cases. Requires that the prepared
geometries are in effect. Applies to \code{Intersection}.
\item KEEP_LOWER_DIMENSION_GEOMETRIES=YES/NO. Set to NO to skip result features
with lower dimension geometry that would otherwise be added to the output
layer. The default is YES, to add features with lower dimension geometry, but
only if the result output has an UNKNOWN geometry type. Applies to
\code{Intersection}, \code{Union}, \code{Identity}.
}

The input and method layers should have the same spatial reference system. No
on-the-fly reprojection is done. When an output layer is created it will have
the SRS of \code{input_lyr}.
}
\note{
The first geometry field on a layer is always used.

For best performance use the minimum amount of features in the method layer
and copy into a memory layer.
}
\examples{
# MTBS fires in Yellowstone National Park 1984-2022
dsn <- system.file("extdata/ynp_fires_1984_2022.gpkg", package="gdalraster")

# layer filtered to fires after 1988
lyr1 <- new(GDALVector, dsn, "mtbs_perims")
lyr1$setAttributeFilter("ig_year > 1988")
lyr1$getFeatureCount()

# second layer for the 1988 North Fork fire perimeter
sql <- "SELECT incid_name, ig_year, geom FROM mtbs_perims
        WHERE incid_name = 'NORTH FORK'"
lyr2 <- new(GDALVector, dsn, sql)
lyr2$getFeatureCount()

# intersect to obtain areas in the North Fork perimeter that have re-burned
tmp_dsn <- tempfile(fileext = ".gpkg")
opt <- c("INPUT_PREFIX=layer1_",
         "METHOD_PREFIX=layer2_",
         "PROMOTE_TO_MULTI=YES")

lyr_out <- ogr_proc(mode = "Intersection",
                    input_lyr = lyr1,
                    method_lyr = lyr2,
                    out_dsn = tmp_dsn,
                    out_lyr_name = "north_fork_reburned",
                    out_geom_type = "MULTIPOLYGON",
                    mode_opt = opt)

# the output layer has attributes of both the input and method layers
(d <- lyr_out$fetch(-1))

# clean up
lyr1$close()
lyr2$close()
lyr_out$close()
\dontshow{deleteDataset(tmp_dsn)}
}
\seealso{
\code{\link[=GDALVector]{GDALVector-class}}, \code{\link{ogr_define}},
\code{\link{ogr_manage}}

Vector overlay operators:\cr
\url{https://en.wikipedia.org/wiki/Vector_overlay}
}
