\name{krige.conv}

\alias{krige.conv}
\alias{krige.control}
%- Also NEED an `\alias' for EACH other topic documented here.

\title{Spatial Prediction -- Conventional Kriging}


\description{
  This function performs spatial prediction for fixed covariance
  parameters using global neighborhood.
  
  Available options implement the following kriging
  types: \emph{SK} (simple kriging), \emph{OK} (ordinary kriging),
    \emph{KTE} (external trend kriging) and \emph{UK} (universal kriging). 
  
}

\usage{
krige.conv(geodata, coords = geodata$coords, data = geodata$data, 
           locations,
           krige = krige.control(type.krige, beta = NULL,
                     trend.d, trend.l, cov.model, cov.pars,
                     kappa = 0.5, nugget = 0, micro.scale = 0,
                     dist.epsilon = 1e-10, aniso.pars = NULL, 
                     lambda = 1, signal = FALSE,
                     n.samples.backtransform = 500, n.sim = 0),
           messages.screen = TRUE) 
}

%- maybe also `usage' for other objects documented here.

\arguments{
  \item{geodata}{a list containing elements \code{coords} and
    \code{data} as described next. Typically an object of the \code{\link{class}}
    \code{"geodata"} - a \bold{geoR} data-set. If not provided the arguments
    \code{coords} and \code{data} must be provided instead.  }
  \item{coords}{an \eqn{n \times 2}{n x 2} containing in each row have the 2-D
    coordinates of the \eqn{n} data locations.
    By default it takes the
    component \code{coords} of the argument \code{geodata}, if provided.  }
  \item{data}{a vector with \emph{n} data values. By default it takes the
    component \code{data} of the argument \code{geodata}, if provided.}
  \item{locations}{an \eqn{N \times 2}{N x 2} matrix or data-frame with the 2-D
    coordinates of the \eqn{N} prediction locations.  }
  \item{krige}{defines the model components and the type of
    kriging. See section DETAILS below.
    \cr ATTENTION: the argument \code{cov.pars} is obligatory whilst
    all the others have default options.  }
  \item{messages.screen}{logical. Indicates
    whether or not status messages are printed on the screen (or other output device)
    while the function is running.}
}


\details{
  One of the following different types of kriging: \emph{SK}, \emph{OK},
  \emph{UK} or  \emph{KTE}
  is performed, according to the input options.
  Defaults correspond to ordinary kriging.
  
  \bold{Arguments for \code{krige = krige.control(\dots)} :} 

  \describe{
    \item{type.krige}{type of kriging to be performed. Options are
      \code{"SK", "OK"} corresponding to simple or ordinary
      kriging. Kriging with external trend and universal kriging can be
      defined setting \code{type.krige = "OK"} and specifying the
      trend model using the arguments \code{trend.d} and
      \code{trend.l}.  } 
    \item{beta}{numerical value of the mean (vector) parameter.
      Only used if \code{type.krige="SK"}.}  
    \item{trend.d}{specifies the trend (covariates) values at the data
      locations. Possible values are:
      \code{"cte"} - model with constant mean, i.e. no trend
      \code{"1st"} - trend is defined as a first degree polynomial on the coordinates
      \code{"2nd"} - trend is defined as a second degree polynomial on the
      coordinates
      a formula of the type \code{~X}, where \code{X} is a matrix with covariates (external trend)
      at data locations.
      Defaults to \code{"cte"}.  }
    \item{trend.l}{specifies the trend (covariate) at prediction
      locations. It must be of the same type as defined in \code{trend.d}.
      Only used if prediction locations are provided in the argument
      \code{locations}.  }
    \item{cov.model}{string indicating the name of the model for the
      correlation function. Further details  in the
      documentation for
      \code{\link{cov.spatial}}.  }
    \item{cov.pars}{a vector with 2 elements or an \eqn{n \times 2}{n x
	2} matrix with the covariance parameters \eqn{\sigma^2}{sigma^2}
      (partial sill)  and \eqn{\phi}{phi} (range parameter). If a vector, the elements are the values of
    \eqn{\sigma^2}{sigma^2} and \eqn{\phi}{phi}, respectively.
    If a matrix, corresponding to a model with several structures, the
    values of \eqn{\sigma^2}{sigma^2} 
    are in the first column and the values of \eqn{\phi}{phi} are in the second.  }
    \item{kappa}{additional smoothness parameter required by the following correlation
      functions: \code{"matern"}, \code{"powered.exponential"}, \code{"cauchy"} and
      \code{"gneiting.matern"}.   }
    \item{nugget}{the value of the nugget variance parameter \eqn{\tau^2}{tau^2}. Defaults to zero.  }
    \item{micro.scale}{micro-scale variance. If different from zero, the
      nugget variance is divided into 2 terms: \emph{micro-scale variance}
      and \emph{measurement error}. This might affect the precision of
      the predictions. 
      In practice, these two variance components are usually indistinguishable but the
      distinction can be made here if justifiable.  } 
    \item{dist.epsilon}{a numeric value. Points which are separated
      by a distance less than this value
      are considered co-located.  }
    \item{aniso.pars}{parameters for geometric anisotropy
      correction. If \code{aniso.pars = FALSE} no correction is made, otherwise
      a two elements vector with values for the anisotropy parameters
      must be provided. Anisotropy correction consists of a
      transformation of the data and prediction coordinates performed
      by the function \code{\link{coords.aniso}}.  }
    \item{lambda}{numeric value of the Box-Cox transformation parameter.
      The value \eqn{\lambda = 1}{lambda = 1} corresponds to
      no transformation and \eqn{\lambda = 0}{lambda = 0} corresponds to
      the log-transformation. 
      Prediction results are back-transformed and
      returned is the same scale as for the original data.  }
    \item{signal}{logical. If \code{TRUE} the signal is predicted, otherwise the
      variable is predicted. If no transformation is performed the
      expectations are the same in both cases and the difference is only for
      values of the kriging variance, if the value of the nugget is
      different from zero.  }
    \item{n.samples.backtransform}{number of samples used in the
      back-transformation. When transformations are used
      (specified by an argument \code{lambda}), back-transformations
      are usually performed by sampling from the predictive distribution and
      then back-transforming the sampled values. The exceptions are for
      \eqn{\lambda = 0}{lambda = 0} (log-transformation)  and
      \eqn{\lambda = 1}{lambda = 1} (no transformation).  }
    \item{n.sim}{optional. Number of simulations from the
      posterior distribution. If \code{n.sim} is provided, samples are
      taken from the predictive distribution. This corresponds to
      realizations of conditional simulations, given the data.}    
  }
}


\value{
  An object of the \code{\link{class}} \code{kriging} which is a  list
  with the following components:
  
  \item{predict}{a vector with predicted values.  }
  \item{krige.var}{a vector with predicted variances.  }
  \item{beta.est}{estimates of the \eqn{\beta}{beta}, parameter
    implicit in kriging procedure. Not valid if
    \code{type.krige = "SK"}.  }
  \item{simulations}{an \eqn{ni \times n.sim}{ni x n.sim} matrix where \eqn{ni} is the
    number of prediction locations. Each column corresponds to a
    conditional simulation of the predictive distribution.
    Only returned if \code{n.sim > 0}.  }
  \item{message}{messages about the type of prediction performed.  }
  \item{call}{the function call.  }
  
}


\references{
  Further information about \bold{geoR} can be found at:\cr
  \url{http://www.maths.lancs.ac.uk/~ribeiro/geoR.html}.
}

%\note{}

\author{
  Paulo J. Ribeiro Jr. \email{Paulo.Ribeiro@est.ufpr.br}, \cr
  Peter J. Diggle \email{p.diggle@lancaster.ac.uk}.
}

\seealso{\code{\link{image.kriging}} for graphical output of the results,
  \code{\link{krige.bayes}} for Bayesian prediction and \code{\link{ksline}}
  for a different implementation of kriging allowing for moving neighborhood.
}


\examples{
if(is.R()) data(s100) 
loci <- expand.grid(seq(0,1,l=31), seq(0,1,l=31))
kc <- krige.conv(s100, loc=loci,
                 krige=krige.control(cov.pars=c(1, .25)))
par(mfrow=c(1,2))
image.kriging(kc, loc=loci, main="kriging estimates")
image.kriging(kc, loc=loci, val=sqrt(kc$krige.var),
              main="kriging std. errors")

}


\keyword{spatial}

