#' Add dendrogram to ggplot object.
#'
#' @keywords internal
#'
#' @param plt ggplot object to add dendrogram to.
#' @param dendro Dendrogram segment and node data obtained from the `prepare_dendrogram` function.
#' @param dend_col String specifying colour of dendrogram (used if the colours have not been changed using other options).
#' @param dend_lwd Line width of dendrogram segments (used if not changed using other options).
#' @param dend_lty Line type of dendrogram (used if not changed using other options).
#'
#' @returns A ggplot object with a dendrogram added.
#'
add_dendrogram <- function(plt, dendro, dend_col = "black", dend_lwd = 0.3, dend_lty = 1) {
  x <- y <- xend <- yend <- NULL

  seg <- dendro$seg
  nod <- dendro$nod

  # Draw segments
  # Fill in NAs of col, lwd and lty as using "by_labels_branches_*" in dendextend::set may result in
  # only some segments having specifications and the rest being NA
  seg[["col"]][is.na(seg[["col"]])] <- dend_col
  seg[["lwd"]][is.na(seg[["lwd"]])] <- dend_lwd
  seg[["lty"]][is.na(seg[["lty"]])] <- dend_lty
  seg[["col"]] <- as.character(seg[["col"]])
  seg_colr <- dplyr::pull(dplyr::distinct(seg, col), col)
  # Colours for manual scale
  names(seg_colr) <- seg_colr

  plt <- plt +
    ggnewscale::new_scale_colour() +
    ggplot2::geom_segment(ggplot2::aes(x = x, y = y, xend = xend, yend = yend, colour = col), seg,
                          linewidth = if (all(is.na(seg$lwd))) {dend_lwd}
                          else {seg$lwd},
                          linetype = if (all(is.na(seg$lty))) {dend_lty}
                          else {seg$lty},
                          lineend = "square") +
    ggplot2::scale_colour_manual(values = seg_colr, guide = "none")

  # Draw nodes
  if (nrow(nod) > 0) {
    nod[["col"]] <- as.character(nod[["col"]])
    nod_colr <- dplyr::pull(dplyr::distinct(nod, col), col)
    names(nod_colr) <- nod_colr

    plt <- plt +
      ggnewscale::new_scale_colour() +
      ggplot2::geom_point(ggplot2::aes(x = x, y = y, colour = col), nod,
                          size = 1 * nod$cex, shape = nod$pch) +
      ggplot2::scale_colour_manual(values = nod_colr, guide = "none")
  }

  return(plt)
}


#' Prepare dendrogram by transforming dendrogram segments.
#'
#' @keywords internal
#'
#' @param dendro_in Dendrogram object generated by
#' `some_matrix |> dist() |> hclust() |> as.dendrogram() |> dendextend::as.ggdend()`.
#' @param context Dimension the dendrogram will be plotted against (rows or columns).
#' @param dend_side String for dendrogram side.
#' @param dend_defaults List with dendrogram default parameters.
#' @param dend_params List with dendrogram parameters to overwrite defaults.
#' @param full_plt Logical indicating if the whole heatmap is plotted or not.
#' @param layout The heatmap layout (for reordering rows).
#' @param x_long Data frame containing the values that will be plotted in the heatmap.
#' @param annot_df Data frame containing annotations.
#' @param annot_side Logical specifying which side the annotation will be drawn, analogous to 'dend_down' or 'dend_left'
#' (use the one in the same dimension as the dendrogram).
#' @param annot_pos Vector of the annotation positions along the opposite dimension.
#' @param annot_size Size of annotation cells, specified in heatmap cells (1 being the size of one cell).
#'
#' @return Data frame with dendrogram segment and node parameters.
#'
prepare_dendrogram <- function(dendro_in, context = c("rows", "cols"), dend_side,
                               dend_defaults, dend_params, full_plt, layout, x_long,
                               annot_df, annot_side, annot_pos, annot_size) {
  pch <- cex <- x <- y <- NULL

  # Replace default parameters if any are provided
  if (!is.list(dend_params) && !is.null(dend_params)) {
    var_name <- paste0("dend_", context[1], "_params")
    cli::cli_warn("{.var {var_name}} should be a {.cls list} or NULL, not {.cls {class(dend_params)}}.",
                  class = "dend_params_warn")
  }
  dend_params <- replace_default(dend_defaults, dend_params,
                                      warning_context = paste0("In {.var dend_", context[1], "_params}: "))

  dend_height <- dend_params[["height"]]
  dend_dist <- dend_params[["dist"]]

  # Check that numeric arguments are numeric
  if (any(c(!is.numeric(dend_dist), !is.numeric(dend_height))) ||
      any(c(length(dend_dist) > 1, length(dend_height) > 1))) {
    non_num_var <- c("dend_dist", "dend_height")[
      which(sapply(list(dend_dist, dend_height), function(x) {
        !is.numeric(x) || length(x) > 1
      }))
    ]
    cli::cli_abort("{.var {non_num_var}} must be {.cls numeric} with length one.",
                   class = "dend_nonnum_error")
  }

  # Check that the dendrogram side is ok
  if ((dend_side == "left" && context[1] == "rows") ||
      (dend_side == "bottom" && context[1] == "cols")) {
    ldend_side <- TRUE
  } else if ((dend_side == "right" && context[1] == "rows") ||
             (dend_side == "top" && context[1] == "cols")) {
    ldend_side <- FALSE
  } else {
    var_name <- switch(context[1], "rows" = "annot_rows_side", "cols" = "annot_cols_side")
    val_name <- switch(context[1], "rows" = c("left", "right"), "cols" = c("top", "bottom"))
    cli::cli_warn("{.var {var_name}} should be {.val {val_name[1]}} or {.val {val_name[2]}},
                  not {.val {dend_side}}.",
                  class = "dend_side_warn")
    ldend_side <- switch(context[1], "rows" = FALSE, "cols" = TRUE)
  }
  # Annotation side validity is checked earlier in prepare_annotation
  lannot_side <- ifelse(annot_side %in% c("left", "bottom"), TRUE, FALSE)

  context <- context[1]
  # Segments
  dend_seg <- dendro_in$segments
  # Nodes
  dend_nod  <- dendro_in$nodes


  dend_seg <- orient_dendrogram(dend_seg, context, full_plt, layout, ldend_side)

  # Move dendrogram next to heatmap
  dend_seg <- move_dendrogram(dend_seg, x_long, context, ldend_side,
                              dend_dist, annot_df, lannot_side, annot_pos, annot_size)

  # Rescale height of dendrogram
  if (dend_height != 1) {
    dend_seg <- scale_dendrogram(dend_seg, context, ldend_side, dend_height)
  }

  # If nodes have been adjusted in any way, repeat the procedure for nodes
  if (any(!is.na(unlist(dplyr::select(dend_nod, pch, cex, col))))) {

    # Add xend and yend columns to work with transformation functions
    dend_nod <- dplyr::mutate(dend_nod, xend = x, yend = y)

    dend_nod <- orient_dendrogram(dend_nod, context, full_plt, layout, ldend_side)

    # Move dendrogram next to heatmap
    dend_nod <- move_dendrogram(dend_nod, x_long, context, ldend_side,
                                dend_dist, annot_df, lannot_side, annot_pos, annot_size)

    # Rescale height of dendrogram
    if (dend_height != 1) {
      dend_nod <- scale_dendrogram(dend_nod, context, ldend_side, dend_height)
    }

  }

  # Get the same behaviour as in dendextend (nodes_pch must be specified for nodes to show up,
  # but setting nodes_pch automatically sets col and cex to defaults)
  # If this is done before 'move_dendrogram', the nodes can end up in the wrong positions as
  # it tries to move the nodes so they start at the heatmap edge (bad if some nodes are NA and removed)
  dend_nod <- dplyr::filter(dend_nod, !is.na(pch))
  dend_nod[["cex"]][is.na(dend_nod[["cex"]])] <- 1
  dend_nod[["col"]][is.na(dend_nod[["col"]])] <- "black"

  return(list("seg" = dend_seg, "nod" = dend_nod, "params" = dend_params))
}


#' Cluster data using hierarchical clustering or use provided clustering.
#'
#' @keywords internal
#'
#' @param cluster_input Either a logical indicating if data should be clustered, or a `hclust` or `dendrogram` object.
#' @param mat Matrix to cluster.
#' @param cluster_distance Distance metric for clustering.
#' @param cluster_method Clustering method for `hclust`.
#' @param dend_options List or functional sequence specifying `dendextend` functions to use.
#'
#' @returns List containing the dendrogram and clustering objects.
#'
cluster_data <- function(cluster_input, mat, cluster_distance, cluster_method, dend_options = NULL) {
  UseMethod("cluster_data")
}

#' @export
cluster_data.default <- function(cluster_input, mat, cluster_distance, cluster_method, dend_options = NULL) {
  # Error if incompatible class
  cli::cli_abort("{.var cluster_rows} and {.var cluster_cols} must be {.cls logical}, {.cls hclust},
                   or {.cls dendrogram} objects.",
                 class = "clust_class_error")

  return(NULL)
}

#' @export
cluster_data.logical <- function(cluster_input, mat, cluster_distance, cluster_method, dend_options = NULL) {
  # If TRUE, cluster the data
  if (cluster_input) {
    clust <- hclust(dist(mat, method = cluster_distance), method = cluster_method)
    dendro <- as.dendrogram(clust)

    dendro <- process_dendrogram(mat, dendro, dend_options)

    return(list("dendro" = dendro, "clust" = clust))
  }
}

#' @export
cluster_data.hclust <- function(cluster_input, mat, cluster_distance, cluster_method, dend_options = NULL) {
  # If hclust object use as is and make a dendrogram from it
  clust <- cluster_input
  dendro <- as.dendrogram(clust)

  dendro <- process_dendrogram(mat, dendro, dend_options)

  return(list("dendro" = dendro, "clust" = clust))
}

#' @export
cluster_data.dendrogram <- function(cluster_input, mat, cluster_distance, cluster_method, dend_options = NULL) {
  # If a dendrogram use as is, but also get a hclust object out of it
  clust <- as.hclust(cluster_input)
  dendro <- cluster_input

  dendro <- process_dendrogram(mat, dendro, dend_options)

  return(list("dendro" = dendro, "clust" = clust))
}


#' Process dendrogram with customisation options
#'
#' @keywords internal
#'
#' @param mat Data that was used for clustering.
#' @param dendro Dendrogram object.
#' @param dend_options Dendrogram extension options (list or fseq or NULL).
#'
#' @returns Processed dendrogram object.
#'
process_dendrogram <- function(mat, dendro, dend_options) {
  # Apply dendextend options if any are given
  if (!is.null(dend_options)) {
    dendro <- apply_dendextend(dend_options, dendro)
  }

  dendro <- dendextend::as.ggdend(dendro)
  dendro$labels$label <- as.character(dendro$labels$label)

  # Add the labels to the segment data frame to later compare final coordinates with plot coordinate system
  dendro$segments$lbl <- dendro$labels[match(dendro$segments$x, dendro$labels$x), "label"]

  if (any(!dendro$labels$label %in% rownames(mat))) {
    cli::cli_abort("The labels in the clustering don't match the labels in the data.",
                   class = "cluster_labels_error")
  }

  return(dendro)
}


#' Orient a dendrogram.
#'
#' @keywords internal
#'
#' @param dend Dendrogram segments or nodes data frame (containing x, y, xend, yend).
#' @param dim String, rows or cols to know which dimensions dendrogram should be on.
#' @param full_plt Logical indicating if it's for the full layout.
#' @param layout The heatmap layout to take mixed layout into account.
#' @param dend_side Logical indicating if the dendrogram should be placed on the left (if row dend) or bottom (if col dend).
#'
#' @returns The input dendrogram data frame but rotated and mirrored to fit the plot.
#'
orient_dendrogram <- function(dend, dim = c("rows", "cols"), full_plt, layout, dend_side) {
  y <- yend <- x <- xend <- nx <- nxend <- ny <- nyend <- NULL

  # If mixed layout (treated as full plot) with topleft and bottomright, the row dendrogram must be flipped
  mixed_tl_br <- if (length(layout) == 2) {
    sum(c("tl", "topleft") %in% layout) == 1 & sum(c("br", "bottomright") %in% layout) == 1
  } else {
    FALSE
  }

  dend_new <- if (full_plt) {
    if (dim[1] == "rows" & dend_side) {
      # dend_left row dendrogram
      dplyr::mutate(dend, nx = -y, nxend = -yend,
                    ny = -x * ifelse(mixed_tl_br, -1, 1), nyend = -xend * ifelse(mixed_tl_br, -1, 1))
    } else if (dim[1] == "rows" & !dend_side) {
      # right row dendrogram
      dplyr::mutate(dend, nx = y, nxend = yend,
                    ny = -x * ifelse(mixed_tl_br, -1, 1), nyend = -xend * ifelse(mixed_tl_br, -1, 1))
    } else if (dim[1] == "cols" & dend_side) {
      # bottom column dendrogram
      dplyr::mutate(dend, nx = x, nxend = xend, ny = -y, nyend = -yend)
    } else if (dim[1] == "cols" & !dend_side) {
      # top column dendrogram
      dplyr::mutate(dend, nx = x, nxend = xend, ny = y, nyend = yend)
    }
  } else {
    if (layout %in% c("tl", "topleft")) {
      # top left
      if (dim[1] == "rows") {
        dplyr::mutate(dend, nx = -y, nxend = -yend, ny = x, nyend = xend)
      } else if (dim[1] == "cols") {
        dplyr::mutate(dend, nx = x, nxend = xend, ny = y, nyend = yend)
      }
    } else if (layout %in% c("tr", "topright")) {
      # top right
      if (dim[1] == "rows") {
        dplyr::mutate(dend, nx = y, nxend = yend, ny = -x, nyend = -xend)
      } else if (dim[1] == "cols") {
        dplyr::mutate(dend, nx = x, nxend = xend, ny = y, nyend = yend)
      }
    } else if (layout %in% c("br", "bottomright")) {
      # bottom right
      if (dim[1] == "rows") {
        dplyr::mutate(dend, nx = y, nxend = yend, ny = x, nyend = xend)
      } else if (dim[1] == "cols") {
        dplyr::mutate(dend, nx = x, nxend = xend, ny = -y, nyend = -yend)
      }
    } else if (layout %in% c("bl", "bottomleft")) {
      # bottom left
      if (dim[1] == "rows") {
        dplyr::mutate(dend, nx = -y, nxend = -yend, ny = -x, nyend = -xend)
      } else if (dim[1] == "cols") {
        dplyr::mutate(dend, nx = x, nxend = xend, ny = -y, nyend = -yend)
      }
    }
  }

  dend_new <- dplyr::select(dend_new, -x, -xend, -y, -yend)
  dend_new <- dplyr::rename(dend_new, x = nx, xend = nxend, y = ny, yend = nyend)

  return(dend_new)
}


#' Move coordinates of dendrogram to edges of heatmap
#'
#' @description
#' Move coordinates of dendrogram by calculating distance it has to move to end up at the desired edges of the heatmap
#'
#' @keywords internal
#'
#' @param dend_seg Data frame containing dendrogram segments, attained from `dendextend::as.ggdend()`
#' @param x_long Long format data frame with the values
#' @param context Character specifying whether the dendrogram is linked to rows or columns in the heatmap
#' @param dend_side Logical specifying dendrogram position. TRUE is left of the heatmap if row dendrogram, bottom of heatmap if column dendrogram
#' @param dend_dist Distance from heatmap (or annotation) to dendrogram in cell size.
#' @param annot_df Data frame with annotations for checking that annotations exist as well as their size
#' @param annot_side Logical specifying annotation position, analogous to `dend_side`
#' @param annot_pos Numeric vector of annotation coordinates (x coordinates for row annotations, y for column annotations)
#' @param annot_size Numeric of length 1, the specified size (width or height) of annotation cells
#'
#' @return Data frame with updated dendrogram coordinates
#'
move_dendrogram <- function(dend_seg, x_long, context = c("rows", "cols"), dend_side, dend_dist,
                            annot_df, annot_side, annot_pos, annot_size) {

  if (context[1] == "rows") {
    xmove <- ifelse(
      dend_side,
      # Dendrograms on the left. If there is no annotation start at 0.5 (center of leftmost cell is at 1, move half a cell left)
      # If there is annotation, need to take annotation width into account (annot_size)
      ifelse(    # Annotation left of plot or not
        is.data.frame(annot_df) & annot_side,
        min(annot_pos) - 0.5 * annot_size - max(c(dend_seg$x, dend_seg$xend)),
        0.5 - max(c(dend_seg$x, dend_seg$xend))
      ) - dend_dist,
      ifelse(    # Dendrograms on the right, annotation right of plot or not
        is.data.frame(annot_df) & !annot_side,
        max(annot_pos) + 0.5 * annot_size - min(c(dend_seg$x, dend_seg$xend)),
        length(unique(x_long$col)) + 0.5 - min(c(dend_seg$x, dend_seg$xend))
      ) + dend_dist
    )

    ymove <- length(unique(x_long$row)) - max(c(dend_seg$y, dend_seg$yend))

  } else {

    xmove <- 1 - min(c(dend_seg$x, dend_seg$xend))

    ymove <- ifelse(
      # Dendrogram above or below heatmap
      dend_side,
      ifelse(    # Dendrogram below heatmap, annotation or not
        is.data.frame(annot_df) & annot_side,
        min(annot_pos) - 0.5 * annot_size - max(c(dend_seg$y, dend_seg$yend)),
        0.5 - max(c(dend_seg$y, dend_seg$yend))
      ) - dend_dist,
      ifelse(    # Dendrogram above heatmap, annotation or not
        is.data.frame(annot_df) & !annot_side,
        max(annot_pos) + 0.5 * annot_size - min(c(dend_seg$y, dend_seg$yend)),
        length(unique(x_long$row)) + 0.5 - min(c(dend_seg$y, dend_seg$yend))
      ) + dend_dist
    )
  }

  dend_seg[, c("x", "xend")] <- dend_seg[, c("x", "xend")] + xmove
  dend_seg[, c("y", "yend")] <- dend_seg[, c("y", "yend")] + ymove

  return(dend_seg)
}


#' Scale height of dendrogram.
#'
#' @keywords internal
#'
#' @param dend_seg Data frame containing dendrogram segments in format obtained from `dendextend`.
#' @param context Dimension to draw dendrogram along (rows or columns).
#' @param dend_side Logical indicating which side to draw dendrogram on.
#' If row dendrogram TRUE is left. If column dendrogram TRUE is down.
#' @param dend_height Scaling parameter for dendrogram height (1 is no scaling).
#'
#' @return Data frame containing coordinates for dendrogram segments (and any colour, linewidth, line type parameters)
#'
scale_dendrogram <- function(dend_seg, context = c("rows", "cols"), dend_side, dend_height) {
  xend <- x <- yend <- y <- NULL

  if (context[1] == "rows") {
    dend_seg_out <- dplyr::mutate(
      dend_seg,
      # Move horizontal segments by scaling the distance from the points touching the heatmap
      x = ifelse(dend_side, max(xend), min(xend)) +
        (x - ifelse(dend_side, max(xend), min(xend))) * dend_height,
      xend = ifelse(dend_side, max(xend), min(xend)) +
        (xend - ifelse(dend_side, max(xend), min(xend))) * dend_height
    )
  } else {
    dend_seg_out <- dplyr::mutate(
      dend_seg,
      y = ifelse(!dend_side, min(yend), max(yend)) +
        (y - ifelse(!dend_side, min(yend), max(yend))) * dend_height,
      yend = ifelse(!dend_side, min(yend), max(yend)) +
        (yend - ifelse(!dend_side, min(yend), max(yend))) * dend_height
    )
  }

  return(dend_seg_out)
}

#' Check that dendrograms are positioned correctly
#'
#' @keywords internal
#'
#' @param dat Long format data for plotting.
#' @param context Dimension to which the dendrogram is added. These are used to directly get the columns
#' in the long data and thus need to be "row" or "col".
#' @param dendro The dendrogram segments and nodes list.
#'
#' @returns `dendro` is returned as is if the positions are correct. Otherwise there is an error.
#'
check_dendrogram_pos <- function(dat, context = c("row", "col"), dendro) {
  lbl <- NULL

  coord_dim <- if (context[1] == "row") "y" else if (context[1] == "col") "x" else NA

  seg <- dendro$seg
  dend_lab <- dplyr::select(seg, lbl, !!coord_dim)
  dend_lab <- dplyr::filter(dend_lab, !is.na(lbl))
  # Take only distinct rows, in case a segment ends up on the same coordinate as the lowest node
  dend_lab <- dplyr::distinct(dend_lab, lbl, !!coord_dim, .keep_all = TRUE)

  # Add plot coordinates of labels
  dend_lab$plt_coord <- seq_along(levels(dat[[context[1]]]))[match(dend_lab$lbl, levels(dat[[context[1]]]))]

  # Check if same coordinates (within tolerance)
  if (!all(dplyr::near(dend_lab[[coord_dim]], dend_lab[["plt_coord"]]))) {
    cli::cli_warn("Something went wrong with dendrogram positioning! The leaves may be in the wrong coordinates. Please inform the author.")
  }

  return(dendro)
}


#' Successively apply dendextend functions to a dendrogram.
#'
#' @keywords internal
#'
#' @param dendro Dendrogram object obtained from `stats::as.dendrogram`.
#' @param dend_list List specifying dendextend functions to apply. For usage see the details of `gghm`.
#'
#' @returns A dendrogram object modified with dendextend functions.
#'
apply_dendextend <- function(dend_list, dendro) {
  UseMethod("apply_dendextend")
}

#' @export
apply_dendextend.default <- function(dend_list, dendro) {
  # If not a list or functional sequence, return without modification
  cli::cli_warn("{.var dend_rows_extend} and {.var dend_cols_extend} must be a {.cls list}
                or {.cls fseq} object if not NULL.",
                class = "extend_class_warn")
  return(dendro)
}

#' @export
apply_dendextend.list <- function(dend_list, dendro) {
  # Go through options list and update the dendrogram successively with do.call
  # Use append() to make named list of input arguments
  for (i in seq_along(dend_list)) {
    # Make a temporary function by calling the provided function from dendextend
    dend_fun <- do.call(`::`, list("dendextend", names(dend_list)[i]))
    dendro <- do.call(dend_fun, append(list(dendro), dend_list[[i]]))
  }

  return(dendro)
}

#' @export
apply_dendextend.fseq <- function(dend_list, dendro) {
  # Apply the functional sequence and return the dendrogram
  dendro <- dend_list(dendro)

  return(dendro)
}
