% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{ddmc}
\alias{ddmc}
\alias{ddmc_parallel}
\title{Compute Probability Density of Drift-Diffusion Model}
\usage{
ddmc(x, pVec, precision = 2.5, minLike = 1e-10)

ddmc_parallel(x, pVec, precision = 2.5, minLike = 1e-10)
}
\arguments{
\item{x}{a data frame containing choices and RTs. This is akin to dnorm's x
argument.}

\item{pVec}{a parameter vector. For example,
p.vector <- c(a=1.25, v.f1=.20, v.f2=.33, z=.67, sz=.26, sv=.67, t0=.26)}

\item{precision}{a precision parameter. The larger, the higher precision to
estimate diffusion probability density. A general recommendation for the low
bound of precision is 2.5. If you need to use a precision higher than that,
you may want to consider using \code{ddmc_parallel}. A default value is set
at 2.5.}

\item{minLike}{a minimal log likelihood. If a estimated density is
less than minLike, the function returns minLike. A default value is set at
1e-10.}
}
\value{
a double vector with drift-diffusion probability density
}
\description{
This function implements the equations in Voss, Rothermund, and Voss (2004).
These equations calculate Ratcliff's drift-diffusion model (RDM, 1978).
\code{ddmc} re-implements Voss, Rothermund, and Voss's (2004) equations A1
to A4 (page 1217) via Rcpp. This Rcpp function is akin to DMC's
\code{likelihood.dmc}. There is a \code{ddmc_parallel} with same arguments
using Open MPI to calculate RDM probability densities when data sets are
large (e.g., more than 5000 trials per condition) or when precision is set
high.
}
\examples{
## Set up a basic RDM design
m1 <- model.dmc(
    p.map=list(a="1", v="1", z="1", d="1", sz="1", sv="1", t0="1", st0="1"),
    constants=c(st0=0, d=0),
    match.map=list(M=list(s1="r1", s2="r2")),
    factors=list(S=c("s1", "s2")),
    responses=c("r1", "r2"),
    type="rd")

pVec <- c(a=1, v=1, z=.5, sz=.25, sv=.2,t0=.15)

## Set up a model-data instance
raw.data <- simulate(m1, nsim=1e2, p.vector=pVec)
mdi <- data.model.dmc(raw.data, m1)

## Print probability densities on the screen
ddmc(mdi, pVec)
## [1]  0.674039828 0.194299671 0.377715949 1.056043402 0.427508806
## [6]  2.876521367 0.289301835 2.701315781 0.419909813 1.890380685
## [11] 2.736803321 1.278188278 0.607926999 0.024858130 1.878312335
## [16] 0.826923114 0.189952016 1.758620294 1.696875882 2.191290074
## [21] 2.666750382 ...

mdi.large <- data.model.dmc(simulate(m1, nsim=1e4, p.vector=pVec), m1)

## d.large <- dplyr::tbl_df(mdi.large)
## d.large
## Source: local data frame [20,000 x 3]
##
##         S      R        RT
##    (fctr) (fctr)     (dbl)
## 1      s1     r2 0.2961630
## 2      s1     r1 0.9745674
## 3      s1     r2 0.6004945
## 4      s1     r1 0.1937356
## 5      s1     r1 0.3608844
## 6      s1     r1 0.5343379
## 7      s1     r1 0.2595707
## 8      s1     r1 0.3247941
## 9      s1     r1 0.4901325
## 10     s1     r2 0.5011935
## ..    ...    ...       ...

## system.time(den1 <- ddmc(mdi.large, pVec))
##  user  system elapsed
## 0.028   0.004   0.046

## system.time(den2 <- ddmc_parallel(mdi.large, pVec))
##  user  system elapsed
## 0.200   0.000   0.023
## all.equal(den1, den2)
## [1] TRUE
}
\references{
Voss, A., Rothermund, K., & Voss, J. (2004).  Interpreting the
parameters of the diffusion model: An empirical validation.
\emph{Memory & Cognition}, \bold{32(7)}, 1206-1220. \cr\cr
Ratcliff, R. (1978). A theory of memory retrival. \emph{Psychological
Review}, \bold{85}, 238-255.
}

