% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/guide_axis_nested_date.R
\name{guide_axis_nested_date}
\alias{guide_axis_nested_date}
\title{Nested axis guide for date scales}
\usage{
guide_axis_nested_date(
  sep = "[^[:alnum:]]+",
  regular_key = "auto",
  type = "bracket",
  mode = "simple",
  pad_date = NULL,
  oob = "none",
  ...
)
}
\arguments{
\item{sep}{A regular expression pattern used to split axis labels into
hierarchical components. Default is \code{"[^[:alnum:]]+"} which splits on
non-alphanumeric characters.}

\item{regular_key}{Default is \code{"auto"}, which generates the nested axis based on the date labels and the separator above.
This option can be used to provide your own specification for the nested key. See \code{\link[legendry:key_standard]{legendry::key_standard()}}}

\item{type}{The visual type of nested axis guide to create. Options include:
\itemize{
\item \code{"bracket"} (default): Creates bracket-style nested labels
\item \code{"fence"}: Creates fence-style nested labels (like Excel)
\item \code{"box"}: Creates box-style nested labels
}}

\item{mode}{Processing mode for the guide. Default is \code{"simple"}. Currently, this is the only supported mode.}

\item{pad_date}{Numeric value controlling the padding around date levels,
i.e. extending the length of the bracket or box or for correctly positioning the fences.
If \code{NULL} (default), automatically sets to 0.5 for "fence" type and
0.25 for other types.}

\item{oob}{How to handle out-of-bounds values of the scale labels. Default is \code{"none"}.
Another option is \code{"squish"}, but this can result in overlapping labels.}

\item{...}{Additional arguments passed to \code{\link[legendry:guide_axis_nested]{legendry::guide_axis_nested()}}.}
}
\value{
A nested axis guide object that can be used with \code{\link[ggplot2:scale_date]{ggplot2::scale_x_date()}} etc. or \code{\link[ggplot2:guides]{ggplot2::guides()}}.
}
\description{
A specialized axis guide for date scales that creates nested axis labels
by automatically detecting hierarchical patterns in date labels (e.g.,
separating day-month from year components). This guide is particularly
useful for time series data, where the axis can get crowded when
showing the full dates. This is similar to the date scale from Excel.
}
\examples{
library(ggplot2)

# Create sample epidemic curve data
epi_data <- data.frame(
  date = rep(as.Date("2023-12-15") + 0:100, times = rpois(101, 2))
)

ggplot(epi_data, aes(x = date)) +
  geom_epicurve(date_resolution = "week") +
  scale_x_date(
    date_breaks = "2 weeks", date_labels = "\%d-\%b-\%Y",
    guide = guide_axis_nested_date()
  )

# Using fence type with ISO week labels
ggplot(epi_data, aes(x = date)) +
  geom_epicurve(date_resolution = "week") +
  scale_x_date(
    date_breaks = "2 weeks", date_labels = "W\%V.\%G",
    guide = guide_axis_nested_date(type = "fence")
  )

# Using box type with custom padding
ggplot(epi_data, aes(x = date)) +
  geom_epicurve(date_resolution = "month") +
  scale_x_date(
    date_breaks = "1 month", date_labels = "\%b.\%Y",
    guide = guide_axis_nested_date(type = "box", pad_date = 0.3)
  )

# Custom separator for different label formats
ggplot(epi_data, aes(x = date)) +
  geom_epicurve(date_resolution = "week") +
  scale_x_date(
    date_breaks = "1 week", date_labels = "\%d-\%b-\%Y",
    guide = guide_axis_nested_date(type = "bracket", sep = "-")
  )

# Datetime example with fence type
datetime_data <- data.frame(
  datetime = rep(as.POSIXct("2024-02-05 01:00:00") + 0:50 * 3600,
    times = rpois(51, 3)
  )
)

ggplot(datetime_data, aes(x = datetime)) +
  geom_epicurve(date_resolution = "2 hours") +
  scale_x_datetime(
    date_breaks = "6 hours", date_labels = "\%Hh \%e.\%b",
    limits = c(as.POSIXct("2024-02-04 22:00:00"), NA),
    guide = guide_axis_nested_date()
  )

}
\seealso{
\code{\link[legendry:guide_axis_nested]{legendry::guide_axis_nested()}}
}
