% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kmeans.R
\name{gmfd_kmeans}
\alias{gmfd_kmeans}
\title{k-means clustering algorithm}
\usage{
gmfd_kmeans(FD, n.cl = 2, metric, p = NULL, k_trunc = NULL)
}
\arguments{
\item{FD}{a functional data object of type \code{funData}.}

\item{n.cl}{an integer representing the number of clusters.}

\item{metric}{the chosen distance to be used: \code{"L2"} for the classical L2-distance, \code{"trunc"} for the truncated Mahalanobis semi-distance, \code{"mahalanobis"} for the generalized Mahalanobis distance.}

\item{p}{a positive numeric value containing the parameter of the regularizing function for the generalized Mahalanobis distance.}

\item{k_trunc}{a positive numeric value representing the number of components at which the truncated mahalanobis distance must be truncated}
}
\value{
The function returns a list with the following components:
\code{cluster}: a vector of integers (from \code{1} to \code{n.cl}) indicating the cluster to which each curve is allocated;
\code{centers}: a list of \code{d} matrices (\code{k} x \code{T}) containing the centroids of the clusters
}
\description{
This function performs a k-means clustering algorithm on an univariate or multivariate functional data using a generalization of Mahalanobis distance.
}
\examples{
# Define parameters
n <- 50
P <- 100
K <- 150

# Grid of the functional dataset
t <- seq( 0, 1, length.out = P )

# Define the means and the parameters to use in the simulation
m1 <- t^2 * ( 1 - t )

rho <- rep( 0, K )
theta <- matrix( 0, K, P )
for ( k in 1:K) {
  rho[k] <- 1 / ( k + 1 )^2
  if ( k\%\%2 == 0 )
    theta[k, ] <- sqrt( 2 ) * sin( k * pi * t )
  else if ( k\%\%2 != 0 && k != 1 )
    theta[k, ] <- sqrt( 2 ) * cos( ( k - 1 ) * pi * t )
  else
    theta[k, ] <- rep( 1, P )
}

s <- 0
for (k in 4:K) {
 s <- s + sqrt( rho[k] ) * theta[k, ]
}

m2 <- m1 + s

# Simulate the functional data
x1 <- gmfd_simulate( n, m1, rho = rho, theta = theta )
x2 <- gmfd_simulate( n, m2, rho = rho, theta = theta )

# Create a single functional dataset containing the simulated datasets:
FD <- funData(t, rbind( x1, x2 ) )

output <- gmfd_kmeans( FD, n.cl = 2, metric = "mahalanobis", p = 10^6 )
}
\references{
Martino A., Ghiglietti A., Ieva F., Paganoni A. M. (2017). A k-means procedure based on a Mahalanobis type
distance for clustering multivariate functional data, \emph{MOX report 44/2017}

Ghiglietti A., Ieva F., Paganoni A. M. (2017). Statistical inference for stochastic processes:
Two-sample hypothesis tests, \emph{Journal of Statistical Planning and Inference}, 180:49-68.

Ghiglietti A., Paganoni A. M. (2017). Exact tests for the means of gaussian stochastic processes.
\emph{Statistics & Probability Letters}, 131:102--107.
}
\seealso{
\code{\link{funDist}}
}
\keyword{Clustering}
