% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/joint.R
\name{joint}
\alias{joint}
\title{Fit a joint model to time-to-event and multivariate longitudinal data}
\usage{
joint(
  long.formulas,
  surv.formula,
  data,
  family,
  post.process = TRUE,
  control = list()
)
}
\arguments{
\item{long.formulas}{A list of formula objects specifying the \eqn{K} responses. Each must be 
usable by \code{\link[glmmTMB]{glmmTMB}}. A restriction is that unique identifiers must 
be named \code{id}, and increment in intervals of at exactly one. The variable for time
must be named \code{time}.}

\item{surv.formula}{A formula specifying the time-to-event sub-model. Must be usable by 
\code{\link[survival]{coxph}}.}

\item{data}{A \code{data.frame} containing all covariates and responses.}

\item{family}{A list of families corresponding in order to \code{long.formula}.}

\item{post.process}{Logical, should post processing be done to obtain standard errors and 
log-likelihood? Defaults to \code{TRUE}.}

\item{control}{A list of control values: \describe{

  \item{\code{verbose}}{Logical: If \code{TRUE}, at each iteration parameter information will 
  be printed to console. Default is \code{verbose=FALSE}.}
  \item{\code{conv}}{Character: Convergence criterion, see \strong{details}.}
  \item{\code{tol.abs}}{Numeric: Tolerance value used to assess convergence, see 
  \strong{details}. Default is \code{tol.abs=1e-3}.}
  \item{\code{tol.rel}}{Numeric: Tolerance value used to assess convergence, see 
  \strong{details}. Default is \code{tol.rel=1e-2}.}
  \item{\code{tol.den}}{Numeric: Tolerance value used to assess convergence, see 
  \strong{details}. Default is \code{tol.den=1e-3}.}
  \item{\code{tol.thr}}{Numeric: Threshold used when \code{conv = 'sas'}, see 
  \strong{details}. Default is \code{tol.thr=1e-1}.}
  \item{\code{correlated}}{Logical: Should covariance parameters \strong{between} responses 
  be estimated and used in determination of model convergence? Default is 
  \code{correlated=TRUE}. A choice of \code{correlated=FALSE} is equivalent to imposing the 
  belief that deviations in longitudinal trajectories are not correlated across responses, but
  can \strong{greatly decrease} computation time.}
  \item{\code{gh.nodes}}{Integer: Number of weights and abscissae to use in gauss-hermite 
  quadrature. Defaults to \code{gh.nodes=3}, which is usually sufficient.}
  \item{\code{gh.sigma}}{Numeric: Standard deviation for gauss-hermite approximation of normal
  distribution. Defaults to \code{gh.sigma=1}. This should rarely (if ever) need altering.}
  \item{\code{hessian}}{Character: Determines if the variance-covariance matrix for 
  \eqn{\hat{b}_i}, \eqn{\hat{\Sigma}_i} should be calculated as part of the \code{optim} step
  in minimising the negative log-likelihood, or calculated post-hoc using forward differencing.
  Default is \code{hessian="auto"} for the former, with \code{hessian="manual"} the 
  option for the latter.}
  \item{\code{return.dmats}}{Logical: Should data matrices be returned? Defaults to 
  \code{return.dmats=TRUE}. Note that some S3 methods for \code{\link{joint.object}}s
  greatly benefit from inclusion of these data matrices.}
  \item{\code{center.ph}}{Should the survival covariates be mean-centered? Defaults
  to \code{center.ph=TRUE}.}

}}
}
\value{
An object with class \code{joint}. See \code{\link{joint.object}} for information.
}
\description{
Fit a joint model to time-to-event and multivariate longitudinal data
}
\details{
Function \code{joint} fits a joint model to time-to-event data and multivariate 
longitudinal data. The longitudinal data can be specified by numerous models encompassing
a fairly wide range of data. This joint model fit is achieved by the use of an approximate
EM algorithm first proposed in Bernhardt et al. (2015), and later used in the 'classic' 
multivariate joint model in Murray and Philipson (2022). Each longitudinal response is 
modelled by 

\deqn{h_k(E[Y_{ik}|b_{ik};\Omega]) = X_{ik}\beta_k + Z_{ik}b_{ik}} 

where \eqn{h_k} is a known, monotonic link function. An association is induced between the 
\eqn{K}th response and the hazard \eqn{\lambda_i(t)} by: 

\deqn{\lambda_i(t)=\lambda_0(t)\exp\{S_i^T\zeta + \sum_{k=1}^K\gamma_kW_k(t)^Tb_{ik}\}} 

where \eqn{\gamma_k} is the association parameter and \eqn{W_k(t)} is the vector function of 
time imposed on the \eqn{K}th random effects structure (i.e. intercept-and-slope; spline).
}
\section{Family specification}{

  Currently, five families are available for implementation, spanning continuous, binary and 
  count data types: \describe{
  
    \item{\code{'gaussian'}}{Normally distributed. The identity link is used. A term 
    \eqn{\sigma_k} will be estimated, denoting the \emph{variance} of this response}
    \item{\code{'binomial'}}{For binary data types, a logit link is used.}
    \item{\code{'poisson'}}{For count data types where dispersion is either non-consequential 
    or ignored. A log link is used.}
    \item{\code{'genpois'}}{For count data types where dispersion is at least of some
    secondary interest. A log link is used. A term \eqn{\sigma_k} is estimated, denoting
    the dispersion, \eqn{\varphi} of the response. This follows interpretation of Zamani & 
    Ismail (2012): \eqn{\varphi>0}: Over-dispersion; \eqn{\varphi<0}: Under-dispersion.
    \eqn{Var[Y]=(1+\varphi)^2\mu}.}
    \item{\code{'Gamma'}}{For continuous data where a Gamma distribution might be sensible.
    The log link is used. A term \eqn{\sigma_k} is be estimated, denoting the shape of the
    distribution.}
  
  } 
  
  For families where dispersion is estimated, this is \strong{always} specified by an 
  "intercept-only" formula only. This might change in future.
}

\section{Standard error estimation}{
 
  We follow the approximation of the observed empirical information matrix detailed by 
  Mclachlan and Krishnan (2008), and later used in \code{joineRML} (Hickey et al., 2018).
  These are only calculated if \code{post.process=TRUE}. Generally, these SEs are well-behaved,
  but their reliability will depend on multiple factors: Sample size; number of events; 
  collinearity of REs of responses; number of observed times, and so on. Some more discussion/
  references are given in \code{\link{vcov.joint}}.
}

\section{Convergence of the algorithm}{

A few convergence criteria (specified by \code{control$conv}) are available: \describe{
  \item{\code{abs}}{Convergence reached when maximum absolute change in parameter estimates
  is \code{<tol.abs}.}
  \item{\code{rel}}{Convergence reached when maximum absolute relative change in parameter
  estimates is \code{<tol.rel}. A small amount (\code{tol.den}) is added to the denominator 
  to eschew numerical issues if parameters are nearly zero.}
  \item{\code{either}}{Convergence is reached when either \code{abs} or \code{rel} are met.}
  \item{\code{sas}}{Assess convergence for parameters \eqn{|\Omega_a|}\code{<tol.thr} by the
  \code{abs} criterion, else \code{rel}. This is the default.}

}
Note that the baseline hazard is updated at each EM iteration, but is not monitored for 
convergence.
}

\examples{

# 1) Fit on simulated bivariate data, (1x gaussian, 1x poisson) --------
beta <- do.call(rbind, replicate(2, c(2, -0.1, 0.1, -0.2), simplify = FALSE))
gamma <- c(0.3, -0.3)
D <- diag(c(0.25, 0.09, 0.25, 0.05))
family <- list('gaussian', 'poisson')
data <- simData(ntms = 10, beta = beta, D = D, n = 100,
                family = family, zeta = c(0, -0.2),
                sigma = c(0.16, 0), gamma = gamma)$data

# Specify formulae and target families
long.formulas <- list(
  Y.1 ~ time + cont + bin + (1 + time|id),  # Gaussian
  Y.2 ~ time + cont + bin + (1 + time|id)  # Poisson
)
surv.formula <- Surv(survtime, status) ~ bin

fit <- joint(long.formulas, surv.formula, data, family)

\donttest{
# 2) Fit on PBC data -----------------------------------------------------
data(PBC)
PBC$serBilir <- log(PBC$serBilir)

# Subset data and remove NAs
PBC <- subset(PBC, select = c('id', 'survtime', 'status', 'drug', 'time',
                              'serBilir', 'albumin', 'spiders', 'platelets'))
PBC <- na.omit(PBC) 

# Specify GLMM sub-models, including interaction and quadratic time terms
long.formulas <- list(
  serBilir ~ drug * (time + I(time^2)) + (1 + time + I(time^2)|id),
  albumin ~ drug * time + (1 + time|id),
  platelets ~ drug * time + (1 + time|id),
  spiders ~ drug * time + (1|id)
)
surv.formula <- Surv(survtime, status) ~ drug

fit <-  joint(long.formulas, surv.formula, PBC, 
              family = list("gaussian", "gaussian", "poisson", "binomial"),
              control = list(verbose = TRUE))
fit
}
}
\references{
Bernhardt PW, Zhang D and Wang HJ. A fast EM Algorithm for Fitting Joint Models of a Binary 
Response to Multiple Longitudinal Covariates Subject to Detection Limits. 
\emph{Computational Statistics and Data Analysis} 2015; \strong{85}; 37--53

Hickey GL, Philipson P, Jorgensen A, Kolamunnage-Dona R. \code{joineRML}: a joint model and
software package for time-to-event and multivariate longitudinal outcomes.
\emph{BMC Med. Res. Methodol.} 2018; \strong{50}
 
McLachlan GJ, Krishnan T. \emph{The EM Algorithm and Extensions.} Second Edition. 
Wiley-Interscience; 2008.

Murray, J and Philipson P. A fast approximate EM algorithm for joint models of survival and
multivariate longitudinal data.\emph{Computational Statistics and Data Analysis} 2022; 
\strong{170}; 107438

Zamani H and Ismail N. Functional Form for the Generalized Poisson Regression Model, 
\emph{Communications in Statistics - Theory and Methods} 2012; \strong{41(20)}; 3666-3675.
}
\seealso{
\code{\link{summary.joint}}, \code{\link{logLik.joint}}, 
\code{\link{extractAIC.joint}}, \code{\link{fixef.joint}}, \code{\link{ranef.joint}},
\code{\link{vcov.joint}} and \code{\link{joint.object}}.
}
\author{
James Murray (\email{j.murray7@ncl.ac.uk}).
}
