% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lipsitz.R
\name{lipsitz}
\alias{lipsitz}
\title{Lipsitz Test for Categorical Response Models}
\usage{
lipsitz(model, group = 10, customFreq = NULL)
}
\arguments{
\item{model}{a model object or data.frame of observed and estimated
values. The following class of objects can be directly passed to the
\code{lipsitz} function: vglm(), serp(), polr(), and clm(). Other class of
objects require providing a data.frame of observed and predicted values.}

\item{group}{Default to 10. The number of groups to be formed from the
original observations.}

\item{customFreq}{a vector of custom group frequencies to be used instead of
the default equal group frequencies. The vector should, however, sum up to
the total number of original observations.}
}
\value{
\item{LR}{the realized value of the likelihood ratio statistic.}

\item{df}{the degrees of freedom.}

\item{p.value}{the p-value of the test.}

\item{newVar}{a numeric vector of the newly generated variable.}
}
\description{
This provides a post estimation goodness-of-fit test for
the ordinal response models. Supported models include the proportional odds,
adjacent-category, and constrained continuation-ratio models.
}
\details{
Similar to the ordinal Hosmer-Lemeshow test (\code{\link{hosmerlem}}),
the Lipsitz test also group the observations into k separate groups using
the ordinal scores of the estimated values. According to Lipsitz, Fitzmaurice,
and Molenberghs (1996), the number of groups should be such that 6 <= k < n/5r,
with r the number of response category. An indicator variable is used to
denote the observations belonging to each group, producing additional pseudo
variables with which the original model is updated. Supposing the original
model fits correctly, then the coefficients of the pseudo variables all equal
zero. The likelihood ratio statistic calculated from the log likelihoods of
the original and the refitted models is subsequently compared with the
chi-squared distribution with k - 1 degrees of freedom.

The Lipsitz test compliments the ordinal Hosmer-Lemeshow and the
Pulkstenis-Robinson tests. Fagerland and Hosmer (2013, 2016, 2017)
recommend comparing the three test.
}
\examples{

require(VGAM)

set.seed(1)
n <- 200
y <- ordered(rbinom(n, 2, 0.5))
x1 <- factor(rbinom(n, 1, 0.7))
x2 <- runif(n)

## proportional odds model
vg <- vglm(y ~ x1 + x2, model = TRUE,
           family = cumulative(link = "logitlink", parallel = TRUE))
lipsitz(vg, group=6)

## adjacent category model
ac <- update(vg, family = acat(parallel = TRUE))
lipsitz(ac)

## continuation ratio model
cr <- update(vg, family = cratio(parallel = TRUE))
lipsitz(cr)

}
\references{
Fagerland, M. W. and Hosmer, D. W. (2013). A goodness-of-fit test for the
    proportional odds regression model. \emph{Statistics in Medicine},
    32, 2235-2249.

Fagerland, M. W. and Hosmer, D. W. (2016). Tests for goodness of fit in
    ordinal logistic regression models. \emph{Journal of Statistical
    Computation and Simulation}, 86, 3398-3418.

Fagerland, M. W. and Hosmer, D. W. (2017). How to test for goodness of fit
    in ordinal logistic regression models. \emph{Stata Journal}, 17, 668-686.
}
\seealso{
\code{\link{hosmerlem}}, \code{\link{pulkroben}}, \code{\link{brant.test}},
\code{\link{LR.test}}
}
