% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scenarios-fr.R
\name{flm_term}
\alias{flm_term}
\title{Functional linear model term with bivariate kernel}
\usage{
flm_term(X_fdata, beta, t, int_rule = "trapezoid", equispaced = NULL,
  concurrent = FALSE)
}
\arguments{
\item{X_fdata}{sample of functional data as an
\code{\link[fda.usc]{fdata}} object of length \code{n}.}

\item{beta}{matrix containing the values  \eqn{\beta(s, t)},
for each grid point \eqn{s} in  \eqn{[a, b]} and \eqn{t} in \eqn{[c, d]}. If
\code{concurrent = TRUE}, a row/column vector must be introduced, valued in
the same grid as \code{error_fdata}, with the same length as
\code{length(X_fdata$argvals)}.}

\item{t}{vectors with grid points where functions are valued.}

\item{int_rule}{quadrature rule for approximating the definite
unidimensional integral: trapezoidal rule (\code{int_rule = "trapezoid"})
and extended Simpson rule (\code{int_rule = "Simpson"}) are available.
Defaults to \code{"trapezoid"}.}

\item{equispaced}{flag to indicate if \code{X_fdata$data} is valued in
an equispaced grid or not. Defaults to \code{FALSE}.}

\item{concurrent}{flag to consider a concurrent FLMFR (degenerate case).
Defaults to \code{FALSE}.}
}
\value{
Functional linear model term as the integral (in \code{s}) between
\code{X_fdata} and \code{beta}, as an \code{\link[fda.usc]{fdata}} object of
length \code{n}.
}
\description{
Computation of the functional linear term
\deqn{\int_a^b \beta(s, t) X(s)\,\mathrm{d}s,}{\int_a^b \beta(s, t) X(s) ds,}
of a Functional Linear Model with Functional Response (FLMFR), where
\eqn{X} is a random variable in the Hilbert space of
square-integrable functions in \eqn{[a, b]}, \eqn{L^2([a, b])},
\eqn{\beta} is the bivariate kernel of the FLMFR, and
\eqn{\varepsilon}{\epsilon} is a random variable in \eqn{L^2([c, d])}.
}
\examples{
## Generate a sample of functional responses via FLMFR

# Bivariate kernel beta(s,t) as an egg carton shape
s <- seq(0, 1, l = 101)
t <- seq(0, 1, l = 201)
beta <- outer(s, t, FUN = function(s, t) sin(6 * pi * s) + cos(6 * pi * t))

# Functional data as zero-mean Gaussian process with exponential variogram
X_fdata <- fda.usc::rproc2fdata(n = 50, t = s, sigma = "vexponential",
                                list = list(scale = 2.5))

# Functional error as an OU process with variance 0.075
sigma <- sqrt(0.075) * 2
error_fdata <- r_ou(n = 50, t = t, sigma = sigma)
Y_fdata <- flm_term(X_fdata = X_fdata, beta = beta, t = t) + error_fdata
plot(Y_fdata)

## Generate a sample of functional responses via concurrent model

# Function beta(t)
s <- seq(1, 3, l = 201)
t <- seq(2, 5, l = 201)
beta <- sin(pi * t) + cos(pi * t)

# Functional data as zero-mean Gaussian process with exponential variogram
X_fdata <- fda.usc::rproc2fdata(n = 50, t = s, sigma = "vexponential",
                                list = list(scale = 2.5))

# Functional error as an OU process with variance 0.075
sigma <- sqrt(0.075) * 2
error_fdata <- r_ou(n = 50, t = t, sigma = sigma)
Y_fdata <- flm_term(X_fdata = X_fdata, beta = beta, t = t,
                    concurrent = TRUE) + error_fdata
plot(Y_fdata)
}
\references{
García-Portugués, E., Álvarez-Liébana, J., Álvarez-Pérez, G. and
Gonzalez-Manteiga, W. (2021). A goodness-of-fit test for the functional
linear model with functional response. \emph{Scandinavian Journal of
Statistics}, 48(2):502--528. \doi{10.1111/sjos.12486}
}
\author{
Javier Álvarez-Liébana.
}
