#' Builds a PubsubMessage Object
#'
#' @param data `character` The message data field as a base64 encoded string
#' @param message_id `character` ID of this message, assigned by the server when the message
#'   is published
#' @param ordering_key `character` If non-empty, identifies related messages for which publish 
#'   order should be respected
#' @param attributes `list` Key-value pairs attributes for this message
#' @param publish_time `character` The time at which the message was published, populated by the server 
#'   when it receives the `Publish` call
#'
#' @return `PubsubMessage` object
#'
#' @family Object functions
#' @export
PubsubMessage <- function(data = NULL, message_id = NULL,  ordering_key = NULL, 
                          attributes = NULL, publish_time = NULL) {
  structure(list(
    messageId = message_id, data = data, orderingKey = ordering_key,
    attributes = attributes, publishTime = publish_time
  ),
  class = c("PubsubMessage", "list")
  )
}

#' Builds a Topic Object
#'
#' @param name `character` Name of the topic
#' @param kms_key_name `character` The resource name of the Cloud KMS CryptoKey to be used
#'    to protect access to messages published on this topic
#' @param satisfies_pzs `logical` Reserved for future use
#' @param message_storage_policy `MessageStoragePolicy` Policy constraining the set of Google Cloud
#'   Platform regions where messages published to the topic may be stored
#' @param schema_settings `SchemaSettings` Settings for validating messages published against
#'   a schema
#' @param labels `list` Key-value pairs for topic labels
#' @param message_retention_duration `character` Indicates the minimum duration to retain a message 
#'   after it is published to the topic
#'
#' @return `Topic` object
#' @export
#'
#' @family Object functions
Topic <- function(labels = NULL, name = NULL, kms_key_name = NULL, satisfies_pzs = NULL,
                  message_storage_policy = NULL, schema_settings = NULL, 
                  message_retention_duration = NULL) {
  structure(list(
    labels                   = labels,
    name                     = name,
    kmsKeyName               = kms_key_name,
    satisfiesPzs             = satisfies_pzs,
    messageStoragePolicy     = message_storage_policy,
    schemaSettings           = schema_settings,
    messageRetentionDuration = message_retention_duration
  ),
  class = c("Topic", "list")
  )
}

#' Builds a DqlPolicy object
#'
#' @param dlq_topic `character`, `Topic` Required, topic name or instance of a topic object
#' @param max_delivery_attempts `numeric` Number of delivery attempts for any message. The 
#'   value must be between 5 and 100.
#'
#' @return `DlqPolicy` object
#' @export
#'
#' @family Object functions
DlqPolicy <- function(dlq_topic, max_delivery_attempts) {
  dlq_topic <- as.topic_name(dlq_topic)
  structure(
    list(deadLetterTopic = dlq_topic, maxDeliveryAttempts = max_delivery_attempts),
    class = c("DlqPolicy", "list")
  )
}

#' Builds a MessageStoragePolicy object
#'
#' @param regions `character` A list of IDs of GCP regions
#'
#' @return `MessageStoragePolicy` object
#' @export
#' 
#' @family Object functions
MessageStoragePolicy <- function(regions) {
  structure(
    list(allowedPersistenceRegions = regions),
    class = c("MessageStoragePolicy", "list")
  )
}

#' Builds a retry policy object
#' 
#' More on this \href{https://cloud.google.com/pubsub/docs/reference/rest/v1/projects.subscriptions#RetryPolicy}{here}
#' 
#' @param min_backoff `numeric` The minimum delay between consecutive deliveries of a 
#'   given message
#' @param max_backoff `numeric` The maximum delay between consecutive deliveries of a  
#'   given message
#'
#' @return `RetryPolicy` object
#' @export
#' 
#' @family Object functions
RetryPolicy <- function(min_backoff = 600, max_backoff = 600) {
  min_backoff <- secs_to_str(min_backoff)
  max_backoff <- secs_to_str(max_backoff) 
  structure(
    list(minimumBackoff = min_backoff, maximumBackoff = max_backoff),
    class = c("RetryPolicy", "list")
  )
}

# #' TestIamPermissionsResponse Object
# #'
# #' @details
# #' Autogenerated via \code{\link[googleAuthR]{gar_create_api_objects}}
# #' Response message for `TestIamPermissions` method.
# #'
# #' @param permissions A subset of `TestPermissionsRequest
# #'
# #' @return TestIamPermissionsResponse object
# #'
# #' @family TestIamPermissionsResponse functions
# #' @export
# TestIamPermissionsResponse <- function(permissions = NULL) {
#   structure(list(permissions = permissions), class = c(
#     "gar_TestIamPermissionsResponse",
#     "list"
#   ))
# }

#' Builds a PushConfig Object
#'
#' @param attributes `list` The attributes object or list of objects
#' @param push_endpoint `character` A URL locating the endpoint to which messages should 
#'   be pushed
#' @param oidcToken `character` If specified, Pub/Sub will generate and attach an OIDC JWT
#'   token as an `Authorization` header in the HTTP request for every pushed message
#'
#' @return `PushConfig` object
#' @export
#'
#' @family Object functions
PushConfig <- function(attributes = NULL, push_endpoint = NULL, oidcToken = NULL) {
  structure(list(
    pushEndpoint = push_endpoint, oidcToken = oidcToken, attributes = attributes
    ), class = c("PushConfig", "list")
  )
}

# #' Builds a TestIamPermissionsRequest Object
# #'
# #' @param permissions The set of permissions to check for the `resource`
# #'
# #' @return TestIamPermissionsRequest object
# #'
# #' @family TestIamPermissionsRequest functions
# #' @export
# TestIamPermissionsRequest <- function(permissions = NULL) {
#   structure(list(permissions = permissions), class = c(
#     "gar_TestIamPermissionsRequest",
#     "list"
#   ))
# }

#' Builds a Subscription Object
#'
#' @param dead_letter_policy `DeadLetterPolicy` A policy that specifies the conditions for dead
#'   lettering messages in this subscription
#' @param msg_retention_duration `character` How long to retain unacknowledged messages in
#'   the subscription's backlog, from the moment a message is published
#' @param labels `list` See Creating and managing labels
#' @param retry_policy `RetryPolicy` A policy that specifies how Pub/Sub retries message 
#'   delivery for this subscription
#' @param push_config `PushConfig` If push delivery is used with this subscription, this
#'   field is used to configure it
#' @param ack_deadline `character` The approximate amount of time (on a best-effort basis) 
#'   Pub/Sub waits for the subscriber to acknowledge receipt before resending the message
#' @param expiration_policy `ExpirationPolicy` A policy that specifies the conditions for
#'   this subscription's expiration
#' @param filter `character` An expression written in the Pub/Sub [filter language](https://cloud.google.com/pubsub/docs/filtering)
#' @param detached `logical` Indicates whether the subscription is detached from its topic
#' @param retain_acked_msgs `logical` Indicates whether to retain acknowledged messages
#' @param topic `character`, `Topic` A `Topic` object or topic name
#' @param name `character` A name for the subscription
#' @param enable_msg_ordering `logical` If true, messages published with the same `ordering_key` 
#'   in `PubsubMessage` will be delivered to the subscribers in the order in which they are
#'   received by the Pub/Sub system
#' @param topic_msg_retention `character` minimum duration for which a message is retained after
#'   it is published to the subscription's topic
#'
#' @return `Subscription` object
#' @export
#'
#' @family Object functions
Subscription <- function(dead_letter_policy = NULL,
                         msg_retention_duration= NULL,
                         labels = NULL,
                         retry_policy= NULL, 
                         push_config = NULL,
                         ack_deadline = NULL,
                         expiration_policy = NULL,
                         filter = NULL,
                         detached = NULL,
                         retain_acked_msgs = NULL,
                         topic = NULL,
                         name = NULL,
                         enable_msg_ordering = NULL,
                         topic_msg_retention = NULL) {
  structure(list(
    deadLetterPolicy              = dead_letter_policy,
    messageRetentionDuration      = msg_retention_duration,
    labels                        = labels,
    retryPolicy                   = retry_policy,
    pushConfig                    = push_config,
    ackDeadlineSeconds            = ack_deadline,
    expirationPolicy              = expiration_policy,
    filter                        = filter,
    detached                      = detached,
    retainAckedMessages           = retain_acked_msgs,
    topic                         = topic,
    name                          = name,
    enableMessageOrdering         = enable_msg_ordering,
    topicMessageRetentionDuration = topic_msg_retention
  ),
  class = c("Subscription", "list")
  )
}

#' ValidateMessageRequest Object
#'
#' @param message Message to validate against the provided `schema_spec`
#' @param name Name of the schema against which to validate
#' @param schema Ad-hoc schema against which to validate
#' @param encoding The encoding expected for messages
#'
#' @return `ValidateMessageRequest` object
#' @keywords internal
#' @noRd
#'
#' @family Object functions
ValidateMessageRequest <- function(message = NULL, name = NULL, schema = NULL, encoding = NULL) {
  structure(list(message = message, name = name, schema = schema, encoding = encoding),
    class = c("gar_ValidateMessageRequest", "list")
  )
}

#' Builds a Schema Object
#'
#' @param type `character` The type of the schema definition
#' @param definition `character` The definition of the schema
#' @param name `character` The schema name
#'
#' @return `Schema` object
#' @export
#'
#' @family Object functions
Schema <- function(type = NULL, definition = NULL, name = NULL) {
  structure(list(type = type, definition = definition, name = name), class = c(
    "Schema",
    "list"
  ))
}

# #' Binding Object
# #'
# #' @details
# #' Autogenerated via \code{\link[googleAuthR]{gar_create_api_objects}}
# #' Associates `members`, or principals, with a `role`.
# #'
# #' @param condition The condition that is associated with this binding
# #' @param members Specifies the principals requesting access for a Cloud Platform resource
# #' @param role Role that is assigned to the list of `members`, or principals
# #'
# #' @return Binding object
# #' @export
# #'
# #' @family Object functions
# Binding <- function(condition = NULL, members = NULL, role = NULL) {
#   structure(list(condition = condition, members = members, role = role), class = c(
#     "gar_Binding",
#     "list"
#   ))
# }

#' Build update request for an object
#'
#' @param object An object for which preparing a patch request
#'
#' @return An `Update*` object to be passed as a request body
#' @keywords internal
#' @noRd
#'
UpdateObjectRequest <- function(object = NULL) {
  get_update_mask <- function(object) {
    paste(names(object), collapse = ",")
  }

  if (inherits(object, "Topic")) {
    object <- rmNullObs(object)
    update_mask <- get_update_mask(object)
    out <- structure(list(updateMask = update_mask, topic = object), class = c(
      "UpdateTopicRequest",
      "list"
    ))
  } else if (inherits(object, "Subscription")) {
    object <- rmNullObs(object)
    update_mask <- get_update_mask(object)
    out <- structure(list(updateMask = update_mask, subscription = object), class = c(
      "UpdateSubscriptionRequest",
      "list"
    ))
  } else if (inherits(object, "Snapshot")) {
    object <- rmNullObs(object)
    update_mask <- get_update_mask(object)
    out <- structure(list(snapshot = object, updateMask = update_mask), class = c(
      "UpdateSnapshotRequest",
      "list"
    ))
  }

  return(out)
}

# #' SetIamPolicyRequest Object
# #'
# #' @details
# #' Autogenerated via \code{\link[googleAuthR]{gar_create_api_objects}}
# #' Request message for `SetIamPolicy` method.
# #'
# #' @param policy REQUIRED: The complete policy to be applied to the `resource`
# #'
# #' @return SetIamPolicyRequest object
# #'
# #' @family SetIamPolicyRequest functions
# #' @export
# SetIamPolicyRequest <- function(policy = NULL) {
#   structure(list(policy = policy), class = c("gar_SetIamPolicyRequest", "list"))
# }

#' SchemaSettings Object
#'
#' @param encoding `character` The encoding of messages validated against `schema`
#' @param schema `Schema, character` Required, schema object or schema name
#'
#' @return `SchemaSettings` object
#' @export
#'
#' @family Object functions 
SchemaSettings <- function(encoding = NULL, schema = NULL) {
  if(inherits(schema, "Schema")) {
    schema <- schema$name
  }
  
  structure(list(encoding = encoding, schema = schema), class = c(
    "SchemaSettings",
    "list"
  ))
}

#' Builds a Snapshot Object
#'
#' @param topic `character`, `Topic` The name of the topic from which this snapshot is
#'   retaining messages
#' @param expire_time `character` The snapshot is guaranteed to exist up until this time
#' @param name `character` The name of the snapshot
#' @param labels `list` Key-value pairs for topic labels
#'
#' @return `Snapshot` object
#' @export
#'
#' @family Object functions
Snapshot <- function(topic = NULL, expire_time = NULL, name = NULL,
                     labels = NULL) {
  topic <- as.topic_name(topic)
  structure(list(
    topic = topic, expireTime = expire_time, name = name, labels = labels
  ), class = c("Snapshot", "list"))
}

#' Builds a DeadLetterPolicy Object
#'
#' @param max_delivery_attempts `numeric` The maximum number of delivery attempts for any message
#' @param dead_letter_topic `character`, `Topic` The name of the topic to which dead letter
#'   messages should be published
#'
#' @return `DeadLetterPolicy` object
#'
#' @family Object functions
#' @export
DeadLetterPolicy <- function(max_delivery_attempts = NULL, dead_letter_topic = NULL) {
  structure(list(maxDeliveryAttempts = max_delivery_attempts, deadLetterTopic = dead_letter_topic),
    class = c("DeadLetterPolicy", "list")
  )
}

#' Builds a ExpirationPolicy Object
#'
#' @details
#' Autogenerated via \code{\link[googleAuthR]{gar_create_api_objects}}
#' A policy that specifies the conditions for resource expiration (i.e., automatic resource deletion).
#'
#' @param ttl `numeric` Specifies the 'time-to-live' duration (in seconds, can be float) for an 
#' associated resource
#'
#' @return `ExpirationPolicy` object
#' @export
#'
#' @family Object functions
ExpirationPolicy <- function(ttl = NULL) {
  ttl <- paste0(ttl, "s")
  structure(list(ttl = ttl), class = c("gar_ExpirationPolicy", "list"))
}

# #' Policy Object
# #'
# #' @details
# #' Autogenerated via \code{\link[googleAuthR]{gar_create_api_objects}}
# #' An Identity and Access Management (IAM) policy, which specifies access controls for Google Cloud resources. A `Policy` is a collection of `bindings`. A `binding` binds one or more `members`, or principals, to a single `role`. Principals can be user accounts, service accounts, Google groups, and domains (such as G Suite). A `role` is a named list of permissions; each `role` can be an IAM predefined role or a user-created custom role. For some types of Google Cloud resources, a `binding` can also specify a `condition`, which is a logical expression that allows access to a resource only if the expression evaluates to `true`. A condition can add constraints based on attributes of the request, the resource, or both. To learn which resources support conditions in their IAM policies, see the [IAM documentation](https://cloud.google.com/iam/help/conditions/resource-policies). **JSON example:** { 'bindings': [ { 'role': 'roles/resourcemanager.organizationAdmin', 'members': [ 'user:mike@example.com', 'group:admins@example.com', 'domain:google.com', 'serviceAccount:my-project-id@appspot.gserviceaccount.com' ] }, { 'role': 'roles/resourcemanager.organizationViewer', 'members': [ 'user:eve@example.com' ], 'condition': { 'title': 'expirable access', 'description': 'Does not grant access after Sep 2020', 'expression': 'request.time < timestamp('2020-10-01T00:00:00.000Z')', } } ], 'etag': 'BwWWja0YfJA=', 'version': 3 } **YAML example:** bindings: - members: - user:mike@example.com - group:admins@example.com - domain:google.com - serviceAccount:my-project-id@appspot.gserviceaccount.com role: roles/resourcemanager.organizationAdmin - members: - user:eve@example.com role: roles/resourcemanager.organizationViewer condition: title: expirable access description: Does not grant access after Sep 2020 expression: request.time < timestamp('2020-10-01T00:00:00.000Z') etag: BwWWja0YfJA= version: 3 For a description of IAM and its features, see the [IAM documentation](https://cloud.google.com/iam/docs/).
# #'
# #' @param bindings Associates a list of `members`, or principals, with a `role`
# #' @param version Specifies the format of the policy
# #' @param etag `etag` is used for optimistic concurrency control as a way to help prevent simultaneous updates of a policy from overwriting each other
# #'
# #' @return Policy object
# #'
# #' @family Policy functions
# #' @export
# Policy <- function(bindings = NULL, version = NULL, etag = NULL) {
#   structure(list(bindings = bindings, version = version, etag = etag), class = c(
#     "gar_Policy",
#     "list"
#   ))
# }


# #' Expr Object
# #'
# #' @param title Optional
# #' @param location Optional
# #' @param expression Textual representation of an expression in Common Expression Language syntax
# #' @param description Optional
# #'
# #' @return Expr object
# #'
# #' @family Expr functions
# #' @export
# 
# 
# Expr <- function(title = NULL, location = NULL, expression = NULL, description = NULL) {
  # structure(list(title = title, location = location, expression = expression, description = description),
    # class = c("gar_Expr", "list")
  # )
# }
