% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/grain.R
\name{grain}
\alias{grain}
\alias{grain,goc-method}
\title{Extract a grain of connectivity (GOC) tessellation at a given scale}
\usage{
grain(x, ...)

\S4method{grain}{goc}(x, whichThresh, ...)
}
\arguments{
\item{x}{A \code{goc} object created by \code{\link{GOC}}.}

\item{...}{Additional arguments (not used).}

\item{whichThresh}{Integer giving the grain threshold to extract.
This is the index of the threshold extracted by \code{\link{GOC}}.}
}
\value{
A list object containing the following elements:

\describe{
  \item{\code{summary}}{gives the properties of the specified scale/grain \code{whichThresh}
  of the GOC model;}

  \item{\code{voronoi}}{a \code{RasterLayer} giving the Voronoi tessellation the
  specified scale/grain \code{whichThresh} of the GOC model;}

  \item{\code{centroids}}{a \code{SpatialPoints} objects giving the centroids
  of the polygons in the Voronoi tessellation at the specified scale/grain \code{whichThresh};}

  \item{\code{th}}{a \code{igraph} object giving the graph describing the relationship
  among the polygons at the specified scale/grain \code{whichThresh}}
}
}
\description{
Extract a grain (i.e. a scaled version of a Voronoi tessellation) from a GOC model.
}
\examples{
## Load raster landscape
tiny <- raster::raster(system.file("extdata/tiny.asc", package = "grainscape"))

## Create a resistance surface from a raster using an is-becomes reclassification
tinyCost <- raster::reclassify(tiny, rcl = cbind(c(1, 2, 3, 4), c(1, 5, 10, 12)))

## Produce a patch-based MPG where patches are resistance features=1
tinyPatchMPG <- MPG(cost = tinyCost, patch = tinyCost == 1)

## Extract a representative subset of 5 grains of connectivity
tinyPatchGOC <- GOC(tinyPatchMPG, nThresh = 5)

## Very quick visualization at the finest scale/grain/threshold
tinyPatchGOCgrain <- grain(tinyPatchGOC, whichThresh = 1)
if (interactive())
  plot(tinyPatchGOCgrain, col = topo.colors(10))

## Visualize the model at the finest scale/grain/threshold
## Manual control of plotting
if (interactive()) {
  plot(grain(tinyPatchGOC, whichThresh = 1)@voronoi,
       col = sample(rainbow(100)), legend = FALSE, main = "Threshold 1")
}
}
\references{
Fall, A., M.-J. Fortin, M. Manseau, D. O'Brien. (2007) Spatial graphs:
Principles and applications for habitat connectivity. Ecosystems 10:448:461.

Galpern, P., M. Manseau. (2013a) Finding the functional grain: comparing methods
for scaling resistance surfaces. Landscape Ecology 28:1269-1291.

Galpern, P., M. Manseau. (2013b) Modelling the influence of landscape connectivity
on animal distribution: a functional grain approach. Ecography 36:1004-1016.

Galpern, P., M. Manseau, A. Fall. (2011) Patch-based graphs of landscape connectivity:
a guide to construction, analysis, and application for conservation.
Biological Conservation 144:44-55.

Galpern, P., M. Manseau, P.J. Wilson. (2012) Grains of connectivity: analysis
at multiple spatial scales in landscape genetics. Molecular Ecology 21:3996-4009.
}
\seealso{
\code{\link{GOC}}
}
\author{
Paul Galpern and Alex Chubaty
}
