% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/basecor.R
\docType{class}
\name{basecor-class}
\alias{basecor-class}
\alias{basecor}
\alias{basecor.numeric}
\alias{basecor.matrix}
\alias{print.basecor}
\title{Information for a base model for correlation matrices}
\usage{
basecor(base, p, parametrization = "cpc", itheta)

\method{basecor}{numeric}(base, p, parametrization = "cpc", itheta)

\method{basecor}{matrix}(base, p, parametrization = "cpc", itheta)

\method{print}{basecor}(x, ...)
}
\arguments{
\item{base}{numeric vector/matrix used to define the base
correlation matrix. If numeric vector with length 'm',
'm' should be 'p(p-1)/2' in the dense model case and
'length(itheta)' in the sparse model case.}

\item{p}{integer with the dimension,
the number of rows/columns of the correlation matrix.}

\item{parametrization}{character to specify the
parametrization used. The available ones are
"cpc" (or "CPC") or "sap" (or "SAP").
See Details. The default is "cpc".}

\item{itheta}{integer vector to specify the (vectorized) position
where 'theta' will be placed in the (initial, before fill-in)
Cholesky (lower triangle) factor. Default is missing and assumes
the dense case for when \code{itheta = which(lower.tri(...))}.}

\item{x}{a basecor object.}

\item{...}{further arguments passed on.}
}
\value{
a \code{basecor} object
}
\description{
The \code{basecor} class contain a correlation matrix \code{base},
the parameter vector \code{theta}, that generates
or is generated by \code{base}, the dimention \code{p},
the index \code{itheta} for \code{theta} in the (lower) Cholesky,
and the Hessian around it \code{I0}, see details.
}
\details{
For 'parametrization' = "CPC" or 'parametrization' = "cpc":
The Canonical Partial Correlation - CPC parametrization,
Lewandowski, Kurowicka, and Joe (2009), compute
\eqn{r[k]} = tanh(\eqn{\theta[k]}), for \eqn{k=1,...,m},
and the two \eqn{p\times p} matrices
\deqn{A = \left[
\begin{array}{ccccc}
  1 & & & & \\
  r_1 & 1 & & & \\
  r_2 & r_p & 1 & & \\
  \vdots & \vdots & \ddots & \ddots & \\
  r_{p-1} & r_{2p-3} & \ldots & r_m & 1
\end{array} \right]
\textrm{ and } B = \left[
\begin{array}{ccccc}
  1 & & & & \\
  \sqrt{1-r_1^2} & 1 & & & \\
  \sqrt{1-r_2^2} & \sqrt{1-r_p^2} & 1 & & \\
  \vdots & \vdots & \ddots & \ddots & \\
  \sqrt{1-r_{p-1}^2} & \sqrt{r_{2p-3}^2} & \ldots & \sqrt{1-r_m^2} & 1
\end{array} \right] }

The matrices \eqn{A} and \eqn{B} are then used
to build the Cholesky factor of the correlation matrix,
given as
\deqn{L = \left[
\begin{array}{ccccc}
  1 & 0 & 0 & \ldots & 0\\
  A_{2,1} & B_{2,1} & 0 & \ldots & 0\\
  A_{3,1} & A_{3,2}B_{3,1} & B_{3,1}B_{3,2} & & \vdots \\
  \vdots & \vdots & \ddots & \ddots & 0\\
  A_{p,1} & A_{p,2}B_{p,1} & \ldots &
  A_{p,p-1}\prod_{k=1}^{p-1}B_{p,k} & \prod_{k=1}^{p-1}B_{p,k}
\end{array} \right]}
Note: The determinant of the correlation matriz is
\deqn{\prod_{i=2}^p\prod_{j=1}^{i-1}B_{i,j} = \prod_{i=2}^pL_{i,i}}
For 'parametrization' = "SAP" or 'parametrization' = "sap":
The Standard Angles Parametrization - SAP, as described in
Rapisarda, Brigo and Mercurio (2007), compute
\eqn{x[k] = \pi/(1+\exp(-\theta[k]))}, for \eqn{k=1,...,m},
and the two \eqn{p\times p} matrices
\deqn{A = \left[
\begin{array}{ccccc}
  1 & & & & \\
  \cos(x_1) & 1 & & & \\
  \cos(x_2) & \cos(x_p) & 1 & & \\
  \vdots & \vdots & \ddots & \ddots & \\
  \cos(x_{p-1}) & \cos(x_{2p-3}) & \ldots & \cos(x_m) & 1
\end{array} \right] \textrm{ and } B = \left[
\begin{array}{ccccc}
  1 & & & & \\
  \sin(x_1) & 1 & & & \\
  \sin(x_2) & \sin(x_p) & 1 & & \\
  \vdots & \vdots & \ddots & \ddots & \\
  \sin(x_{p-1}) & \sin(x_{2p-3}) & \ldots & \sin(x_m) & 1
\end{array} \right]}

The decomposition of the Hessian matrix around the base model,
\code{I0}, formally \eqn{\mathbf{I}(\theta_0)}, is numerically computed.
This element has the following attributes:
'h.5' as \eqn{\mathbf{I}^{1/2}(\theta_0)}, and
'hneg.5' as \eqn{\mathbf{I}^{-1/2}(\theta_0)}.
}
\section{Functions}{
\itemize{
\item \code{basecor()}: Build a \code{basecor} object.

\item \code{basecor(numeric)}: Build a \code{basecor} from the parameter vector.

\item \code{basecor(matrix)}: Build a \code{basecor} from a correlation matrix.

\item \code{print(basecor)}: Print method for 'basecor'

}}
\examples{
library(graphpcor)

## A correlation matrix
c0 <- matrix(c(1,.8,-.625, 0.8,1,-.5, -0.625,-.5,1), 3)

## build the 'basecor'
pc.c0 <- basecor(c0) ## base as matrix
pc.c0

## elements
pc.c0$base
pc.c0$theta
pc.c0$I0

## from 'theta' 
th0 <- pc.c0$theta
pc.th0 <- basecor(th0) ## base as vector
pc.th0

## numerically the same
all.equal(c0, pc.th0$base)

## from a numeric vector (theta)
th2 <- c(-1, -0.5)
b2 <- basecor(th2, p = 3, itheta = c(2,3))
b2

## from the correlation matrix
b2c <- basecor(b2$base, itheta = c(2,3))

all.equal(th2, b2c$theta, tol = 1e-4)

## Hessian around the base (and its decomposition, etc.)
b2$I0
}
\references{
Rapisarda, Brigo and Mercurio (2007).
Parameterizing correlations: a geometric interpretation.
IMA Journal of Management Mathematics (2007) 18, 55-73.
<doi 10.1093/imaman/dpl010>

Lewandowski, Kurowicka and Joe (2009)
Generating Random Correlation Matrices Based
on Vines and Extended Onion Method.
Journal of Multivariate Analysis 100: 1989–2001.
<doi: 10.1016/j.jmva.2009.04.008>

Simpson, et. al. (2017)
Penalising Model Component Complexity:
A Principled, Practical Approach to Constructing Priors.
Statist. Sci. 32(1): 1-28 (February 2017).
<doi: 10.1214/16-STS576>
}
