% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/text_transform.R
\name{text_transform}
\alias{text_transform}
\title{Perform text transformations with a custom function}
\usage{
text_transform(data, fn, locations = cells_body())
}
\arguments{
\item{data}{\emph{The gt table data object}

\verb{obj:<gt_tbl>} // \strong{required}

This is the \strong{gt} table object that is commonly created through use of the
\code{\link[=gt]{gt()}} function.}

\item{fn}{\emph{Function for text transformation}

\verb{<function>} // \strong{required}

The function to use for text transformation. It should include \code{x} as an
argument and return a character vector of the same length as the input \code{x}.}

\item{locations}{\emph{Locations to target}

\code{\link[=location-helper]{<locations expressions>}} // \emph{default:} \code{cells_body()}

The cell or set of cells to be associated with the text transformation.
Only \code{\link[=cells_column_spanners]{cells_column_spanners()}}, \code{\link[=cells_column_labels]{cells_column_labels()}},
\code{\link[=cells_row_groups]{cells_row_groups()}}, \code{\link[=cells_stub]{cells_stub()}}, and \code{\link[=cells_body]{cells_body()}} can
be used here. We can enclose several of these calls within a \code{list()} if we
wish to make the transformation happen at different locations.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
Text transforming in \strong{gt} is the act of modifying formatted strings in
targeted cells. \code{text_transform()} provides the most flexibility
of all the \verb{text_*()} functions in their family of functions. With it, you
target the cells to undergo modification in the \code{locations} argument while
also supplying a function to the \code{fn} argument. The function given to \code{fn}
should ideally at the very least take \code{x} as an input (it stands for the
character vector that is essentially the targeted cells) and return a
character vector of the same length as the input. Using the construction
\code{function(x) { .. }} for the function is recommended.
}
\section{Examples}{


Use a subset of the \code{\link{sp500}} dataset to create a \strong{gt} table. Transform the
text in the \code{date} column using a function supplied to \code{text_transform()}
(via the \code{fn} argument). Note that the \code{x} in the \verb{fn = function (x)} part
consists entirely of ISO 8601 date strings (which are acceptable as input to
\code{\link[=vec_fmt_date]{vec_fmt_date()}} and \code{\link[=vec_fmt_datetime]{vec_fmt_datetime()}}).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{sp500 |>
  dplyr::slice_head(n = 10) |>
  dplyr::select(date, open, close) |>
  dplyr::arrange(-dplyr::row_number()) |>
  gt() |>
  fmt_currency() |>
  text_transform(
    fn = function(x) \{
      paste0(
        "<strong>",
        vec_fmt_date(x, date_style = "m_day_year"),
        "</strong>",
        "&mdash;W",
        vec_fmt_datetime(x, format = "w")
      )
    \},
    locations = cells_body(columns = date)
  ) |>
  cols_label(
    date = "Date and Week",
    open = "Opening Price",
    close = "Closing Price"
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_text_transform_1.png" alt="This image of a table was generated from the first code example in the `text_transform()` help file." style="width:100\%;">
}}

Let's use a summarized version of the \code{\link{gtcars}} dataset to create a \strong{gt}
table. First, the numeric values in the \code{n} column are formatted as
spelled-out numbers with \code{\link[=fmt_spelled_num]{fmt_spelled_num()}}. The output values are indeed
spelled out but exclusively with lowercase letters. We actually want these
words to begin with a capital letter and end with a period. To make this
possible, \code{text_transform()} will be used since it can modify
already-formatted text. Through the \code{fn} argument, we provide a custom
function that uses R's \code{toTitleCase()} operating on \code{x} (the numbers-as-text
strings) within \code{paste0()} so that a period can be properly placed.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{gtcars |>
  dplyr::filter(ctry_origin \%in\% c("Germany", "Italy", "Japan")) |>
  dplyr::count(mfr, ctry_origin, sort = TRUE) |>
  dplyr::arrange(ctry_origin) |>
  gt(rowname_col = "mfr", groupname_col = "ctry_origin") |>
  cols_label(n = "No. of Entries") |>
  tab_stub_indent(rows = everything(), indent = 2) |>
  cols_align(align = "center", columns = n) |>
  fmt_spelled_num() |>
  text_transform(
    fn = function(x) \{
      paste0(tools::toTitleCase(x), ".")
    \},
    locations = cells_body(columns = n)
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_text_transform_2.png" alt="This image of a table was generated from the second code example in the `text_transform()` help file." style="width:100\%;">
}}

There may be occasions where you'd want to remove all text. Here in this
example based on the \code{\link{pizzaplace}} dataset, we generate a \strong{gt} table that
summarizes an entire year of data by colorizing the daily sales revenue.
Individual cell values are not needed here (since the encoding by color
suffices), so, \code{text_transform()} is used to turn every value to an empty
string: \code{""}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{pizzaplace |>
  dplyr::group_by(date) |>
  dplyr::summarize(rev = sum(price)) |>
  dplyr::ungroup() |>
  dplyr::mutate(
    month = lubridate::month(date, label = TRUE),
    day_num = lubridate::mday(date)
  ) |>
  dplyr::select(-date) |>
  tidyr::pivot_wider(names_from = month, values_from = rev) |>
  gt(rowname_col = "day_num") |>
  data_color(
    method = "numeric",
    palette = "wesanderson::Zissou1",
    na_color = "white"
  ) |>
  text_transform(
    fn = function(x) "",
    locations = cells_body()
  ) |>
  opt_table_lines(extent = "none") |>
  opt_all_caps() |>
  cols_width(everything() ~ px(35)) |>
  cols_align(align = "center")
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_text_transform_3.png" alt="This image of a table was generated from the third code example in the `text_transform()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

4-4
}

\section{Function Introduced}{

\code{v0.2.0.5} (March 31, 2020)
}

\seealso{
Other text transforming functions: 
\code{\link{text_case_match}()},
\code{\link{text_case_when}()},
\code{\link{text_replace}()}
}
\concept{text transforming functions}
