% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/format_vec.R
\name{vec_fmt_date}
\alias{vec_fmt_date}
\title{Format a vector as date values}
\usage{
vec_fmt_date(
  x,
  date_style = "iso",
  pattern = "{x}",
  locale = NULL,
  output = c("auto", "plain", "html", "latex", "rtf", "word")
)
}
\arguments{
\item{x}{A numeric vector.}

\item{date_style}{The date style to use. By default this is \code{"iso"} which
corresponds to ISO 8601 date formatting. The other date styles can be
viewed using \code{\link[=info_date_style]{info_date_style()}}.}

\item{pattern}{A formatting pattern that allows for decoration of the
formatted value. The value itself is represented by \code{{x}} and all other
characters are taken to be string literals.}

\item{locale}{An optional locale identifier that can be used for formatting
the value according the locale's rules. Examples include \code{"en"} for English
(United States) and \code{"fr"} for French (France). The use of a locale ID will
override any locale-specific values provided. We can use the
\code{\link[=info_locales]{info_locales()}} function as a useful reference for all of the locales that
are supported.}

\item{output}{The output style of the resulting character vector. This can
either be \code{"auto"} (the default), \code{"plain"}, \code{"html"}, \code{"latex"}, \code{"rtf"},
or \code{"word"}. In \strong{knitr} rendering (i.e., Quarto or R Markdown), the
\code{"auto"} option will choose the correct \code{output} value}
}
\value{
A character vector.
}
\description{
Format vector values to date values using one of 41 preset date styles. Input
can be in the form of \code{POSIXt} (i.e., datetimes), the \code{Date} type, or
\code{character} (must be in the ISO 8601 form of \verb{YYYY-MM-DD HH:MM:SS} or
\code{YYYY-MM-DD}).
}
\section{Formatting with the \code{date_style} argument}{


We need to supply a preset date style to the \code{date_style} argument. The date
styles are numerous and can handle localization to any supported locale. A
large segment of date styles are termed flexible date formats and this means
that their output will adapt to any \code{locale} provided. That feature makes the
flexible date formats a better option for locales other than \code{"en"} (the
default locale).

The following table provides a listing of all date styles and their output
values (corresponding to an input date of \code{2000-02-29}).\tabular{llll}{
    \tab Date Style \tab Output \tab Notes \cr
   1 \tab \code{"iso"} \tab \code{"2000-02-29"} \tab ISO 8601 \cr
   2 \tab \code{"wday_month_day_year"} \tab \code{"Tuesday, February 29, 2000"} \tab  \cr
   3 \tab \code{"wd_m_day_year"} \tab \code{"Tue, Feb 29, 2000"} \tab  \cr
   4 \tab \code{"wday_day_month_year"} \tab \code{"Tuesday 29 February 2000"} \tab  \cr
   5 \tab \code{"month_day_year"} \tab \code{"February 29, 2000"} \tab  \cr
   6 \tab \code{"m_day_year"} \tab \code{"Feb 29, 2000"} \tab  \cr
   7 \tab \code{"day_m_year"} \tab \code{"29 Feb 2000"} \tab  \cr
   8 \tab \code{"day_month_year"} \tab \code{"29 February 2000"} \tab  \cr
   9 \tab \code{"day_month"} \tab \code{"29 February"} \tab  \cr
   10 \tab \code{"day_m"} \tab \code{"29 Feb"} \tab  \cr
   11 \tab \code{"year"} \tab \code{"2000"} \tab  \cr
   12 \tab \code{"month"} \tab \code{"February"} \tab  \cr
   13 \tab \code{"day"} \tab \code{"29"} \tab  \cr
   14 \tab \code{"year.mn.day"} \tab \code{"2000/02/29"} \tab  \cr
   15 \tab \code{"y.mn.day"} \tab \code{"00/02/29"} \tab  \cr
   16 \tab \code{"year_week"} \tab \code{"2000-W09"} \tab  \cr
   17 \tab \code{"year_quarter"} \tab \code{"2000-Q1"} \tab  \cr
   18 \tab \code{"yMd"} \tab \code{"2/29/2000"} \tab flexible \cr
   19 \tab \code{"yMEd"} \tab \code{"Tue, 2/29/2000"} \tab flexible \cr
   20 \tab \code{"yMMM"} \tab \code{"Feb 2000"} \tab flexible \cr
   21 \tab \code{"yMMMM"} \tab \code{"February 2000"} \tab flexible \cr
   22 \tab \code{"yMMMd"} \tab \code{"Feb 29, 2000"} \tab flexible \cr
   23 \tab \code{"yMMMEd"} \tab \code{"Tue, Feb 29, 2000"} \tab flexible \cr
   24 \tab \code{"GyMd"} \tab \code{"2/29/2000 A"} \tab flexible \cr
   25 \tab \code{"GyMMMd"} \tab \code{"Feb 29, 2000 AD"} \tab flexible \cr
   26 \tab \code{"GyMMMEd"} \tab \code{"Tue, Feb 29, 2000 AD"} \tab flexible \cr
   27 \tab \code{"yM"} \tab \code{"2/2000"} \tab flexible \cr
   28 \tab \code{"Md"} \tab \code{"2/29"} \tab flexible \cr
   29 \tab \code{"MEd"} \tab \code{"Tue, 2/29"} \tab flexible \cr
   30 \tab \code{"MMMd"} \tab \code{"Feb 29"} \tab flexible \cr
   31 \tab \code{"MMMEd"} \tab \code{"Tue, Feb 29"} \tab flexible \cr
   32 \tab \code{"MMMMd"} \tab \code{"February 29"} \tab flexible \cr
   33 \tab \code{"GyMMM"} \tab \code{"Feb 2000 AD"} \tab flexible \cr
   34 \tab \code{"yQQQ"} \tab \code{"Q1 2000"} \tab flexible \cr
   35 \tab \code{"yQQQQ"} \tab \code{"1st quarter 2000"} \tab flexible \cr
   36 \tab \code{"Gy"} \tab \code{"2000 AD"} \tab flexible \cr
   37 \tab \code{"y"} \tab \code{"2000"} \tab flexible \cr
   38 \tab \code{"M"} \tab \code{"2"} \tab flexible \cr
   39 \tab \code{"MMM"} \tab \code{"Feb"} \tab flexible \cr
   40 \tab \code{"d"} \tab \code{"29"} \tab flexible \cr
   41 \tab \code{"Ed"} \tab \code{"29 Tue"} \tab flexible \cr
}


We can use the \code{\link[=info_date_style]{info_date_style()}} function within the console to view a
similar table of date styles with example output.
}

\section{Examples}{


Let's create a character vector of dates in the ISO-8601 format for the next
few examples:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{str_vals <- c("2022-06-13", "2019-01-25", "2015-03-23", NA)
}\if{html}{\out{</div>}}

Using \code{vec_fmt_date()} (here with the \code{"wday_month_day_year"} date style)
will result in a character vector of formatted dates. Any \code{NA} values remain
as \code{NA} values. The rendering context will be autodetected unless specified
in the \code{output} argument (here, it is of the \code{"plain"} output type).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_date(str_vals, date_style = "wday_month_day_year")
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "Monday, June 13, 2022" "Friday, January 25, 2019"
#> [3] "Monday, March 23, 2015" NA
}\if{html}{\out{</div>}}

We can choose from any of 41 different date formatting styles. Many of these
styles are flexible, meaning that the structure of the format will adapt
to different locales. Let's use the \code{"yMMMEd"} date style to demonstrate this
(first in the default locale of \code{"en"}):

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_date(str_vals, date_style = "yMMMEd")
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "Mon, Jun 13, 2022" "Fri, Jan 25, 2019" "Mon, Mar 23, 2015" NA
}\if{html}{\out{</div>}}

Let's perform the same type of formatting in the French (\code{"fr"}) locale:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_date(str_vals, date_style = "yMMMEd", locale = "fr")
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "lun. 13 juin 2022" "ven. 25 janv. 2019" "lun. 23 mars 2015" NA
}\if{html}{\out{</div>}}

We can always use \code{\link[=info_date_style]{info_date_style()}} to call up an info table that serves as
a handy reference to all of the \code{date_style} options.

As a last example, one can wrap the date values in a pattern with the
\code{pattern} argument. Note here that \code{NA} values won't have the pattern
applied.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_date(str_vals, pattern = "Date: \{x\}")
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "Date: 2022-06-13" "Date: 2019-01-25" "Date: 2015-03-23" NA
}\if{html}{\out{</div>}}
}

\section{Function ID}{

15-13
}

\section{Function Introduced}{

\code{v0.7.0} (Aug 25, 2022)
}

\seealso{
The variant function intended for formatting \strong{gt} table data:
\code{\link[=fmt_date]{fmt_date()}}.

Other vector formatting functions: 
\code{\link{vec_fmt_bytes}()},
\code{\link{vec_fmt_currency}()},
\code{\link{vec_fmt_datetime}()},
\code{\link{vec_fmt_duration}()},
\code{\link{vec_fmt_engineering}()},
\code{\link{vec_fmt_fraction}()},
\code{\link{vec_fmt_index}()},
\code{\link{vec_fmt_integer}()},
\code{\link{vec_fmt_markdown}()},
\code{\link{vec_fmt_number}()},
\code{\link{vec_fmt_partsper}()},
\code{\link{vec_fmt_percent}()},
\code{\link{vec_fmt_roman}()},
\code{\link{vec_fmt_scientific}()},
\code{\link{vec_fmt_spelled_num}()},
\code{\link{vec_fmt_time}()}
}
\concept{vector formatting functions}
