/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5FSmodule.h" 

#include "H5private.h"   
#include "H5ACprivate.h" 
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5FSpkg.h"     
#include "H5MFprivate.h" 
#include "H5MMprivate.h" 
#include "H5SLprivate.h" 

static herr_t H5FS__sinfo_free_sect_cb(void *item, void *key, void *op_data);
static herr_t H5FS__sinfo_free_node_cb(void *item, void *key, void *op_data);

bool H5_PKG_INIT_VAR = false;

H5FL_SEQ_DEFINE(H5FS_section_class_t);

H5FL_DEFINE(H5FS_t);

H5FS_t *
H5FS_create(H5F_t *f, haddr_t *fs_addr, const H5FS_create_t *fs_create, uint16_t nclasses,
            const H5FS_section_class_t *classes[], void *cls_init_udata, hsize_t alignment, hsize_t threshold)
{
    H5FS_t *fspace    = NULL; 
    H5FS_t *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)
#ifdef H5FS_DEBUG
    Rfprintf(Rstderr, "%s: Creating free space manager, nclasses = %llu\n", __func__, (unsigned long long)nclasses);
#endif 

    
    assert(fs_create->shrink_percent);
    assert(fs_create->shrink_percent < fs_create->expand_percent);
    assert(fs_create->max_sect_size);
    assert(nclasses == 0 || classes);

    
    if (NULL == (fspace = H5FS__new(f, nclasses, classes, cls_init_udata)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for free space free list");

    
    fspace->client         = fs_create->client;
    fspace->shrink_percent = fs_create->shrink_percent;
    fspace->expand_percent = fs_create->expand_percent;
    fspace->max_sect_addr  = fs_create->max_sect_addr;
    fspace->max_sect_size  = fs_create->max_sect_size;
    fspace->swmr_write     = (H5F_INTENT(f) & H5F_ACC_SWMR_WRITE) > 0;

    fspace->alignment   = alignment;
    fspace->align_thres = threshold;

    
    if (fs_addr) {
        
        if (HADDR_UNDEF == (fspace->addr = H5MF_alloc(f, H5FD_MEM_FSPACE_HDR, (hsize_t)fspace->hdr_size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "file allocation failed for free space header");

        
        if (H5AC_insert_entry(f, H5AC_FSPACE_HDR, fspace->addr, fspace, H5AC__PIN_ENTRY_FLAG) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTINIT, NULL, "can't add free space header to cache");

        
        *fs_addr = fspace->addr;
    } 

    
    fspace->rc = 1;

    
    ret_value = fspace;
#ifdef H5FS_DEBUG
    Rfprintf(Rstderr, "%s: fspace = %p, fspace->addr = %" PRIuHADDR "\n", __func__, (void *)fspace,
            fspace->addr);
#endif 

done:
    if (!ret_value && fspace)
        if (H5FS__hdr_dest(fspace) < 0)
            HDONE_ERROR(H5E_FSPACE, H5E_CANTFREE, NULL, "unable to destroy free space header");

#ifdef H5FS_DEBUG
    Rfprintf(Rstderr, "%s: Leaving, ret_value = %p\n", __func__, (void *)ret_value);
#endif 
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5FS_t *
H5FS_open(H5F_t *f, haddr_t fs_addr, uint16_t nclasses, const H5FS_section_class_t *classes[],
          void *cls_init_udata, hsize_t alignment, hsize_t threshold)
{
    H5FS_t             *fspace = NULL;    
    H5FS_hdr_cache_ud_t cache_udata;      
    H5FS_t             *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)
#ifdef H5FS_DEBUG
    Rfprintf(Rstderr, "%s: Opening free space manager, fs_addr = %" PRIuHADDR ", nclasses = %llu\n", __func__,
            fs_addr, (unsigned long long)nclasses);
#endif 

    
    assert(H5_addr_defined(fs_addr));
    assert(nclasses);
    assert(classes);

    
    cache_udata.f              = f;
    cache_udata.nclasses       = nclasses;
    cache_udata.classes        = classes;
    cache_udata.cls_init_udata = cls_init_udata;
    cache_udata.addr           = fs_addr;

    
    if (NULL ==
        (fspace = (H5FS_t *)H5AC_protect(f, H5AC_FSPACE_HDR, fs_addr, &cache_udata, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTPROTECT, NULL, "unable to load free space header");
#ifdef H5FS_DEBUG
    Rfprintf(Rstderr, "%s: fspace->sect_addr = %" PRIuHADDR "\n", __func__, fspace->sect_addr);
    Rfprintf(Rstderr, "%s: fspace->sect_size = %" PRIuHSIZE "\n", __func__, fspace->sect_size);
    Rfprintf(Rstderr, "%s: fspace->alloc_sect_size = %" PRIuHSIZE "\n", __func__, fspace->alloc_sect_size);
    Rfprintf(Rstderr, "%s: fspace->sinfo = %p\n", __func__, (void *)fspace->sinfo);
    Rfprintf(Rstderr, "%s: fspace->rc = %u\n", __func__, fspace->rc);
#endif 

    
    assert(fspace->rc <= 1);
    if (H5FS__incr(fspace) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTINC, NULL, "unable to increment ref. count on free space header");

    fspace->alignment   = alignment;
    fspace->align_thres = threshold;

    
    if (H5AC_unprotect(f, H5AC_FSPACE_HDR, fs_addr, fspace, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTUNPROTECT, NULL, "unable to release free space header");

    
    ret_value = fspace;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FS_delete(H5F_t *f, haddr_t fs_addr)
{
    H5FS_t             *fspace = NULL;       
    H5FS_hdr_cache_ud_t cache_udata;         
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)
#ifdef H5FS_DEBUG
    Rfprintf(Rstderr, "%s: Deleting free space manager, fs_addr = %" PRIuHADDR "\n", __func__, fs_addr);
#endif 

    
    assert(f);
    assert(H5_addr_defined(fs_addr));

    
    
    cache_udata.f              = f;
    cache_udata.nclasses       = 0;
    cache_udata.classes        = NULL;
    cache_udata.cls_init_udata = NULL;
    cache_udata.addr           = fs_addr;

#ifdef H5FS_DEBUG
    {
        unsigned fspace_status = 0; 

        
        assert(H5_addr_defined(fs_addr));

        
        if (H5AC_get_entry_status(f, fs_addr, &fspace_status) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL,
                        "unable to check metadata cache status for free space section info");

        Rfprintf(Rstderr, "%s: fspace_status = %0x: ", __func__, fspace_status);
        if (fspace_status) {
            bool printed = false;

            if (fspace_status & H5AC_ES__IN_CACHE) {
                Rfprintf(Rstderr, "H5AC_ES__IN_CACHE");
                printed = true;
            } 
            if (fspace_status & H5AC_ES__IS_DIRTY) {
                Rfprintf(Rstderr, "%sH5AC_ES__IS_DIRTY", (printed ? " | " : ""));
                printed = true;
            } 
            if (fspace_status & H5AC_ES__IS_PROTECTED) {
                Rfprintf(Rstderr, "%sH5AC_ES__IS_PROTECTED", (printed ? " | " : ""));
                printed = true;
            } 
            if (fspace_status & H5AC_ES__IS_PINNED) {
                Rfprintf(Rstderr, "%sH5AC_ES__IS_PINNED", (printed ? " | " : ""));
                printed = true;
            } 
            if (fspace_status & H5AC_ES__IS_FLUSH_DEP_PARENT) {
                Rfprintf(Rstderr, "%sH5AC_ES__IS_FLUSH_DEP_PARENT", (printed ? " | " : ""));
                printed = true;
            } 
            if (fspace_status & H5AC_ES__IS_FLUSH_DEP_CHILD) {
                Rfprintf(Rstderr, "%sH5AC_ES__IS_FLUSH_DEP_CHILD", (printed ? " | " : ""));
                printed = true;
            } 
        }     
        Rfprintf(Rstderr, "\n");
    }
#endif 

    
    if (NULL ==
        (fspace = (H5FS_t *)H5AC_protect(f, H5AC_FSPACE_HDR, fs_addr, &cache_udata, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTPROTECT, FAIL, "unable to protect free space header");

    
    assert(fspace->sinfo == NULL);

    
#ifdef H5FS_DEBUG
    Rfprintf(Rstderr, "%s: fspace->sect_addr = %" PRIuHADDR "\n", __func__, fspace->sect_addr);
#endif 
    if (fspace->serial_sect_count > 0) {
        unsigned sinfo_status = 0; 

        
        assert(H5_addr_defined(fspace->sect_addr));
        assert(fspace->alloc_sect_size > 0);

        
        if (H5AC_get_entry_status(f, fspace->sect_addr, &sinfo_status) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL,
                        "unable to check metadata cache status for free space section info");

        
        if (sinfo_status & H5AC_ES__IN_CACHE) {
            
            assert(!(sinfo_status & H5AC_ES__IS_PINNED));
            assert(!(sinfo_status & H5AC_ES__IS_PROTECTED));

#ifdef H5FS_DEBUG
            Rfprintf(Rstderr, "%s: Expunging free space section info from cache\n", __func__);
#endif 
            
            
            {
                unsigned cache_flags = H5AC__NO_FLAGS_SET;

                
                if (!H5F_IS_TMP_ADDR(f, fspace->sect_addr))
                    cache_flags |= H5AC__FREE_FILE_SPACE_FLAG;

                if (H5AC_expunge_entry(f, H5AC_FSPACE_SINFO, fspace->sect_addr, cache_flags) < 0)
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTREMOVE, FAIL,
                                "unable to remove free space section info from cache");
            } 

#ifdef H5FS_DEBUG
            Rfprintf(Rstderr, "%s: Done expunging free space section info from cache\n", __func__);
#endif    
        } 
        else {
#ifdef H5FS_DEBUG
            Rfprintf(Rstderr, "%s: Deleting free space section info from file\n", __func__);
#endif 
            
            if (!H5F_IS_TMP_ADDR(f, fspace->sect_addr))
                if (H5MF_xfree(f, H5FD_MEM_FSPACE_SINFO, fspace->sect_addr, fspace->alloc_sect_size) < 0)
                    HGOTO_ERROR(H5E_FSPACE, H5E_CANTFREE, FAIL, "unable to release free space sections");
        } 
    }     

done:
    if (fspace && H5AC_unprotect(f, H5AC_FSPACE_HDR, fs_addr, fspace,
                                 H5AC__DELETED_FLAG | H5AC__FREE_FILE_SPACE_FLAG) < 0)
        HDONE_ERROR(H5E_FSPACE, H5E_CANTUNPROTECT, FAIL, "unable to release free space header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FS_close(H5F_t *f, H5FS_t *fspace)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(fspace);
#ifdef H5FS_DEBUG
    Rfprintf(Rstderr, "%s: Entering, fspace = %p, fspace->addr = %" PRIuHADDR ", fspace->sinfo = %p\n",
            __func__, (void *)fspace, fspace->addr, (void *)fspace->sinfo);
#endif 

    
    
    if (fspace->sinfo) {
#ifdef H5FS_DEBUG
        Rfprintf(Rstderr,
                "%s: fspace->tot_sect_count = %" PRIuHSIZE ", fspace->serial_sect_count = %" PRIuHSIZE
                ", fspace->sect_addr = %" PRIuHADDR ", fspace->rc = %u\n",
                __func__, fspace->tot_sect_count, fspace->serial_sect_count, fspace->sect_addr, fspace->rc);
        Rfprintf(Rstderr, "%s: fspace->alloc_sect_size = %" PRIuHSIZE ", fspace->sect_size = %" PRIuHSIZE "\n",
                __func__, fspace->alloc_sect_size, fspace->sect_size);
#endif 
        
        
        if (fspace->serial_sect_count > 0 && H5_addr_defined(fspace->addr)) {
#ifdef H5FS_DEBUG
            Rfprintf(Rstderr, "%s: Real sections to store in file\n", __func__);
#endif 
            if (fspace->sinfo->dirty) {
                
                if (!H5_addr_defined(fspace->sect_addr)) {
                    
                    assert(fspace->sect_size > 0);

                    
                    if (H5F_USE_TMP_SPACE(f)) {
                        if (HADDR_UNDEF == (fspace->sect_addr = H5MF_alloc_tmp(f, fspace->sect_size)))
                            HGOTO_ERROR(H5E_FSPACE, H5E_NOSPACE, FAIL,
                                        "file allocation failed for free space sections");
                    } 
                    else {
                        if (HADDR_UNDEF ==
                            (fspace->sect_addr = H5MF_alloc(f, H5FD_MEM_FSPACE_SINFO, fspace->sect_size)))
                            HGOTO_ERROR(H5E_FSPACE, H5E_NOSPACE, FAIL,
                                        "file allocation failed for free space sections");
                    } 
                    fspace->alloc_sect_size = (size_t)fspace->sect_size;

                    
                    if (H5AC_mark_entry_dirty(fspace) < 0)
                        HGOTO_ERROR(H5E_FSPACE, H5E_CANTMARKDIRTY, FAIL,
                                    "unable to mark free space header as dirty");
                } 
            }     
            else
                
                assert(H5_addr_defined(fspace->sect_addr));

            
            if (H5AC_insert_entry(f, H5AC_FSPACE_SINFO, fspace->sect_addr, fspace->sinfo,
                                  H5AC__NO_FLAGS_SET) < 0)
                HGOTO_ERROR(H5E_FSPACE, H5E_CANTINIT, FAIL, "can't add free space sections to cache");
        } 
        else {
#ifdef H5FS_DEBUG
            Rfprintf(Rstderr, "%s: NOT storing section info in file\n", __func__);
#endif 
            
            if (H5_addr_defined(fspace->sect_addr)) {
                
                
                assert(H5_addr_defined(fspace->addr));

#ifdef H5FS_DEBUG
                Rfprintf(Rstderr, "%s: Section info allocated though\n", __func__);
#endif 
                
                
                if (fspace->client == H5FS_CLIENT_FILE_ID) {
                    htri_t status; 

#ifdef H5FS_DEBUG
                    Rfprintf(Rstderr, "%s: Section info is for file free space\n", __func__);
#endif 
                    
                    if (H5F_IS_TMP_ADDR(f, fspace->sect_addr)) {
#ifdef H5FS_DEBUG
                        Rfprintf(Rstderr, "%s: Section info in temp. address space went 'go away'\n", __func__);
#endif 
                        
                        fspace->sect_addr       = HADDR_UNDEF;
                        fspace->alloc_sect_size = 0;

                        
                        if (H5AC_mark_entry_dirty(fspace) < 0)
                            HGOTO_ERROR(H5E_FSPACE, H5E_CANTMARKDIRTY, FAIL,
                                        "unable to mark free space header as dirty");
                    } 
                    else {
                        if ((status = H5MF_try_shrink(f, H5FD_MEM_FSPACE_SINFO, fspace->sect_addr,
                                                      fspace->alloc_sect_size)) < 0)
                            HGOTO_ERROR(H5E_FSPACE, H5E_CANTMERGE, FAIL,
                                        "can't check for absorbing section info");
                        else if (status == false) {
                            
#ifdef H5FS_DEBUG
                            Rfprintf(Rstderr, "%s: Section info can't 'go away', header will own it\n",
                                    __func__);
#endif                    
                        } 
                        else {
#ifdef H5FS_DEBUG
                            Rfprintf(Rstderr, "%s: Section info went 'go away'\n", __func__);
#endif 
                            
                            fspace->sect_addr       = HADDR_UNDEF;
                            fspace->alloc_sect_size = 0;

                            
                            if (H5AC_mark_entry_dirty(fspace) < 0)
                                HGOTO_ERROR(H5E_FSPACE, H5E_CANTMARKDIRTY, FAIL,
                                            "unable to mark free space header as dirty");
                        } 
                    }     
                }         
                else {
                    haddr_t old_sect_addr = fspace->sect_addr; 
                    hsize_t old_alloc_sect_size =
                        fspace->alloc_sect_size; 

#ifdef H5FS_DEBUG
                    Rfprintf(Rstderr, "%s: Section info is NOT for file free space\n", __func__);
#endif 
                    
                    fspace->sect_addr       = HADDR_UNDEF;
                    fspace->alloc_sect_size = 0;

                    
                    if (H5AC_mark_entry_dirty(fspace) < 0)
                        HGOTO_ERROR(H5E_FSPACE, H5E_CANTMARKDIRTY, FAIL,
                                    "unable to mark free space header as dirty");

                    
                    if (!H5F_IS_TMP_ADDR(f, old_sect_addr)) {
                        if (H5MF_xfree(f, H5FD_MEM_FSPACE_SINFO, old_sect_addr, old_alloc_sect_size) < 0)
                            HGOTO_ERROR(H5E_FSPACE, H5E_CANTFREE, FAIL, "unable to free free space sections");
                    } 
                }     
            }         

            
            if (H5FS__sinfo_dest(fspace->sinfo) < 0)
                HGOTO_ERROR(H5E_FSPACE, H5E_CANTCLOSEOBJ, FAIL, "unable to destroy free space section info");
        } 

        
        fspace->sinfo = NULL;
    } 
    else {
        
        if (fspace->serial_sect_count > 0)
            
            assert(H5_addr_defined(fspace->sect_addr));
    } 

    
    if (H5FS__decr(fspace) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTDEC, FAIL, "unable to decrement ref. count on free space header");

done:
#ifdef H5FS_DEBUG
    Rfprintf(Rstderr, "%s: Leaving, ret_value = %d, fspace->rc = %u\n", __func__, ret_value, fspace->rc);
#endif 
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5FS_t *
H5FS__new(const H5F_t *f, uint16_t nclasses, const H5FS_section_class_t *classes[], void *cls_init_udata)
{
    H5FS_t *fspace = NULL;    
    size_t  u;                
    H5FS_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(nclasses == 0 || (nclasses > 0 && classes));

    
    if (NULL == (fspace = H5FL_CALLOC(H5FS_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for free space free list");

    
    H5_CHECKED_ASSIGN(fspace->nclasses, uint16_t, nclasses, size_t);
    if (nclasses > 0) {
        if (NULL == (fspace->sect_cls = H5FL_SEQ_MALLOC(H5FS_section_class_t, nclasses)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL,
                        "memory allocation failed for free space section class array");

        
        for (u = 0; u < nclasses; u++) {
            
            assert(u == classes[u]->type);

            
            H5MM_memcpy(&fspace->sect_cls[u], classes[u], sizeof(H5FS_section_class_t));

            
            if (fspace->sect_cls[u].init_cls)
                if ((fspace->sect_cls[u].init_cls)(&fspace->sect_cls[u], cls_init_udata) < 0)
                    HGOTO_ERROR(H5E_RESOURCE, H5E_CANTINIT, NULL, "unable to initialize section class");

            
            if (fspace->sect_cls[u].serial_size > fspace->max_cls_serial_size)
                fspace->max_cls_serial_size = fspace->sect_cls[u].serial_size;
        } 
    }     

    
    fspace->addr      = HADDR_UNDEF;
    fspace->hdr_size  = (size_t)H5FS_HEADER_SIZE(f);
    fspace->sect_addr = HADDR_UNDEF;

    
    ret_value = fspace;

done:
    if (!ret_value)
        if (fspace) {
            
            if (fspace->sect_cls)
                fspace->sect_cls =
                    (H5FS_section_class_t *)H5FL_SEQ_FREE(H5FS_section_class_t, fspace->sect_cls);
            fspace = H5FL_FREE(H5FS_t, fspace);
        } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FS_size(const H5FS_t *fspace, hsize_t *meta_size)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(fspace);
    assert(meta_size);

    
    *meta_size += fspace->hdr_size + (fspace->sinfo ? fspace->sect_size : fspace->alloc_sect_size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5FS__incr(H5FS_t *fspace)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE
#ifdef H5FS_DEBUG
    Rfprintf(Rstderr, "%s: Entering, fpace->addr = %" PRIuHADDR ", fspace->rc = %u\n", __func__, fspace->addr,
            fspace->rc);
#endif 

    
    assert(fspace);

    
    if (fspace->rc == 0 && H5_addr_defined(fspace->addr))
        if (H5AC_pin_protected_entry(fspace) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTPIN, FAIL, "unable to pin free space header");

    
    fspace->rc++;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FS__decr(H5FS_t *fspace)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE
#ifdef H5FS_DEBUG
    Rfprintf(Rstderr, "%s: Entering, fpace->addr = %" PRIuHADDR ", fspace->rc = %u\n", __func__, fspace->addr,
            fspace->rc);
#endif 

    
    assert(fspace);

    
    fspace->rc--;

    
    if (fspace->rc == 0) {
        if (H5_addr_defined(fspace->addr)) {
            if (H5AC_unpin_entry(fspace) < 0)
                HGOTO_ERROR(H5E_FSPACE, H5E_CANTUNPIN, FAIL, "unable to unpin free space header");
        } 
        else {
            if (H5FS__hdr_dest(fspace) < 0)
                HGOTO_ERROR(H5E_FSPACE, H5E_CANTCLOSEOBJ, FAIL, "unable to destroy free space header");
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FS__dirty(H5FS_t *fspace)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(fspace);

    
    if (H5_addr_defined(fspace->addr))
        
        if (H5AC_mark_entry_dirty(fspace) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTMARKDIRTY, FAIL, "unable to mark free space header as dirty");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FS_alloc_hdr(H5F_t *f, H5FS_t *fspace, haddr_t *fs_addr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(fspace);

    if (!H5_addr_defined(fspace->addr)) {
        
        if (HADDR_UNDEF == (fspace->addr = H5MF_alloc(f, H5FD_MEM_FSPACE_HDR, (hsize_t)H5FS_HEADER_SIZE(f))))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "file allocation failed for free space header");

        
        if (H5AC_insert_entry(f, H5AC_FSPACE_HDR, fspace->addr, fspace, H5AC__PIN_ENTRY_FLAG) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTINIT, FAIL, "can't add free space header to cache");
    } 

    if (fs_addr)
        *fs_addr = fspace->addr;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FS_alloc_sect(H5F_t *f, H5FS_t *fspace)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(fspace);

    if (!H5_addr_defined(fspace->sect_addr) && fspace->sinfo && fspace->serial_sect_count > 0) {
        if (HADDR_UNDEF == (fspace->sect_addr = H5MF_alloc(f, H5FD_MEM_FSPACE_SINFO, fspace->sect_size)))
            HGOTO_ERROR(H5E_FSPACE, H5E_NOSPACE, FAIL, "file allocation failed for section info");
        fspace->alloc_sect_size = fspace->sect_size;

        
        if (H5FS__dirty(fspace) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTMARKDIRTY, FAIL, "unable to mark free space header as dirty");

        
        if (H5AC_insert_entry(f, H5AC_FSPACE_SINFO, fspace->sect_addr, fspace->sinfo, H5AC__NO_FLAGS_SET) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTINIT, FAIL, "can't add free space sections to cache");

        
        fspace->sinfo = NULL;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FS_free(H5F_t *f, H5FS_t *fspace, bool free_file_space)
{
    haddr_t  saved_addr;          
    unsigned cache_flags;         
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(fspace);

    cache_flags = H5AC__DELETED_FLAG | H5AC__TAKE_OWNERSHIP_FLAG;

    
    if (H5_addr_defined(fspace->sect_addr)) {
        hsize_t  saved_size;       
        unsigned sinfo_status = 0; 

        
        if (H5AC_get_entry_status(f, fspace->sect_addr, &sinfo_status) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTGET, FAIL,
                        "unable to check metadata cache status for free-space section info");

        
        if (sinfo_status & H5AC_ES__IN_CACHE || !fspace->sinfo) {
            H5FS_sinfo_cache_ud_t cache_udata; 

            
            cache_udata.f      = f;
            cache_udata.fspace = fspace;
            if (NULL == (fspace->sinfo = (H5FS_sinfo_t *)H5AC_protect(f, H5AC_FSPACE_SINFO, fspace->sect_addr,
                                                                      &cache_udata, H5AC__READ_ONLY_FLAG)))
                HGOTO_ERROR(H5E_FSPACE, H5E_CANTPROTECT, FAIL, "unable to protect free space section info");

            
            if (H5AC_unprotect(f, H5AC_FSPACE_SINFO, fspace->sect_addr, fspace->sinfo, cache_flags) < 0)
                HGOTO_ERROR(H5E_FSPACE, H5E_CANTUNPROTECT, FAIL, "unable to release free space section info");
        } 

        saved_addr = fspace->sect_addr;
        saved_size = fspace->alloc_sect_size;

        fspace->sect_addr       = HADDR_UNDEF;
        fspace->alloc_sect_size = 0;

        
        if (!H5F_IS_TMP_ADDR(f, saved_addr)) {
            if (free_file_space && H5MF_xfree(f, H5FD_MEM_FSPACE_SINFO, saved_addr, saved_size) < 0)
                HGOTO_ERROR(H5E_FSPACE, H5E_CANTFREE, FAIL, "unable to release free space sections");
        } 

        
        if (H5FS__dirty(fspace) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTMARKDIRTY, FAIL, "unable to mark free space header as dirty");
    } 

    
    if (H5_addr_defined(fspace->addr)) {
        unsigned hdr_status = 0; 

        
        if (H5AC_get_entry_status(f, fspace->addr, &hdr_status) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTGET, FAIL,
                        "unable to check metadata cache status for free-space section info");

        if (hdr_status & H5AC_ES__IN_CACHE) {
            H5FS_hdr_cache_ud_t cache_udata; 

            
            
            cache_udata.f              = f;
            cache_udata.nclasses       = 0;
            cache_udata.classes        = NULL;
            cache_udata.cls_init_udata = NULL;
            if (NULL == (fspace = (H5FS_t *)H5AC_protect(f, H5AC_FSPACE_HDR, fspace->addr, &cache_udata,
                                                         H5AC__READ_ONLY_FLAG)))
                HGOTO_ERROR(H5E_FSPACE, H5E_CANTPROTECT, FAIL, "unable to protect free space section info");

            
            if (H5AC_unpin_entry(fspace) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTUNPIN, FAIL, "unable to unpin fractal heap header");

            
            if (H5AC_unprotect(f, H5AC_FSPACE_HDR, fspace->addr, fspace, cache_flags) < 0)
                HGOTO_ERROR(H5E_FSPACE, H5E_CANTUNPROTECT, FAIL, "unable to release free space section info");
        } 

        saved_addr   = fspace->addr;
        fspace->addr = HADDR_UNDEF;

        
        if (free_file_space &&
            H5MF_xfree(f, H5FD_MEM_FSPACE_HDR, saved_addr, (hsize_t)H5FS_HEADER_SIZE(f)) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTFREE, FAIL, "unable to free free space header");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FS__hdr_dest(H5FS_t *fspace)
{
    unsigned u;                   
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(fspace);

    
    for (u = 0; u < fspace->nclasses; u++) {
        
        if (fspace->sect_cls[u].term_cls)
            if ((fspace->sect_cls[u].term_cls)(&fspace->sect_cls[u]) < 0)
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTRELEASE, FAIL, "unable to finalize section class");
    } 

    
    if (fspace->sect_cls)
        fspace->sect_cls = (H5FS_section_class_t *)H5FL_SEQ_FREE(H5FS_section_class_t, fspace->sect_cls);

    
    fspace = H5FL_FREE(H5FS_t, fspace);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FS__sinfo_free_sect_cb(void *_sect, void H5_ATTR_UNUSED *key, void *op_data)
{
    H5FS_section_info_t *sect  = (H5FS_section_info_t *)_sect;  
    const H5FS_sinfo_t  *sinfo = (const H5FS_sinfo_t *)op_data; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(sect);
    assert(sinfo);

    
    (*sinfo->fspace->sect_cls[sect->type].free)(sect);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5FS__sinfo_free_node_cb(void *item, void H5_ATTR_UNUSED *key, void *op_data)
{
    H5FS_node_t *fspace_node = (H5FS_node_t *)item; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(fspace_node);
    assert(op_data);

    
    H5SL_destroy(fspace_node->sect_list, H5FS__sinfo_free_sect_cb, op_data);

    
    fspace_node = H5FL_FREE(H5FS_node_t, fspace_node);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5FS__sinfo_dest(H5FS_sinfo_t *sinfo)
{
    unsigned u;                   
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(sinfo);
    assert(sinfo->fspace);
    assert(sinfo->bins);

    
    for (u = 0; u < sinfo->nbins; u++)
        if (sinfo->bins[u].bin_list) {
            H5SL_destroy(sinfo->bins[u].bin_list, H5FS__sinfo_free_node_cb, sinfo);
            sinfo->bins[u].bin_list = NULL;
        } 

    
    sinfo->bins = H5FL_SEQ_FREE(H5FS_bin_t, sinfo->bins);

    
    if (sinfo->merge_list)
        if (H5SL_close(sinfo->merge_list) < 0)
            HGOTO_ERROR(H5E_FSPACE, H5E_CANTCLOSEOBJ, FAIL, "can't destroy section merging skip list");

    
    
    sinfo->fspace->sinfo = NULL;
    if (H5FS__decr(sinfo->fspace) < 0)
        HGOTO_ERROR(H5E_FSPACE, H5E_CANTDEC, FAIL, "unable to decrement ref. count on free space header");
    sinfo->fspace = NULL;

    
    sinfo = H5FL_FREE(H5FS_sinfo_t, sinfo);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FS_get_sect_count(const H5FS_t *frsp, hsize_t *tot_sect_count)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(frsp);
    assert(tot_sect_count);

    
    *tot_sect_count = frsp->serial_sect_count;

    FUNC_LEAVE_NOAPI(ret_value)
}

#ifdef H5FS_DEBUG_ASSERT

void
H5FS__assert(const H5FS_t *fspace)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    if (fspace->sinfo) {
        
        H5FS__sect_assert(fspace);

        
        assert(fspace->sinfo->tot_size_count >= fspace->sinfo->serial_size_count);
        assert(fspace->sinfo->tot_size_count >= fspace->sinfo->ghost_size_count);
    } 

    
    assert(fspace->tot_sect_count >= fspace->serial_sect_count);
    assert(fspace->tot_sect_count >= fspace->ghost_sect_count);
    assert(fspace->tot_sect_count == (fspace->serial_sect_count + fspace->ghost_sect_count));

    FUNC_LEAVE_NOAPI_VOID
} 
#endif 
