/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5TSmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5TSpkg.h"    

#ifdef H5_HAVE_THREADS

#ifdef H5_HAVE_C11_THREADS

herr_t
H5TS_mutex_init(H5TS_mutex_t *mutex, int type)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    if (H5_UNLIKELY(mtx_init(mutex, type) != thrd_success))
        HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS_mutex_trylock(H5TS_mutex_t *mutex, bool *acquired)
{
    int    rc;
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    rc = mtx_trylock(mutex);
    if (thrd_success == rc)
        *acquired = true;
    else if (thrd_busy == rc)
        *acquired = false;
    else
        HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS_mutex_destroy(H5TS_mutex_t *mutex)
{
    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    mtx_destroy(mutex);

    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(SUCCEED)
} 
#else
#ifdef H5_HAVE_WIN_THREADS

herr_t
H5TS_mutex_init(H5TS_mutex_t *mutex, int H5_ATTR_UNUSED type)
{
    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    InitializeCriticalSection(mutex);

    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(SUCCEED)
} 

herr_t
H5TS_mutex_trylock(H5TS_mutex_t *mutex, bool *acquired)
{
    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    *acquired = TryEnterCriticalSection(mutex);

    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(SUCCEED)
} 

herr_t
H5TS_mutex_destroy(H5TS_mutex_t *mutex)
{
    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    DeleteCriticalSection(mutex);

    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(SUCCEED)
} 

#else

herr_t
H5TS_mutex_init(H5TS_mutex_t *mutex, int type)
{
    pthread_mutexattr_t  _attr;
    pthread_mutexattr_t *attr      = NULL;
    herr_t               ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(pthread_mutexattr_init(&_attr)))
        HGOTO_DONE(FAIL);
    attr = &_attr;

    
    if (H5TS_MUTEX_TYPE_RECURSIVE == type) {
        if (H5_UNLIKELY(pthread_mutexattr_settype(attr, PTHREAD_MUTEX_RECURSIVE)))
            HGOTO_DONE(FAIL);
    }
    else
        
        if (H5_UNLIKELY(pthread_mutexattr_settype(attr, PTHREAD_MUTEX_NORMAL)))
            HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(pthread_mutex_init(mutex, attr)))
        HGOTO_DONE(FAIL);

done:
    
    if (NULL != attr)
        if (H5_UNLIKELY(pthread_mutexattr_destroy(attr)))
            ret_value = FAIL;

    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS_mutex_trylock(H5TS_mutex_t *mutex, bool *acquired)
{
    int    rc;
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    rc = pthread_mutex_trylock(mutex);
    if (0 == rc)
        *acquired = true;
    else if (EBUSY == rc)
        *acquired = false;
    else
        HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS_mutex_destroy(H5TS_mutex_t *mutex)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    if (H5_UNLIKELY(pthread_mutex_destroy(mutex)))
        HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

#endif
#endif

#endif 
