/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Cmodule.h" 
#define H5F_FRIEND     

#include "H5private.h"  
#include "H5Cpkg.h"     
#include "H5Eprivate.h" 
#include "H5Fpkg.h"     

herr_t
H5C_get_cache_auto_resize_config(const H5C_t *cache_ptr, H5C_auto_size_ctl_t *config_ptr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    if (cache_ptr == NULL)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Bad cache_ptr on entry.");
    if (config_ptr == NULL)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Bad config_ptr on entry.");

    *config_ptr = cache_ptr->resize_ctl;

    config_ptr->set_initial_size = false;
    config_ptr->initial_size     = cache_ptr->max_cache_size;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_get_cache_size(const H5C_t *cache_ptr, size_t *max_size_ptr, size_t *min_clean_size_ptr,
                   size_t *cur_size_ptr, uint32_t *cur_num_entries_ptr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    if (cache_ptr == NULL)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Bad cache_ptr on entry.");

    if (max_size_ptr != NULL)
        *max_size_ptr = cache_ptr->max_cache_size;

    if (min_clean_size_ptr != NULL)
        *min_clean_size_ptr = cache_ptr->min_clean_size;

    if (cur_size_ptr != NULL)
        *cur_size_ptr = cache_ptr->index_size;

    if (cur_num_entries_ptr != NULL)
        *cur_num_entries_ptr = cache_ptr->index_len;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_get_cache_flush_in_progress(const H5C_t *cache_ptr, bool *flush_in_progress_ptr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    if (cache_ptr == NULL)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Bad cache_ptr on entry.");

    if (flush_in_progress_ptr != NULL)
        *flush_in_progress_ptr = cache_ptr->flush_in_progress;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_get_cache_hit_rate(const H5C_t *cache_ptr, double *hit_rate_ptr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    if (cache_ptr == NULL)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Bad cache_ptr on entry.");
    if (hit_rate_ptr == NULL)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Bad hit_rate_ptr on entry.");

    assert(cache_ptr->cache_hits >= 0);
    assert(cache_ptr->cache_accesses >= cache_ptr->cache_hits);

    if (cache_ptr->cache_accesses > 0)
        *hit_rate_ptr = ((double)(cache_ptr->cache_hits)) / ((double)(cache_ptr->cache_accesses));
    else
        *hit_rate_ptr = 0.0;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_get_entry_status(const H5F_t *f, haddr_t addr, size_t *size_ptr, bool *in_cache_ptr, bool *is_dirty_ptr,
                     bool *is_protected_ptr, bool *is_pinned_ptr, bool *is_corked_ptr,
                     bool *is_flush_dep_parent_ptr, bool *is_flush_dep_child_ptr, bool *image_up_to_date_ptr)
{
    H5C_t             *cache_ptr;
    H5C_cache_entry_t *entry_ptr = NULL;
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    cache_ptr = f->shared->cache;
    assert(cache_ptr != NULL);
    assert(H5_addr_defined(addr));
    assert(in_cache_ptr != NULL);

    if (cache_ptr == NULL)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Bad cache_ptr on entry.");

    H5C__SEARCH_INDEX(cache_ptr, addr, entry_ptr, FAIL);

    if (entry_ptr == NULL) {
        
        *in_cache_ptr = false;
    } 
    else {
        *in_cache_ptr = true;
        if (size_ptr != NULL)
            *size_ptr = entry_ptr->size;
        if (is_dirty_ptr != NULL)
            *is_dirty_ptr = entry_ptr->is_dirty;
        if (is_protected_ptr != NULL)
            *is_protected_ptr = entry_ptr->is_protected;
        if (is_pinned_ptr != NULL)
            *is_pinned_ptr = entry_ptr->is_pinned;
        if (is_corked_ptr != NULL)
            *is_corked_ptr = entry_ptr->tag_info ? entry_ptr->tag_info->corked : false;
        if (is_flush_dep_parent_ptr != NULL)
            *is_flush_dep_parent_ptr = (entry_ptr->flush_dep_nchildren > 0);
        if (is_flush_dep_child_ptr != NULL)
            *is_flush_dep_child_ptr = (entry_ptr->flush_dep_nparents > 0);
        if (image_up_to_date_ptr != NULL)
            *image_up_to_date_ptr = entry_ptr->image_up_to_date;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_get_evictions_enabled(const H5C_t *cache_ptr, bool *evictions_enabled_ptr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    if (cache_ptr == NULL)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Bad cache_ptr on entry.");

    if (evictions_enabled_ptr == NULL)
        HGOTO_ERROR(H5E_CACHE, H5E_SYSTEM, FAIL, "Bad evictions_enabled_ptr on entry.");

    *evictions_enabled_ptr = cache_ptr->evictions_enabled;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5C_get_aux_ptr(const H5C_t *cache_ptr)
{
    FUNC_ENTER_NOAPI_NOERR

    
    assert(cache_ptr);

    FUNC_LEAVE_NOAPI(cache_ptr->aux_ptr)
} 

herr_t
H5C_get_entry_ring(const H5F_t *f, haddr_t addr, H5C_ring_t *ring)
{
    H5C_t             *cache_ptr;           
    H5C_cache_entry_t *entry_ptr;           
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(f->shared);
    cache_ptr = f->shared->cache;
    assert(cache_ptr);
    assert(H5_addr_defined(addr));

    
    H5C__SEARCH_INDEX(cache_ptr, addr, entry_ptr, FAIL);
    if (entry_ptr == NULL)
        HGOTO_ERROR(H5E_CACHE, H5E_NOTFOUND, FAIL, "can't find entry in index");

    
    *ring = entry_ptr->ring;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5C_get_mdc_image_info(const H5C_t *cache_ptr, haddr_t *image_addr, hsize_t *image_len)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    if (cache_ptr == NULL)
        HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "bad cache_ptr on entry");

    if (image_addr)
        *image_addr = cache_ptr->image_addr;
    if (image_len)
        *image_len = cache_ptr->image_len;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
