/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5FAmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FApkg.h"     
#include "H5FLprivate.h" 
#include "H5MMprivate.h" 
#include "H5VMprivate.h" 

static H5FA_t *H5FA__new(H5F_t *f, haddr_t fa_addr, bool from_open, void *ctx_udata);

bool H5_PKG_INIT_VAR = false;

const H5FA_class_t *const H5FA_client_class_g[] = {
    H5FA_CLS_CHUNK,      
    H5FA_CLS_FILT_CHUNK, 
    H5FA_CLS_TEST,       
};

H5FL_DEFINE_STATIC(H5FA_t);

H5FL_BLK_DEFINE(fa_native_elmt);

static H5FA_t *
H5FA__new(H5F_t *f, haddr_t fa_addr, bool from_open, void *ctx_udata)
{
    H5FA_t     *fa        = NULL; 
    H5FA_hdr_t *hdr       = NULL; 
    H5FA_t     *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(fa_addr));

    
    if (NULL == (fa = H5FL_CALLOC(H5FA_t)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTALLOC, NULL, "memory allocation failed for fixed array info");

    
    if (NULL == (hdr = H5FA__hdr_protect(f, fa_addr, ctx_udata, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTPROTECT, NULL, "unable to load fixed array header");

    
    if (from_open && hdr->pending_delete)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTOPENOBJ, NULL, "can't open fixed array pending deletion");

    
    fa->hdr = hdr;
    if (H5FA__hdr_incr(fa->hdr) < 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTINC, NULL, "can't increment reference count on shared array header");

    
    if (H5FA__hdr_fuse_incr(fa->hdr) < 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTINC, NULL,
                    "can't increment file reference count on shared array header");

    
    fa->f = f;

    
    ret_value = fa;

done:
    if (hdr && H5FA__hdr_unprotect(hdr, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_FARRAY, H5E_CANTUNPROTECT, NULL, "unable to release fixed array header");
    if (!ret_value)
        if (fa && H5FA_close(fa) < 0)
            HDONE_ERROR(H5E_FARRAY, H5E_CLOSEERROR, NULL, "unable to close fixed array");

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5FA_t *
H5FA_create(H5F_t *f, const H5FA_create_t *cparam, void *ctx_udata)
{
    H5FA_t *fa = NULL; 
    haddr_t fa_addr;   
    H5FA_t *ret_value = NULL;

    FUNC_ENTER_NOAPI(NULL)

    
    assert(f);
    assert(cparam);

    
    HDcompile_assert(H5FA_NUM_CLS_ID == NELMTS(H5FA_client_class_g));

    
    if (HADDR_UNDEF == (fa_addr = H5FA__hdr_create(f, cparam, ctx_udata)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTINIT, NULL, "can't create fixed array header");

    
    if (NULL == (fa = H5FA__new(f, fa_addr, false, ctx_udata)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTINIT, NULL,
                    "allocation and/or initialization failed for fixed array wrapper");

    
    ret_value = fa;

done:
    if (!ret_value)
        if (fa && H5FA_close(fa) < 0)
            HDONE_ERROR(H5E_FARRAY, H5E_CLOSEERROR, NULL, "unable to close fixed array");

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5FA_t *
H5FA_open(H5F_t *f, haddr_t fa_addr, void *ctx_udata)
{
    H5FA_t *fa        = NULL; 
    H5FA_t *ret_value = NULL;

    FUNC_ENTER_NOAPI(NULL)

    
    assert(f);
    assert(H5_addr_defined(fa_addr));

    
    if (NULL == (fa = H5FA__new(f, fa_addr, true, ctx_udata)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTINIT, NULL,
                    "allocation and/or initialization failed for fixed array wrapper");

    
    ret_value = fa;

done:
    if (!ret_value)
        if (fa && H5FA_close(fa) < 0)
            HDONE_ERROR(H5E_FARRAY, H5E_CLOSEERROR, NULL, "unable to close fixed array");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FA_get_nelmts(const H5FA_t *fa, hsize_t *nelmts)
{
    FUNC_ENTER_NOAPI_NOERR

    
    assert(fa);
    assert(nelmts);

    
    *nelmts = fa->hdr->stats.nelmts;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5FA_get_addr(const H5FA_t *fa, haddr_t *addr)
{
    FUNC_ENTER_NOAPI_NOERR

    
    assert(fa);
    assert(fa->hdr);
    assert(addr);

    
    *addr = fa->hdr->addr;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5FA_set(const H5FA_t *fa, hsize_t idx, const void *elmt)
{
    H5FA_hdr_t       *hdr       = fa->hdr;            
    H5FA_dblock_t    *dblock    = NULL;               
    H5FA_dblk_page_t *dblk_page = NULL;               
    unsigned dblock_cache_flags = H5AC__NO_FLAGS_SET; 
    unsigned dblk_page_cache_flags =
        H5AC__NO_FLAGS_SET;   
    bool   hdr_dirty = false; 
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(fa);
    assert(fa->hdr);

    
    hdr->f = fa->f;

    
    if (!H5_addr_defined(hdr->dblk_addr)) {
        
        hdr->dblk_addr = H5FA__dblock_create(hdr, &hdr_dirty);
        if (!H5_addr_defined(hdr->dblk_addr))
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTCREATE, FAIL, "unable to create fixed array data block");
    }

    assert(idx < hdr->cparam.nelmts);

    
    if (NULL == (dblock = H5FA__dblock_protect(hdr, hdr->dblk_addr, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTPROTECT, FAIL,
                    "unable to protect fixed array data block, address = %llu",
                    (unsigned long long)hdr->dblk_addr);

    
    if (!dblock->npages) {
        
        H5MM_memcpy(((uint8_t *)dblock->elmts) + (hdr->cparam.cls->nat_elmt_size * idx), elmt,
                    hdr->cparam.cls->nat_elmt_size);
        dblock_cache_flags |= H5AC__DIRTIED_FLAG;
    }                             
    else {                        
        size_t  page_idx;         
        size_t  dblk_page_nelmts; 
        size_t  elmt_idx;         
        haddr_t dblk_page_addr;   

        
        page_idx = (size_t)(idx / dblock->dblk_page_nelmts);
        elmt_idx = (size_t)(idx % dblock->dblk_page_nelmts);

        
        dblk_page_addr =
            dblock->addr + H5FA_DBLOCK_PREFIX_SIZE(dblock) + ((hsize_t)page_idx * dblock->dblk_page_size);

        
        if ((page_idx + 1) == dblock->npages)
            dblk_page_nelmts = dblock->last_page_nelmts;
        else
            dblk_page_nelmts = dblock->dblk_page_nelmts;

        
        if (!H5VM_bit_get(dblock->dblk_page_init, page_idx)) {
            
            if (H5FA__dblk_page_create(hdr, dblk_page_addr, dblk_page_nelmts) < 0)
                HGOTO_ERROR(H5E_FARRAY, H5E_CANTCREATE, FAIL, "unable to create data block page");

            
            H5VM_bit_set(dblock->dblk_page_init, page_idx, true);
            dblock_cache_flags |= H5AC__DIRTIED_FLAG;
        } 

        
        if (NULL ==
            (dblk_page = H5FA__dblk_page_protect(hdr, dblk_page_addr, dblk_page_nelmts, H5AC__NO_FLAGS_SET)))
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTPROTECT, FAIL,
                        "unable to protect fixed array data block page, address = %llu",
                        (unsigned long long)dblk_page_addr);

        
        H5MM_memcpy(((uint8_t *)dblk_page->elmts) + (hdr->cparam.cls->nat_elmt_size * elmt_idx), elmt,
                    hdr->cparam.cls->nat_elmt_size);
        dblk_page_cache_flags |= H5AC__DIRTIED_FLAG;
    } 

done:
    
    if (hdr_dirty)
        if (H5FA__hdr_modified(hdr) < 0)
            HDONE_ERROR(H5E_FARRAY, H5E_CANTMARKDIRTY, FAIL, "unable to mark fixed array header as modified");

    
    if (dblock && H5FA__dblock_unprotect(dblock, dblock_cache_flags) < 0)
        HDONE_ERROR(H5E_FARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release fixed array data block");
    if (dblk_page && H5FA__dblk_page_unprotect(dblk_page, dblk_page_cache_flags) < 0)
        HDONE_ERROR(H5E_FARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release fixed array data block page");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FA_get(const H5FA_t *fa, hsize_t idx, void *elmt)
{
    H5FA_hdr_t       *hdr       = fa->hdr; 
    H5FA_dblock_t    *dblock    = NULL;    
    H5FA_dblk_page_t *dblk_page = NULL;    
    herr_t            ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(fa);
    assert(fa->hdr);

    
    hdr->f = fa->f;

    
    if (!H5_addr_defined(hdr->dblk_addr)) {
        
        if ((hdr->cparam.cls->fill)(elmt, (size_t)1) < 0)
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTSET, FAIL, "can't set element to class's fill value");
    } 
    else {
        
        assert(H5_addr_defined(hdr->dblk_addr));
        if (NULL == (dblock = H5FA__dblock_protect(hdr, hdr->dblk_addr, H5AC__READ_ONLY_FLAG)))
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTPROTECT, FAIL,
                        "unable to protect fixed array data block, address = %llu",
                        (unsigned long long)hdr->dblk_addr);

        
        if (!dblock->npages)
            
            H5MM_memcpy(elmt, ((uint8_t *)dblock->elmts) + (hdr->cparam.cls->nat_elmt_size * idx),
                        hdr->cparam.cls->nat_elmt_size);
        else {               
            size_t page_idx; 

            
            page_idx = (size_t)(idx / dblock->dblk_page_nelmts);

            
            if (!H5VM_bit_get(dblock->dblk_page_init, page_idx)) {
                
                if ((hdr->cparam.cls->fill)(elmt, (size_t)1) < 0)
                    HGOTO_ERROR(H5E_FARRAY, H5E_CANTSET, FAIL, "can't set element to class's fill value");

                
                HGOTO_DONE(SUCCEED);
            }                             
            else {                        
                size_t  dblk_page_nelmts; 
                size_t  elmt_idx;         
                haddr_t dblk_page_addr;   

                
                elmt_idx = (size_t)(idx % dblock->dblk_page_nelmts);

                
                dblk_page_addr = dblock->addr + H5FA_DBLOCK_PREFIX_SIZE(dblock) +
                                 ((hsize_t)page_idx * dblock->dblk_page_size);

                
                if ((page_idx + 1) == dblock->npages)
                    dblk_page_nelmts = dblock->last_page_nelmts;
                else
                    dblk_page_nelmts = dblock->dblk_page_nelmts;

                
                if (NULL == (dblk_page = H5FA__dblk_page_protect(hdr, dblk_page_addr, dblk_page_nelmts,
                                                                 H5AC__READ_ONLY_FLAG)))
                    HGOTO_ERROR(H5E_FARRAY, H5E_CANTPROTECT, FAIL,
                                "unable to protect fixed array data block page, address = %llu",
                                (unsigned long long)dblk_page_addr);

                
                H5MM_memcpy(elmt, ((uint8_t *)dblk_page->elmts) + (hdr->cparam.cls->nat_elmt_size * elmt_idx),
                            hdr->cparam.cls->nat_elmt_size);
            } 
        }     
    }         

done:
    if (dblock && H5FA__dblock_unprotect(dblock, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_FARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release fixed array data block");
    if (dblk_page && H5FA__dblk_page_unprotect(dblk_page, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_FARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release fixed array data block page");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FA_close(H5FA_t *fa)
{
    bool    pending_delete = false;       
    haddr_t fa_addr        = HADDR_UNDEF; 
    herr_t  ret_value      = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(fa);

    
    if (fa->hdr) {
        
        if (0 == H5FA__hdr_fuse_decr(fa->hdr)) {
            
            fa->hdr->f = fa->f;

            

            
            if (fa->hdr->pending_delete) {
                
                pending_delete = true;
                fa_addr        = fa->hdr->addr;
            } 
        }     

        
        if (pending_delete) {
            H5FA_hdr_t *hdr; 

#ifndef NDEBUG
            {
                unsigned hdr_status = 0; 

                
                if (H5AC_get_entry_status(fa->f, fa_addr, &hdr_status) < 0)
                    HGOTO_ERROR(H5E_FARRAY, H5E_CANTGET, FAIL,
                                "unable to check metadata cache status for fixed array header");

                
                assert(hdr_status & H5AC_ES__IN_CACHE);
                assert(hdr_status & H5AC_ES__IS_PINNED);
                assert(!(hdr_status & H5AC_ES__IS_PROTECTED));
            }
#endif 

            
            
            if (NULL == (hdr = H5FA__hdr_protect(fa->f, fa_addr, NULL, H5AC__NO_FLAGS_SET)))
                HGOTO_ERROR(H5E_FARRAY, H5E_CANTLOAD, FAIL, "unable to load fixed array header");

            
            hdr->f = fa->f;

            
            
            if (H5FA__hdr_decr(fa->hdr) < 0)
                HGOTO_ERROR(H5E_FARRAY, H5E_CANTDEC, FAIL,
                            "can't decrement reference count on shared array header");

            
            if (H5FA__hdr_delete(hdr) < 0)
                HGOTO_ERROR(H5E_FARRAY, H5E_CANTDELETE, FAIL, "unable to delete fixed array");
        } 
        else {
            
            
            if (H5FA__hdr_decr(fa->hdr) < 0)
                HGOTO_ERROR(H5E_FARRAY, H5E_CANTDEC, FAIL,
                            "can't decrement reference count on shared array header");
        } 
    }     

    
    fa = H5FL_FREE(H5FA_t, fa);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FA_delete(H5F_t *f, haddr_t fa_addr, void *ctx_udata)
{
    H5FA_hdr_t *hdr       = NULL; 
    herr_t      ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(H5_addr_defined(fa_addr));

    
    if (NULL == (hdr = H5FA__hdr_protect(f, fa_addr, ctx_udata, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTPROTECT, FAIL, "unable to protect fixed array header, address = %llu",
                    (unsigned long long)fa_addr);

    
    if (hdr->file_rc)
        hdr->pending_delete = true;
    else {
        
        hdr->f = f;

        
        if (H5FA__hdr_delete(hdr) < 0)
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTDELETE, FAIL, "unable to delete fixed array");
        hdr = NULL;
    }

done:
    
    if (hdr && H5FA__hdr_unprotect(hdr, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_FARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release fixed array header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5FA_iterate(H5FA_t *fa, H5FA_operator_t op, void *udata)
{
    uint8_t *elmt = NULL;
    hsize_t  u;
    int      ret_value = H5_ITER_CONT;

    FUNC_ENTER_NOAPI(H5_ITER_ERROR)

    
    assert(fa);
    assert(op);
    assert(udata);

    
    if (NULL == (elmt = H5FL_BLK_MALLOC(fa_native_elmt, fa->hdr->cparam.cls->nat_elmt_size)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTALLOC, H5_ITER_ERROR,
                    "memory allocation failed for fixed array element");

    
    for (u = 0; u < fa->hdr->stats.nelmts && ret_value == H5_ITER_CONT; u++) {
        
        if (H5FA_get(fa, u, elmt) < 0)
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTGET, H5_ITER_ERROR, "unable to delete fixed array");

        
        if ((ret_value = (*op)(u, elmt, udata)) < 0) {
            HERROR(H5E_FARRAY, H5E_BADITER, "iteration callback error");
            break;
        }
    }

done:
    if (elmt)
        elmt = H5FL_BLK_FREE(fa_native_elmt, elmt);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FA_depend(H5FA_t *fa, H5AC_proxy_entry_t *parent)
{
    H5FA_hdr_t *hdr       = fa->hdr; 
    herr_t      ret_value = SUCCEED;

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(fa);
    assert(hdr);
    assert(parent);

    
    if (NULL == hdr->parent) {
        
        assert(hdr->top_proxy);

        
        hdr->f = fa->f;

        
        if (H5AC_proxy_entry_add_child(parent, hdr->f, hdr->top_proxy) < 0)
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTSET, FAIL, "unable to add fixed array as child of proxy");
        hdr->parent = parent;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FA_patch_file(H5FA_t *fa, H5F_t *f)
{
    FUNC_ENTER_NOAPI_NOERR

    
    assert(fa);
    assert(f);

    if (fa->f != f || fa->hdr->f != f)
        fa->f = fa->hdr->f = f;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
