/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Dmodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Dpkg.h"      
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FLprivate.h" 
#include "H5FOprivate.h" 
#include "H5Iprivate.h"  
#include "H5Lprivate.h"  
#include "H5MMprivate.h" 
#include "H5SLprivate.h" 
#include "H5VLprivate.h" 
#include "H5VMprivate.h" 

typedef struct {
    void   *fl_tbuf;      
    void   *vl_tbuf;      
    size_t  vl_tbuf_size; 
    hsize_t size;         
} H5D_vlen_bufsize_common_t;

typedef struct {
    H5D_t                    *dset;   
    H5S_t                    *fspace; 
    H5S_t                    *mspace; 
    H5D_vlen_bufsize_common_t common; 
} H5D_vlen_bufsize_native_t;

typedef struct {
    H5VL_object_t            *dset_vol_obj; 
    hid_t                     fspace_id;    
    H5S_t                    *fspace;       
    hid_t                     mspace_id;    
    hid_t                     dxpl_id;      
    H5D_vlen_bufsize_common_t common;       
} H5D_vlen_bufsize_generic_t;

static H5D_shared_t *H5D__new(hid_t dcpl_id, hid_t dapl_id, bool creating, bool vl_type);
static herr_t        H5D__init_type(H5F_t *file, const H5D_t *dset, hid_t type_id, H5T_t *type);
static herr_t        H5D__cache_dataspace_info(const H5D_t *dset);
static herr_t        H5D__init_space(H5F_t *file, const H5D_t *dset, const H5S_t *space);
static herr_t        H5D__update_oh_info(H5F_t *file, H5D_t *dset, hid_t dapl_id);
static herr_t H5D__build_file_prefix(const H5D_t *dset, H5F_prefix_open_t prefix_type, char **file_prefix);
static herr_t H5D__open_oid(H5D_t *dataset, hid_t dapl_id);
static herr_t H5D__init_storage(H5D_t *dset, bool full_overwrite, hsize_t old_dim[]);
static herr_t H5D__append_flush_setup(H5D_t *dset, hid_t dapl_id);
static herr_t H5D__close_cb(void *dset_vol_obj, void **request);
static herr_t H5D__use_minimized_dset_headers(H5F_t *file, bool *minimize);
static herr_t H5D__prepare_minimized_oh(H5F_t *file, H5D_t *dset, H5O_loc_t *oloc);
static size_t H5D__calculate_minimum_header_size(H5F_t *file, H5D_t *dset, H5O_t *ohdr);
static void  *H5D__vlen_get_buf_size_alloc(size_t size, void *info);
static herr_t H5D__vlen_get_buf_size_cb(void *elem, const H5T_t *type, unsigned ndim, const hsize_t *point,
                                        void *op_data);
static herr_t H5D__vlen_get_buf_size_gen_cb(void *elem, hid_t type_id, unsigned ndim, const hsize_t *point,
                                            void *op_data);
static herr_t H5D__check_filters(H5D_t *dataset);

H5FL_BLK_DEFINE(vlen_vl_buf);

H5FL_BLK_DEFINE(vlen_fl_buf);

H5FL_DEFINE_STATIC(H5D_t);
H5FL_DEFINE_STATIC(H5D_shared_t);

H5FL_BLK_EXTERN(sieve_buf);

H5FL_EXTERN(H5D_piece_info_t);

H5FL_BLK_EXTERN(type_conv);

H5_WARN_LARGE_STACK_OBJECTS_OFF
static H5D_shared_t H5D_def_dset;
H5_WARN_LARGE_STACK_OBJECTS_ON

static const H5I_class_t H5I_DATASET_CLS[1] = {{
    H5I_DATASET,              
    0,                        
    0,                        
    (H5I_free_t)H5D__close_cb 
}};

static bool H5D_top_package_initialize_s = false;

static const char *H5D_prefix_ext_env = NULL;
static const char *H5D_prefix_vds_env = NULL;

herr_t
H5D_init(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)
    

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__init_package(void)
{
    H5P_genplist_t *def_dcpl;            
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (H5I_register_type(H5I_DATASET_CLS) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to initialize interface");

    
    memset(&H5D_def_dset, 0, sizeof(H5D_shared_t));
    H5D_def_dset.type_id = H5I_INVALID_HID;
    H5D_def_dset.dapl_id = H5I_INVALID_HID;
    H5D_def_dset.dcpl_id = H5I_INVALID_HID;
    
    H5D_def_dset.layout_copied_to_dcpl = false;

    
    if (NULL == (def_dcpl = (H5P_genplist_t *)H5I_object(H5P_LST_DATASET_CREATE_ID_g)))
        HGOTO_ERROR(H5E_DATASET, H5E_BADTYPE, FAIL, "can't get default dataset creation property list");

    
    if (H5P_get(def_dcpl, H5D_CRT_LAYOUT_NAME, &H5D_def_dset.layout) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't retrieve layout");

    
    if (H5P_get(def_dcpl, H5D_CRT_EXT_FILE_LIST_NAME, &H5D_def_dset.dcpl_cache.efl) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't retrieve external file list");
    if (H5P_get(def_dcpl, H5D_CRT_FILL_VALUE_NAME, &H5D_def_dset.dcpl_cache.fill) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't retrieve fill value");
    if (H5P_get(def_dcpl, H5O_CRT_PIPELINE_NAME, &H5D_def_dset.dcpl_cache.pline) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't retrieve pipeline filter");

    
    H5D_top_package_initialize_s = true;

    
    H5D_prefix_vds_env = getenv("HDF5_VDS_PREFIX");
    H5D_prefix_ext_env = getenv("HDF5_EXTFILE_PREFIX");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5D_top_term_package(void)
{
    int n = 0;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    if (H5D_top_package_initialize_s) {
        if (H5I_nmembers(H5I_DATASET) > 0) {
            
            (void)H5I_clear_type(H5I_DATASET, true, false);
            n++; 
        }        

        
        if (0 == n)
            H5D_top_package_initialize_s = false;
    } 

    FUNC_LEAVE_NOAPI(n)
} 

int
H5D_term_package(void)
{
    int n = 0;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    if (H5_PKG_INIT_VAR) {
        
        assert(0 == H5I_nmembers(H5I_DATASET));
        assert(false == H5D_top_package_initialize_s);

        
        n += (H5I_dec_type_ref(H5I_DATASET) > 0);

        
        if (0 == n)
            H5_PKG_INIT_VAR = false;
    } 

    FUNC_LEAVE_NOAPI(n)
} 

static herr_t
H5D__close_cb(void *_dset_vol_obj, void **request)
{
    H5VL_object_t *dset_vol_obj = (H5VL_object_t *)_dset_vol_obj;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset_vol_obj);

    
    if (H5VL_dataset_close(dset_vol_obj, H5P_DATASET_XFER_DEFAULT, request) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "unable to close dataset");

done:
    
    if (H5VL_free_object(dset_vol_obj) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTDEC, FAIL, "unable to free VOL object");

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5D_t *
H5D__create_named(const H5G_loc_t *loc, const char *name, hid_t type_id, const H5S_t *space, hid_t lcpl_id,
                  hid_t dcpl_id, hid_t dapl_id)
{
    H5O_obj_create_t ocrt_info;        
    H5D_obj_create_t dcrt_info;        
    H5D_t           *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(loc);
    assert(name && *name);
    assert(type_id != H5P_DEFAULT);
    assert(space);
    assert(lcpl_id != H5P_DEFAULT);
    assert(dcpl_id != H5P_DEFAULT);
    assert(dapl_id != H5P_DEFAULT);

    
    dcrt_info.type_id = type_id;
    dcrt_info.space   = space;
    dcrt_info.dcpl_id = dcpl_id;
    dcrt_info.dapl_id = dapl_id;

    
    ocrt_info.obj_type = H5O_TYPE_DATASET;
    ocrt_info.crt_info = &dcrt_info;
    ocrt_info.new_obj  = NULL;

    
    if (H5L_link_object(loc, name, &ocrt_info, lcpl_id) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, NULL, "unable to create and link to dataset");
    assert(ocrt_info.new_obj);

    
    ret_value = (H5D_t *)ocrt_info.new_obj;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__get_space_status(const H5D_t *dset, H5D_space_status_t *allocation)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(dset);

    
    if (dset->shared->layout.type == H5D_CHUNKED) {
        hsize_t n_chunks_total = dset->shared->layout.u.chunk.nchunks;
        hsize_t n_chunks_alloc = 0;

        if (H5D__get_num_chunks(dset, dset->shared->space, &n_chunks_alloc) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL,
                        "unable to retrieve number of allocated chunks in dataset");

        assert(n_chunks_alloc <= n_chunks_total);

        if (n_chunks_alloc == 0)
            *allocation = H5D_SPACE_STATUS_NOT_ALLOCATED;
        else if (n_chunks_alloc == n_chunks_total)
            *allocation = H5D_SPACE_STATUS_ALLOCATED;
        else
            *allocation = H5D_SPACE_STATUS_PART_ALLOCATED;
    } 
    else {
        
        if (dset->shared->layout.ops->is_space_alloc(&dset->shared->layout.storage))
            *allocation = H5D_SPACE_STATUS_ALLOCATED;
        else
            *allocation = H5D_SPACE_STATUS_NOT_ALLOCATED;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5D_shared_t *
H5D__new(hid_t dcpl_id, hid_t dapl_id, bool creating, bool vl_type)
{
    H5D_shared_t   *new_dset = NULL;  
    H5P_genplist_t *plist;            
    H5D_shared_t   *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (new_dset = H5FL_MALLOC(H5D_shared_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    H5MM_memcpy(new_dset, &H5D_def_dset, sizeof(H5D_shared_t));

    
    if (!vl_type && creating && dcpl_id == H5P_DATASET_CREATE_DEFAULT) {
        if (H5I_inc_ref(dcpl_id, false) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINC, NULL, "can't increment default DCPL ID");
        new_dset->dcpl_id = dcpl_id;

        new_dset->layout_copied_to_dcpl = true;
    } 
    else {
        
        if (NULL == (plist = (H5P_genplist_t *)H5I_object(dcpl_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a property list");

        new_dset->dcpl_id = H5P_copy_plist(plist, false);

        
        if (creating)
            new_dset->layout_copied_to_dcpl = true;
    } 

    if (!vl_type && creating && dapl_id == H5P_DATASET_ACCESS_DEFAULT) {
        if (H5I_inc_ref(dapl_id, false) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINC, NULL, "can't increment default DAPL ID");
        new_dset->dapl_id = dapl_id;
    } 
    else {
        
        if (NULL == (plist = (H5P_genplist_t *)H5I_object(dapl_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a property list");

        new_dset->dapl_id = H5P_copy_plist(plist, false);
    } 

    
    ret_value = new_dset;

done:
    if (ret_value == NULL)
        if (new_dset != NULL) {
            if (new_dset->dcpl_id != 0 && H5I_dec_ref(new_dset->dcpl_id) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CANTDEC, NULL, "can't decrement temporary datatype ID");
            if (new_dset->dapl_id != 0 && H5I_dec_ref(new_dset->dapl_id) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CANTDEC, NULL, "can't decrement temporary datatype ID");
            new_dset = H5FL_FREE(H5D_shared_t, new_dset);
        } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__init_type(H5F_t *file, const H5D_t *dset, hid_t type_id, H5T_t *type)
{
    htri_t relocatable;         
    htri_t immutable;           
    bool   use_at_least_v18;    
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(file);
    assert(dset);
    assert(type);

    
    if ((relocatable = H5T_is_relocatable(type)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "can't check datatype?");

    
    if ((immutable = H5T_is_immutable(type)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "can't check datatype?");

    
    use_at_least_v18 = (H5F_LOW_BOUND(file) >= H5F_LIBVER_V18);

    
    if (!immutable || relocatable || use_at_least_v18) {
        
        if ((dset->shared->type = H5T_copy(type, H5T_COPY_ALL)) == NULL)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "can't copy datatype");

        
        if (H5T_convert_committed_datatype(dset->shared->type, file) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't get shared datatype info");

        
        if (H5T_set_loc(dset->shared->type, H5F_VOL_OBJ(file), H5T_LOC_DISK) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't set datatype location");

        
        if (H5T_set_version(file, dset->shared->type) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set version of datatype");

        
        if ((dset->shared->type_id = H5I_register(H5I_DATATYPE, dset->shared->type, false)) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTREGISTER, FAIL, "unable to register type");
    } 
    
    else {
        if (H5I_inc_ref(type_id, false) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINC, FAIL, "Can't increment datatype ID");

        
        dset->shared->type_id = type_id;
        dset->shared->type    = type;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__cache_dataspace_info(const H5D_t *dset)
{
    int      sndims;              
    unsigned u;                   
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset);

    
    if ((sndims = H5S_get_simple_extent_dims(dset->shared->space, dset->shared->curr_dims,
                                             dset->shared->max_dims)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't cache dataspace dimensions");
    dset->shared->ndims = (unsigned)sndims;

    
    for (u = 0; u < dset->shared->ndims; u++) {
        hsize_t scaled_power2up; 

        if (!(scaled_power2up = H5VM_power2up(dset->shared->curr_dims[u])))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to get the next power of 2");
        dset->shared->curr_power2up[u] = scaled_power2up;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__init_space(H5F_t *file, const H5D_t *dset, const H5S_t *space)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(file);
    assert(dset);
    assert(space);

    
    if (NULL == (dset->shared->space = H5S_copy(space, false, true)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "can't copy dataspace");

    
    if (H5D__cache_dataspace_info(dset) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "can't cache dataspace info");

    
    if (H5S_set_version(file, dset->shared->space) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set latest version of datatype");

    
    if (H5S_select_all(dset->shared->space, true) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "unable to set all selection");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__use_minimized_dset_headers(H5F_t *file, bool *minimize)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(file);
    assert(minimize);

    
    if (H5CX_get_dset_min_ohdr_flag(minimize) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL,
                    "can't get dataset object header minimize flag from API context");

    if (false == *minimize)
        *minimize = H5F_get_min_dset_ohdr(file);

done:
    if (FAIL == ret_value)
        *minimize = false;
    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5D__calculate_minimum_header_size(H5F_t *file, H5D_t *dset, H5O_t *ohdr)
{
    H5T_t      *type             = NULL;
    H5O_fill_t *fill_prop        = NULL;
    bool        use_at_least_v18 = false;
    const char  continuation[1]  = ""; 
    size_t      get_value        = 0;
    size_t      ret_value        = 0;

    FUNC_ENTER_PACKAGE

    assert(file);
    assert(dset);
    assert(ohdr);

    type             = dset->shared->type;
    fill_prop        = &(dset->shared->dcpl_cache.fill);
    use_at_least_v18 = (H5F_LOW_BOUND(file) >= H5F_LIBVER_V18);

    
    get_value = H5O_msg_size_oh(file, ohdr, H5O_DTYPE_ID, type, 0);
    if (get_value == 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, 0, "Can't get size of datatype message");
    ret_value += get_value;

    
    get_value = H5O_msg_size_oh(file, ohdr, H5O_SDSPACE_ID, dset->shared->space, 0);
    if (get_value == 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, 0, "can't get size of dataspace message");
    ret_value += get_value;

    
    get_value = H5O_msg_size_oh(file, ohdr, H5O_LAYOUT_ID, &dset->shared->layout, 0);
    if (get_value == 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, 0, "can't get size of layout message");
    ret_value += get_value;

    
    get_value = H5O_msg_size_oh(file, ohdr, H5O_FILL_NEW_ID, fill_prop, 0);
    if (get_value == 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, 0, "can't get size of fill value message");
    ret_value += get_value;

    
    
    get_value = H5O_msg_size_oh(file, ohdr, H5O_CONT_ID, continuation, 0);
    if (get_value == 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, 0, "can't get size of continuation message");
    ret_value += get_value;

    
    if (fill_prop->buf && !use_at_least_v18) {
        H5O_fill_t old_fill_prop; 

        
        
        H5MM_memcpy(&old_fill_prop, fill_prop, sizeof(old_fill_prop));

        if (H5O_msg_reset_share(H5O_FILL_ID, &old_fill_prop) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, 0, "can't reset the copied fill property");

        get_value = H5O_msg_size_oh(file, ohdr, H5O_FILL_ID, &old_fill_prop, 0);
        if (get_value == 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, 0,
                        "can't get size of fill value (backwards compat) message");
        ret_value += get_value;
    }

    
    if (H5D_CHUNKED == dset->shared->layout.type) {
        H5O_pline_t *pline = &dset->shared->dcpl_cache.pline;
        if (pline->nused > 0) {
            get_value = H5O_msg_size_oh(file, ohdr, H5O_PLINE_ID, pline, 0);
            if (get_value == 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, 0, "can't get size of filter message");
            ret_value += get_value;
        }
    }

    
    if (dset->shared->dcpl_cache.efl.nused > 0) {
        get_value = H5O_msg_size_oh(file, ohdr, H5O_EFL_ID, &dset->shared->dcpl_cache.efl, 0);
        if (get_value == 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, 0, "can't get size of external file link message");
        ret_value += get_value;
    }

    
    if (H5O_HDR_STORE_TIMES & H5O_OH_GET_FLAGS(ohdr)) {
        assert(H5O_OH_GET_VERSION(ohdr) >= 1); 

        if (H5O_OH_GET_VERSION(ohdr) == 1) {
            
            time_t mtime;
            get_value = H5O_msg_size_oh(file, ohdr, H5O_MTIME_NEW_ID, &mtime, 0);
            if (get_value == 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, 0, "can't get size of modification time message");
            ret_value += get_value;
        }
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__prepare_minimized_oh(H5F_t *file, H5D_t *dset, H5O_loc_t *oloc)
{
    H5O_t *oh        = NULL;
    size_t ohdr_size = 0;
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(file);
    assert(dset);
    assert(oloc);

    oh = H5O_create_ohdr(file, dset->shared->dcpl_id);
    if (NULL == oh)
        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL, "can't instantiate object header");

    ohdr_size = H5D__calculate_minimum_header_size(file, dset, oh);
    if (ohdr_size == 0)
        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL, "computed header size is invalid");

    
    if (H5O_apply_ohdr(file, oh, dset->shared->dcpl_id, ohdr_size, (size_t)1, oloc) == FAIL)
        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, FAIL, "can't apply object header to file");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__update_oh_info(H5F_t *file, H5D_t *dset, hid_t dapl_id)
{
    H5O_t           *oh        = NULL;            
    size_t           ohdr_size = H5D_MINHDR_SIZE; 
    H5O_loc_t       *oloc      = NULL;            
    H5O_layout_t    *layout;                      
    H5T_t           *type;                        
    H5O_fill_t      *fill_prop;                   
    H5D_fill_value_t fill_status;                 
    bool             fill_changed = false;        
    bool             layout_init  = false; 
    bool             use_at_least_v18;     
    bool             use_minimized_header = false;   
    herr_t           ret_value            = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(file);
    assert(dset);

    
    oloc      = &dset->oloc;
    layout    = &dset->shared->layout;
    type      = dset->shared->type;
    fill_prop = &dset->shared->dcpl_cache.fill;

    
    use_at_least_v18 = (H5F_LOW_BOUND(file) >= H5F_LIBVER_V18);

    
    if (H5P_is_fill_value_defined(fill_prop, &fill_status) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't tell if fill value defined");

    
    if (H5T_detect_class(type, H5T_VLEN, false)) {
        
        if (fill_prop->fill_time == H5D_FILL_TIME_IFSET && fill_status == H5D_FILL_VALUE_DEFAULT) {
            
            fill_prop->fill_time = H5D_FILL_TIME_ALLOC;

            
            fill_changed = true;
        } 

        
        if (fill_prop->fill_time == H5D_FILL_TIME_NEVER)
            HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL,
                        "Dataset doesn't support VL datatype when fill value is not defined");
    } 

    
    if (fill_status == H5D_FILL_VALUE_DEFAULT || fill_status == H5D_FILL_VALUE_USER_DEFINED) {
        
        if (fill_prop->buf && fill_prop->size > 0 && H5O_fill_convert(fill_prop, type, &fill_changed) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to convert fill value to dataset type");

        fill_prop->fill_defined = true;
    }
    else if (fill_status == H5D_FILL_VALUE_UNDEFINED)
        fill_prop->fill_defined = false;
    else
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to determine if fill value is defined");

    
    if (fill_prop->fill_defined == false && fill_prop->fill_time == H5D_FILL_TIME_ALLOC)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL,
                    "fill value writing on allocation set, but no fill value defined");

    
    if (fill_changed) {
        H5P_genplist_t *dc_plist; 

        
        assert(dset->shared->dcpl_id != H5P_DATASET_CREATE_DEFAULT);
        if (NULL == (dc_plist = (H5P_genplist_t *)H5I_object(dset->shared->dcpl_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "can't get dataset creation property list");

        
        if (H5P_set(dc_plist, H5D_CRT_FILL_VALUE_NAME, fill_prop) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set fill value info");
    } 

    if (H5D__use_minimized_dset_headers(file, &use_minimized_header) == FAIL)
        HGOTO_ERROR(H5E_ARGS, H5E_CANTGET, FAIL, "can't get minimize settings");

    if (true == use_minimized_header) {
        if (H5D__prepare_minimized_oh(file, dset, oloc) == FAIL)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't create minimized dataset object header");
    } 
    else {
        
        if (H5D_COMPACT == layout->type)
            ohdr_size += layout->storage.u.compact.size;

        
        if (H5O_create(file, ohdr_size, (size_t)1, dset->shared->dcpl_id, oloc ) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to create dataset object header");
    } 

    assert(file == dset->oloc.file);

    
    if (NULL == (oh = H5O_pin(oloc)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTPIN, FAIL, "unable to pin dataset object header");

    
    if (!(H5O_has_chksum(oh) || (H5F_RFIC_FLAGS(file) & H5F_RFIC_UNUSUAL_NUM_UNUSED_NUMERIC_BITS)) &&
        H5T_is_numeric_with_unusual_unused_bits(type))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL,
                    "creating dataset with unusual datatype, see documentation for "
                    "H5Pset_relax_file_integrity_checks for details.");

    
    if (H5S_append(file, oh, dset->shared->space) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to update dataspace header message");

    
    if (H5O_msg_append_oh(file, oh, H5O_DTYPE_ID, H5O_MSG_FLAG_CONSTANT, 0, type) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to update datatype header message");

    
    if (H5O_msg_append_oh(file, oh, H5O_FILL_NEW_ID, H5O_MSG_FLAG_CONSTANT, 0, fill_prop) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to update new fill value header message");

    
    
    if (fill_prop->buf && !use_at_least_v18) {
        H5O_fill_t old_fill_prop; 

        
        
        H5MM_memcpy(&old_fill_prop, fill_prop, sizeof(old_fill_prop));

        
        H5O_msg_reset_share(H5O_FILL_ID, &old_fill_prop);

        
        if (H5O_msg_append_oh(file, oh, H5O_FILL_ID, H5O_MSG_FLAG_CONSTANT, 0, &old_fill_prop) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to update old fill value header message");
    } 

    
    if (H5D__layout_oh_create(file, oh, dset, dapl_id) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to update layout/pline/efl header message");

    
    layout_init = true;

#ifdef H5O_ENABLE_BOGUS
    {
        H5P_genplist_t *dc_plist; 

        
        if (NULL == (dc_plist = (H5P_genplist_t *)H5I_object(dset->shared->dcpl_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "can't get dataset creation property list");

        
        if ((H5P_exist_plist(dc_plist, H5O_BOGUS_MSG_FLAGS_NAME) > 0) &&
            (H5P_exist_plist(dc_plist, H5O_BOGUS_MSG_ID_NAME) > 0)) {

            uint8_t  bogus_flags = 0; 
            unsigned bogus_id;        

            
            if (H5P_get(dc_plist, H5O_BOGUS_MSG_ID_NAME, &bogus_id) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get bogus ID options");
            
            if (H5P_get(dc_plist, H5O_BOGUS_MSG_FLAGS_NAME, &bogus_flags) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get bogus message options");

            
            if (H5O_bogus_oh(file, oh, bogus_id, (unsigned)bogus_flags) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to create 'bogus' message");
        } 
    }
#endif 

    
    
    if (!use_at_least_v18)
        if (H5O_touch_oh(file, oh, true) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to update modification time message");

done:
    
    if (oh != NULL)
        if (H5O_unpin(oh) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTUNPIN, FAIL, "unable to unpin dataset object header");

    
    if (ret_value < 0)
        if (layout_init)
            
            if (dset->shared->layout.ops->dest && (dset->shared->layout.ops->dest)(dset) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL, "unable to destroy layout info");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__build_file_prefix(const H5D_t *dset, H5F_prefix_open_t prefix_type, char **file_prefix )
{
    const char *prefix   = NULL;     
    char       *filepath = NULL;     
    size_t      filepath_len;        
    size_t      prefix_len;          
    size_t      file_prefix_len;     
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset);
    assert(dset->oloc.file);
    filepath = H5F_EXTPATH(dset->oloc.file);
    assert(filepath);

    
    if (H5F_PREFIX_VDS == prefix_type) {
        prefix = H5D_prefix_vds_env;

        if (prefix == NULL || *prefix == '\0') {
            if (H5CX_get_vds_prefix(&prefix) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get the prefix for vds file");
        }
    }
    else if (H5F_PREFIX_EFILE == prefix_type) {
        prefix = H5D_prefix_ext_env;

        if (prefix == NULL || *prefix == '\0') {
            if (H5CX_get_ext_file_prefix(&prefix) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get the prefix for the external file");
        }
    }
    else
        HGOTO_ERROR(H5E_DATASET, H5E_BADTYPE, FAIL, "prefix name is not sensible");

    
    if (prefix == NULL || *prefix == '\0' || strcmp(prefix, ".") == 0) {
        
        *file_prefix = NULL;
    } 
    else {
        if (strncmp(prefix, "${ORIGIN}", strlen("${ORIGIN}")) == 0) {
            
            filepath_len    = strlen(filepath);
            prefix_len      = strlen(prefix);
            file_prefix_len = filepath_len + prefix_len - strlen("${ORIGIN}") + 1;

            if (NULL == (*file_prefix = (char *)H5MM_malloc(file_prefix_len)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "unable to allocate buffer");
            snprintf(*file_prefix, file_prefix_len, "%s%s", filepath, prefix + strlen("${ORIGIN}"));
        } 
        else {
            if (NULL == (*file_prefix = (char *)H5MM_strdup(prefix)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5D_t *
H5D__create(H5F_t *file, hid_t type_id, const H5S_t *space, hid_t dcpl_id, hid_t dapl_id)
{
    H5T_t          *type          = NULL; 
    H5T_t          *dt            = NULL; 
    H5D_t          *new_dset      = NULL;
    H5P_genplist_t *dc_plist      = NULL;  
    bool            has_vl_type   = false; 
    bool            layout_init   = false; 
    bool            layout_copied = false; 
    bool            fill_copied   = false; 
    bool            pline_copied  = false; 
    bool            efl_copied    = false; 
    H5G_loc_t       dset_loc;              
    H5D_t          *ret_value = NULL;      

    FUNC_ENTER_PACKAGE

    
    assert(file);
    assert(H5I_DATATYPE == H5I_get_type(type_id));
    assert(space);
    assert(H5I_GENPROP_LST == H5I_get_type(dcpl_id));

    
    if (NULL == (dt = (H5T_t *)H5I_object(type_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a datatype");

    
    type = H5T_get_actual_type(dt);

    
    if (H5T_is_sensible(type) != true)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "datatype is not sensible");

    
    if (H5T_detect_class(type, H5T_VLEN, false))
        has_vl_type = true;

    
    if (!H5S_has_extent(space))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "dataspace extent has not been set.");

    
    if (NULL == (new_dset = H5FL_CALLOC(H5D_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    dset_loc.oloc = &(new_dset->oloc);
    dset_loc.path = &(new_dset->path);
    H5G_loc_reset(&dset_loc);

    
    if (NULL == (new_dset->shared = H5D__new(dcpl_id, dapl_id, true, has_vl_type)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    if (H5D__init_type(file, new_dset, type_id, type) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, NULL, "can't copy datatype");

    
    if (H5D__init_space(file, new_dset, space) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, NULL, "can't copy dataspace");

    
    new_dset->shared->checked_filters = true;

    
    if (new_dset->shared->dcpl_id != H5P_DATASET_CREATE_DEFAULT) {
        H5O_layout_t *layout;                 
        H5O_pline_t  *pline;                  
        H5O_fill_t   *fill;                   
        H5O_efl_t    *efl;                    
        htri_t        ignore_filters = false; 

        if ((ignore_filters = H5Z_ignore_filters(new_dset->shared->dcpl_id, dt, space)) < 0)
            HGOTO_ERROR(H5E_ARGS, H5E_CANTINIT, NULL, "H5Z_has_optional_filter() failed");

        if (false == ignore_filters) {
            
            assert(new_dset->shared->layout_copied_to_dcpl);

            
            if (H5Z_can_apply(new_dset->shared->dcpl_id, new_dset->shared->type_id) < 0)
                HGOTO_ERROR(H5E_ARGS, H5E_CANTINIT, NULL, "I/O filters can't operate on this dataset");

            
            if (H5Z_set_local(new_dset->shared->dcpl_id, new_dset->shared->type_id) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, NULL, "unable to set local filter parameters");
        } 

        
        if (NULL == (dc_plist = (H5P_genplist_t *)H5I_object(new_dset->shared->dcpl_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "can't get dataset creation property list");

        
        pline = &new_dset->shared->dcpl_cache.pline;
        if (H5P_get(dc_plist, H5O_CRT_PIPELINE_NAME, pline) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, NULL, "can't retrieve pipeline filter");
        pline_copied = true;
        layout       = &new_dset->shared->layout;
        if (H5P_get(dc_plist, H5D_CRT_LAYOUT_NAME, layout) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, NULL, "can't retrieve layout");
        layout_copied = true;
        fill          = &new_dset->shared->dcpl_cache.fill;
        if (H5P_get(dc_plist, H5D_CRT_FILL_VALUE_NAME, fill) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, NULL, "can't retrieve fill value info");
        fill_copied = true;
        efl         = &new_dset->shared->dcpl_cache.efl;
        if (H5P_get(dc_plist, H5D_CRT_EXT_FILE_LIST_NAME, efl) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, NULL, "can't retrieve external file list");
        efl_copied = true;

        if (false == ignore_filters) {
            
            if (pline->nused > 0 && H5D_CHUNKED != layout->type)
                HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, NULL, "filters can only be used with chunked layout");
        }

        
        if (fill->alloc_time == H5D_ALLOC_TIME_DEFAULT)
            HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, NULL, "invalid space allocation state");

        
        if (layout->type == H5D_COMPACT && fill->alloc_time != H5D_ALLOC_TIME_EARLY)
            HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, NULL, "compact dataset must have early space allocation");
    } 

    
    if (H5O_pline_set_version(file, &new_dset->shared->dcpl_cache.pline) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, NULL, "can't set latest version of I/O filter pipeline");

    
    if (H5O_fill_set_version(file, &new_dset->shared->dcpl_cache.fill) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, NULL, "can't set latest version of fill value");

    
    if (H5F_HAS_FEATURE(file, H5FD_FEAT_ALLOCATE_EARLY))
        new_dset->shared->dcpl_cache.fill.alloc_time = H5D_ALLOC_TIME_EARLY;

    
    if (H5F_HAS_FEATURE(file, H5FD_FEAT_HAS_MPI) && (new_dset->shared->dcpl_cache.pline.nused == 0))
        new_dset->shared->dcpl_cache.fill.alloc_time = H5D_ALLOC_TIME_EARLY;

    
    if (H5D__layout_set_io_ops(new_dset) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, NULL, "unable to initialize I/O operations");

    
    if (new_dset->shared->layout.ops->construct &&
        (new_dset->shared->layout.ops->construct)(file, new_dset) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, NULL, "unable to construct layout information");

    
    if (new_dset->shared->layout.version > H5O_layout_ver_bounds[H5F_HIGH_BOUND(file)])
        HGOTO_ERROR(H5E_DATASET, H5E_BADRANGE, NULL, "layout version out of bounds");

    
    if (H5D__update_oh_info(file, new_dset, new_dset->shared->dapl_id) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, NULL, "can't update the metadata cache");

    
    layout_init = true;

    
    if (H5D__append_flush_setup(new_dset, new_dset->shared->dapl_id) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, NULL, "unable to set up flush append property");

    
    if (H5D__build_file_prefix(new_dset, H5F_PREFIX_EFILE, &new_dset->shared->extfile_prefix) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, NULL, "unable to initialize external file prefix");

    
    if (H5D__build_file_prefix(new_dset, H5F_PREFIX_VDS, &new_dset->shared->vds_prefix) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, NULL, "unable to initialize VDS prefix");

    
    if (H5FO_top_incr(new_dset->oloc.file, new_dset->oloc.addr) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINC, NULL, "can't incr object ref. count");
    if (H5FO_insert(new_dset->oloc.file, new_dset->oloc.addr, new_dset->shared, true) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINSERT, NULL, "can't insert dataset into list of open objects");
    new_dset->shared->fo_count = 1;

    
    ret_value = new_dset;

done:
    if (!ret_value && new_dset) {
        if (new_dset->shared) {
            if (layout_init)
                if (new_dset->shared->layout.ops->dest && (new_dset->shared->layout.ops->dest)(new_dset) < 0)
                    HDONE_ERROR(H5E_DATASET, H5E_CANTRELEASE, NULL, "unable to destroy layout info");
            if (pline_copied)
                if (H5O_msg_reset(H5O_PLINE_ID, &new_dset->shared->dcpl_cache.pline) < 0)
                    HDONE_ERROR(H5E_DATASET, H5E_CANTRESET, NULL, "unable to reset I/O pipeline info");
            if (layout_copied)
                if (H5O_msg_reset(H5O_LAYOUT_ID, &new_dset->shared->layout) < 0)
                    HDONE_ERROR(H5E_DATASET, H5E_CANTRESET, NULL, "unable to reset layout info");
            if (fill_copied)
                if (H5O_msg_reset(H5O_FILL_ID, &new_dset->shared->dcpl_cache.fill) < 0)
                    HDONE_ERROR(H5E_DATASET, H5E_CANTRESET, NULL, "unable to reset fill-value info");
            if (efl_copied)
                if (H5O_msg_reset(H5O_EFL_ID, &new_dset->shared->dcpl_cache.efl) < 0)
                    HDONE_ERROR(H5E_DATASET, H5E_CANTRESET, NULL, "unable to reset external file list info");
            if (new_dset->shared->space && H5S_close(new_dset->shared->space) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, NULL, "unable to release dataspace");

            if (new_dset->shared->type) {
                if (new_dset->shared->type_id > 0) {
                    if (H5I_dec_ref(new_dset->shared->type_id) < 0)
                        HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, NULL, "unable to release datatype");
                } 
                else {
                    if (H5T_close_real(new_dset->shared->type) < 0)
                        HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, NULL, "unable to release datatype");
                } 
            }     

            if (H5_addr_defined(new_dset->oloc.addr)) {
                if (H5O_dec_rc_by_loc(&(new_dset->oloc)) < 0)
                    HDONE_ERROR(H5E_DATASET, H5E_CANTDEC, NULL,
                                "unable to decrement refcount on newly created object");
                if (H5O_close(&(new_dset->oloc), NULL) < 0)
                    HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, NULL, "unable to release object header");
                if (file) {
                    if (H5O_delete(file, new_dset->oloc.addr) < 0)
                        HDONE_ERROR(H5E_DATASET, H5E_CANTDELETE, NULL, "unable to delete object header");
                } 
            }     
            if (new_dset->shared->dcpl_id != 0 && H5I_dec_ref(new_dset->shared->dcpl_id) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CANTDEC, NULL, "unable to decrement ref count on property list");
            if (new_dset->shared->dapl_id != 0 && H5I_dec_ref(new_dset->shared->dapl_id) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CANTDEC, NULL, "unable to decrement ref count on property list");
            new_dset->shared->extfile_prefix = (char *)H5MM_xfree(new_dset->shared->extfile_prefix);
            new_dset->shared->vds_prefix     = (char *)H5MM_xfree(new_dset->shared->vds_prefix);
            new_dset->shared                 = H5FL_FREE(H5D_shared_t, new_dset->shared);
        } 
        new_dset->oloc.file = NULL;
        new_dset            = H5FL_FREE(H5D_t, new_dset);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5D_t *
H5D__open_name(const H5G_loc_t *loc, const char *name, hid_t dapl_id)
{
    H5D_t     *dset = NULL;
    H5G_loc_t  dset_loc;          
    H5G_name_t path;              
    H5O_loc_t  oloc;              
    H5O_type_t obj_type;          
    bool       loc_found = false; 
    H5D_t     *ret_value = NULL;  

    FUNC_ENTER_PACKAGE

    
    assert(loc);
    assert(name);

    
    dset_loc.oloc = &oloc;
    dset_loc.path = &path;
    H5G_loc_reset(&dset_loc);

    
    if (H5G_loc_find(loc, name, &dset_loc) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_NOTFOUND, NULL, "not found");
    loc_found = true;

    
    if (H5O_obj_type(&oloc, &obj_type) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, NULL, "can't get object type");
    if (obj_type != H5O_TYPE_DATASET)
        HGOTO_ERROR(H5E_DATASET, H5E_BADTYPE, NULL, "not a dataset");

    
    if (NULL == (dset = H5D_open(&dset_loc, dapl_id)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, NULL, "can't open dataset");

    
    ret_value = dset;

done:
    if (!ret_value)
        if (loc_found && H5G_loc_free(&dset_loc) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTRELEASE, NULL, "can't free location");

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5D_t *
H5D_open(const H5G_loc_t *loc, hid_t dapl_id)
{
    H5D_shared_t *shared_fo      = NULL;
    H5D_t        *dataset        = NULL;
    char         *extfile_prefix = NULL; 
    char         *vds_prefix     = NULL; 
    H5D_t        *ret_value      = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(loc);

    
    if (NULL == (dataset = H5FL_CALLOC(H5D_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    if (H5O_loc_copy_shallow(&(dataset->oloc), loc->oloc) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, NULL, "can't copy object location");

    
    if (H5G_name_copy(&(dataset->path), loc->path, H5_COPY_SHALLOW) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, NULL, "can't copy path");

    
    if (H5D__build_file_prefix(dataset, H5F_PREFIX_EFILE, &extfile_prefix) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, NULL, "unable to initialize external file prefix");

    
    if (H5D__build_file_prefix(dataset, H5F_PREFIX_VDS, &vds_prefix) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, NULL, "unable to initialize VDS prefix");

    
    if (NULL == (shared_fo = (H5D_shared_t *)H5FO_opened(dataset->oloc.file, dataset->oloc.addr))) {
        
        if (H5D__open_oid(dataset, dapl_id) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_NOTFOUND, NULL, "not found");

        
        if (H5FO_insert(dataset->oloc.file, dataset->oloc.addr, dataset->shared, false) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINSERT, NULL, "can't insert dataset into list of open objects");

        
        if (H5FO_top_incr(dataset->oloc.file, dataset->oloc.addr) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINC, NULL, "can't increment object count");

        
        dataset->shared->fo_count = 1;

        
        dataset->shared->extfile_prefix = extfile_prefix;
        
        extfile_prefix = NULL;

        
        dataset->shared->vds_prefix = vds_prefix;
        
        vds_prefix = NULL;

    } 
    else {
        
        dataset->shared = shared_fo;

        
        shared_fo->fo_count++;

        
        if (extfile_prefix && dataset->shared->extfile_prefix) {
            if (strcmp(extfile_prefix, dataset->shared->extfile_prefix) != 0)
                HGOTO_ERROR(
                    H5E_DATASET, H5E_CANTOPENOBJ, NULL,
                    "new external file prefix does not match external file prefix of already open dataset");
        }
        else {
            if (extfile_prefix || dataset->shared->extfile_prefix)
                HGOTO_ERROR(
                    H5E_DATASET, H5E_CANTOPENOBJ, NULL,
                    "new external file prefix does not match external file prefix of already open dataset");
        }

        
        if (H5FO_top_count(dataset->oloc.file, dataset->oloc.addr) == 0) {
            
            if (H5O_open(&(dataset->oloc)) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTOPENOBJ, NULL, "unable to open object header");
        } 

        
        if (H5FO_top_incr(dataset->oloc.file, dataset->oloc.addr) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINC, NULL, "can't increment object count");
    } 

    
    ret_value = dataset;

done:
    extfile_prefix = (char *)H5MM_xfree(extfile_prefix);
    vds_prefix     = (char *)H5MM_xfree(vds_prefix);

    if (ret_value == NULL) {
        
        if (dataset) {
            if (shared_fo == NULL && dataset->shared) { 
                dataset->shared->extfile_prefix = (char *)H5MM_xfree(dataset->shared->extfile_prefix);
                dataset->shared->vds_prefix     = (char *)H5MM_xfree(dataset->shared->vds_prefix);
                dataset->shared                 = H5FL_FREE(H5D_shared_t, dataset->shared);
            }

            H5O_loc_free(&(dataset->oloc));
            H5G_name_free(&(dataset->path));

            dataset = H5FL_FREE(H5D_t, dataset);
        } 
        if (shared_fo)
            shared_fo->fo_count--;
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__append_flush_setup(H5D_t *dset, hid_t dapl_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset);
    assert(dset->shared);

    
    memset(&dset->shared->append_flush, 0, sizeof(dset->shared->append_flush));

    
    if (dapl_id != H5P_DATASET_ACCESS_DEFAULT && dset->shared->layout.type == H5D_CHUNKED) {
        H5P_genplist_t *dapl; 

        
        if (NULL == (dapl = (H5P_genplist_t *)H5I_object(dapl_id)))
            HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for dapl ID");

        
        if (H5P_exist_plist(dapl, H5D_ACS_APPEND_FLUSH_NAME) > 0) {
            H5D_append_flush_t info;

            
            if (H5P_get(dapl, H5D_ACS_APPEND_FLUSH_NAME, &info) < 0)
                HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get append flush info");
            if (info.ndims > 0) {
                hsize_t  curr_dims[H5S_MAX_RANK]; 
                hsize_t  max_dims[H5S_MAX_RANK];  
                int      rank;                    
                unsigned u;                       

                
                if ((rank = H5S_get_simple_extent_dims(dset->shared->space, curr_dims, max_dims)) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get dataset dimensions");
                if (info.ndims != (unsigned)rank)
                    HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL,
                                "boundary dimension rank does not match dataset rank");

                
                for (u = 0; u < info.ndims; u++)
                    if (info.boundary[u] != 0) 
                        
                        if (max_dims[u] != H5S_UNLIMITED && max_dims[u] == curr_dims[u])
                            break;

                
                if (u != info.ndims)
                    HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "boundary dimension is not valid");

                
                dset->shared->append_flush.ndims = info.ndims;
                dset->shared->append_flush.func  = info.func;
                dset->shared->append_flush.udata = info.udata;
                H5MM_memcpy(dset->shared->append_flush.boundary, info.boundary, sizeof(info.boundary));
            } 
        }     
    }         

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__open_oid(H5D_t *dataset, hid_t dapl_id)
{
    H5P_genplist_t *plist;                     
    H5O_fill_t     *fill_prop = NULL;          
    unsigned        alloc_time_state;          
    htri_t          msg_exists;                
    bool            layout_init       = false; 
    bool            must_init_storage = false;
    bool            fill_init         = false;   
    herr_t          ret_value         = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(dataset->oloc.addr)

    
    assert(dataset);

    
    if (NULL == (dataset->shared = H5D__new(H5P_DATASET_CREATE_DEFAULT, dapl_id, false, false)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

    
    if (H5O_open(&(dataset->oloc)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTOPENOBJ, FAIL, "unable to open");

    
    if (NULL == (dataset->shared->type = (H5T_t *)H5O_msg_read(&(dataset->oloc), H5O_DTYPE_ID, NULL)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to load type info from dataset header");

    if (H5T_set_loc(dataset->shared->type, H5F_VOL_OBJ(dataset->oloc.file), H5T_LOC_DISK) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "invalid datatype location");

    if (NULL == (dataset->shared->space = H5S_read(&(dataset->oloc))))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to load dataspace info from dataset header");

    
    if (H5D__cache_dataspace_info(dataset) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "can't cache dataspace info");

    
    if ((dataset->shared->type_id = H5I_register(H5I_DATATYPE, dataset->shared->type, false)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTREGISTER, FAIL, "unable to register type");

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object(dataset->shared->dcpl_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "can't get dataset creation property list");

    
    if (H5D__layout_oh_read(dataset, dapl_id, plist) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get layout/pline/efl info");

    
    layout_init = true;

    
    if (H5D__append_flush_setup(dataset, dapl_id))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "unable to set up flush append property");

    
    fill_prop = &dataset->shared->dcpl_cache.fill;

    
    if ((msg_exists = H5O_msg_exists(&(dataset->oloc), H5O_FILL_NEW_ID)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't check if message exists");
    if (msg_exists) {
        if (NULL == H5O_msg_read(&(dataset->oloc), H5O_FILL_NEW_ID, fill_prop))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't retrieve message");

        
        fill_init = true;
    } 
    else {
        
        if ((msg_exists = H5O_msg_exists(&(dataset->oloc), H5O_FILL_ID)) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't check if message exists");
        if (msg_exists) {
            if (NULL == H5O_msg_read(&(dataset->oloc), H5O_FILL_ID, fill_prop))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't retrieve message");

            
            fill_init = true;
        } 
        else {
            
            switch (dataset->shared->layout.type) {
                case H5D_COMPACT:
                    fill_prop->alloc_time = H5D_ALLOC_TIME_EARLY;
                    break;

                case H5D_CONTIGUOUS:
                    fill_prop->alloc_time = H5D_ALLOC_TIME_LATE;
                    break;

                case H5D_CHUNKED:
                    fill_prop->alloc_time = H5D_ALLOC_TIME_INCR;
                    break;

                case H5D_VIRTUAL:
                    fill_prop->alloc_time = H5D_ALLOC_TIME_INCR;
                    break;

                case H5D_LAYOUT_ERROR:
                case H5D_NLAYOUTS:
                default:
                    HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL, "not implemented yet");
            } 
        }     

        
        if (fill_prop->size == 0)
            fill_prop->size = (ssize_t)-1;
    } 
    alloc_time_state = 0;
    if ((dataset->shared->layout.type == H5D_COMPACT && fill_prop->alloc_time == H5D_ALLOC_TIME_EARLY) ||
        (dataset->shared->layout.type == H5D_CONTIGUOUS && fill_prop->alloc_time == H5D_ALLOC_TIME_LATE) ||
        (dataset->shared->layout.type == H5D_CHUNKED && fill_prop->alloc_time == H5D_ALLOC_TIME_INCR) ||
        (dataset->shared->layout.type == H5D_VIRTUAL && fill_prop->alloc_time == H5D_ALLOC_TIME_INCR))
        alloc_time_state = 1;

    
    if (fill_prop->buf != NULL && fill_prop->type == NULL) {
        H5T_t *tmp_type; 

        
        if (NULL == (tmp_type = H5T_copy(dataset->shared->type, H5T_COPY_TRANSIENT)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "unable to copy dataset datatype for fill value");

        
        if (!H5T_noop_conv(dataset->shared->type, tmp_type)) {
            size_t bkg_size = MAX(H5T_GET_SIZE(tmp_type), H5T_GET_SIZE(dataset->shared->type));

            assert(fill_prop->size >= 0);
            if ((size_t)fill_prop->size < bkg_size) {
                if (H5T_close_real(tmp_type) < 0)
                    HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "Can't free temporary datatype");
                HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL,
                            "fill value size doesn't match dataset's datatype size");
            }
        }

        if (H5T_close_real(tmp_type) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "Can't free temporary datatype");
    }

    
    if (H5P_fill_value_cmp(&H5D_def_dset.dcpl_cache.fill, fill_prop, sizeof(H5O_fill_t))) {
        if (H5P_set(plist, H5D_CRT_FILL_VALUE_NAME, fill_prop) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set fill value");
        if (H5P_set(plist, H5D_CRT_ALLOC_TIME_STATE_NAME, &alloc_time_state) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set allocation time state");
    } 

    
    must_init_storage = (H5F_INTENT(dataset->oloc.file) & H5F_ACC_RDWR) &&
                        !(*dataset->shared->layout.ops->is_space_alloc)(&dataset->shared->layout.storage);
    must_init_storage = must_init_storage && (H5F_HAS_FEATURE(dataset->oloc.file, H5FD_FEAT_ALLOCATE_EARLY) ||
                                              (H5F_HAS_FEATURE(dataset->oloc.file, H5FD_FEAT_HAS_MPI) &&
                                               dataset->shared->dcpl_cache.pline.nused == 0));

    if (must_init_storage && (H5D__alloc_storage(dataset, H5D_ALLOC_OPEN, false, NULL) < 0))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to initialize file storage");

done:
    if (ret_value < 0) {
        if (H5_addr_defined(dataset->oloc.addr) && H5O_close(&(dataset->oloc), NULL) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "unable to release object header");
        if (dataset->shared) {
            if (fill_init)
                H5O_msg_reset(H5O_FILL_ID, fill_prop);
            if (layout_init)
                if (dataset->shared->layout.ops->dest && (dataset->shared->layout.ops->dest)(dataset) < 0)
                    HDONE_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL, "unable to destroy layout info");
            if (dataset->shared->space && H5S_close(dataset->shared->space) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "unable to release dataspace");
            if (dataset->shared->type) {
                if (dataset->shared->type_id > 0) {
                    if (H5I_dec_ref(dataset->shared->type_id) < 0)
                        HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "unable to release datatype");
                } 
                else {
                    if (H5T_close_real(dataset->shared->type) < 0)
                        HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "unable to release datatype");
                } 
            }     
        }         
    }             

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5D_close(H5D_t *dataset)
{
    bool   free_failed = false;   
    bool   corked;                
    bool   file_closed = true;    
    herr_t ret_value   = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(dataset && dataset->oloc.file && dataset->shared);
    assert(dataset->shared->fo_count > 0);

    
#ifdef H5D_CHUNK_DEBUG
    H5D__chunk_stats(dataset, false);
#endif 

    dataset->shared->fo_count--;
    if (dataset->shared->fo_count == 0) {

        
        if (H5D__flush_real(dataset) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "unable to flush cached dataset info");

        
        
        dataset->shared->closing = true;

        
        switch (dataset->shared->layout.type) {
            case H5D_CONTIGUOUS:
                
                if (dataset->shared->cache.contig.sieve_buf)
                    dataset->shared->cache.contig.sieve_buf =
                        (unsigned char *)H5FL_BLK_FREE(sieve_buf, dataset->shared->cache.contig.sieve_buf);
                break;

            case H5D_CHUNKED:
                
                if (dataset->shared->cache.chunk.sel_chunks) {
                    assert(H5SL_count(dataset->shared->cache.chunk.sel_chunks) == 0);
                    H5SL_close(dataset->shared->cache.chunk.sel_chunks);
                    dataset->shared->cache.chunk.sel_chunks = NULL;
                } 

                
                if (dataset->shared->cache.chunk.single_space) {
                    (void)H5S_close(dataset->shared->cache.chunk.single_space);
                    dataset->shared->cache.chunk.single_space = NULL;
                } 

                
                if (dataset->shared->cache.chunk.single_piece_info) {
                    dataset->shared->cache.chunk.single_piece_info =
                        H5FL_FREE(H5D_piece_info_t, dataset->shared->cache.chunk.single_piece_info);
                    dataset->shared->cache.chunk.single_piece_info = NULL;
                } 
                break;

            case H5D_COMPACT:
                
                break;

            case H5D_VIRTUAL: {
                size_t i, j;

                assert(dataset->shared->layout.storage.u.virt.list ||
                       (dataset->shared->layout.storage.u.virt.list_nused == 0));

                
                for (i = 0; i < dataset->shared->layout.storage.u.virt.list_nused; i++) {
                    
                    if (dataset->shared->layout.storage.u.virt.list[i].source_dset.dset) {
                        assert(dataset->shared->layout.storage.u.virt.list[i].source_dset.dset != dataset);
                        if (H5D_close(dataset->shared->layout.storage.u.virt.list[i].source_dset.dset) < 0)
                            HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "unable to close source dataset");
                        dataset->shared->layout.storage.u.virt.list[i].source_dset.dset = NULL;
                    } 

                    
                    for (j = 0; j < dataset->shared->layout.storage.u.virt.list[i].sub_dset_nused; j++)
                        if (dataset->shared->layout.storage.u.virt.list[i].sub_dset[j].dset) {
                            assert(dataset->shared->layout.storage.u.virt.list[i].sub_dset[j].dset !=
                                   dataset);
                            if (H5D_close(dataset->shared->layout.storage.u.virt.list[i].sub_dset[j].dset) <
                                0)
                                HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL,
                                            "unable to close source dataset");
                            dataset->shared->layout.storage.u.virt.list[i].sub_dset[j].dset = NULL;
                        } 
                }         
            }             
            break;

            case H5D_LAYOUT_ERROR:
            case H5D_NLAYOUTS:
            default:
                assert("not implemented yet" && 0);
#ifdef NDEBUG
                HGOTO_ERROR(H5E_IO, H5E_UNSUPPORTED, FAIL, "unsupported storage layout");
#endif
        } 

        
        if (dataset->shared->layout.ops->dest && (dataset->shared->layout.ops->dest)(dataset) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL, "unable to destroy layout info");

        
        dataset->shared->extfile_prefix = (char *)H5MM_xfree(dataset->shared->extfile_prefix);

        
        dataset->shared->vds_prefix = (char *)H5MM_xfree(dataset->shared->vds_prefix);

        
        if (dataset->shared->dcpl_id != H5P_DATASET_CREATE_DEFAULT)
            free_failed |= (H5O_msg_reset(H5O_PLINE_ID, &dataset->shared->dcpl_cache.pline) < 0) ||
                           (H5O_msg_reset(H5O_LAYOUT_ID, &dataset->shared->layout) < 0) ||
                           (H5O_msg_reset(H5O_FILL_ID, &dataset->shared->dcpl_cache.fill) < 0) ||
                           (H5O_msg_reset(H5O_EFL_ID, &dataset->shared->dcpl_cache.efl) < 0);

        
        if (H5AC_cork(dataset->oloc.file, dataset->oloc.addr, H5AC__GET_CORKED, &corked) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to retrieve an object's cork status");
        if (corked)
            if (H5AC_cork(dataset->oloc.file, dataset->oloc.addr, H5AC__UNCORK, NULL) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CANTUNCORK, FAIL, "unable to uncork an object");

        
        free_failed |=
            (H5I_dec_ref(dataset->shared->type_id) < 0) || (H5S_close(dataset->shared->space) < 0) ||
            (H5I_dec_ref(dataset->shared->dcpl_id) < 0) || (H5I_dec_ref(dataset->shared->dapl_id) < 0);

        
        if (H5FO_top_decr(dataset->oloc.file, dataset->oloc.addr) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL, "can't decrement count for object");
        if (H5FO_delete(dataset->oloc.file, dataset->oloc.addr) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL, "can't remove dataset from list of open objects");

        
        
        if (H5O_close(&(dataset->oloc), &file_closed) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "unable to release object header");

        
        if (!file_closed && H5F_SHARED(dataset->oloc.file) && H5F_EVICT_ON_CLOSE(dataset->oloc.file)) {
            if (H5AC_flush_tagged_metadata(dataset->oloc.file, dataset->oloc.addr) < 0)
                HDONE_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "unable to flush tagged metadata");
            if (H5AC_evict_tagged_metadata(dataset->oloc.file, dataset->oloc.addr, false) < 0)
                HDONE_ERROR(H5E_CACHE, H5E_CANTFLUSH, FAIL, "unable to evict tagged metadata");
        } 

        
        dataset->oloc.file = NULL;
        dataset->shared    = H5FL_FREE(H5D_shared_t, dataset->shared);

    } 
    else {
        
        if (H5FO_top_decr(dataset->oloc.file, dataset->oloc.addr) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL, "can't decrement count for object");

        
        if (H5FO_top_count(dataset->oloc.file, dataset->oloc.addr) == 0) {
            if (H5O_close(&(dataset->oloc), NULL) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to close");
        } 
        else
            
            if (H5O_loc_free(&(dataset->oloc)) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL, "problem attempting to free location");
    } 

    
    if (H5G_name_free(&(dataset->path)) < 0)
        free_failed = true;

    
    dataset = H5FL_FREE(H5D_t, dataset);

    
    if (free_failed)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL,
                    "couldn't free a component of the dataset, but the dataset was freed anyway.");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D_mult_refresh_close(hid_t dset_id)
{
    H5D_t *dataset;             
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    if (NULL == (dataset = (H5D_t *)H5VL_object_verify(dset_id, H5I_DATASET)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a dataset");

    
    assert(dataset);
    assert(dataset->oloc.file);
    assert(dataset->shared);
    assert(dataset->shared->fo_count > 0);

    if (dataset->shared->fo_count > 1) {
        
        switch (dataset->shared->layout.type) {
            case H5D_CONTIGUOUS:
                
                if (dataset->shared->cache.contig.sieve_buf)
                    dataset->shared->cache.contig.sieve_buf =
                        (unsigned char *)H5FL_BLK_FREE(sieve_buf, dataset->shared->cache.contig.sieve_buf);
                break;

            case H5D_CHUNKED:
                
                if (dataset->shared->cache.chunk.sel_chunks) {
                    assert(H5SL_count(dataset->shared->cache.chunk.sel_chunks) == 0);
                    H5SL_close(dataset->shared->cache.chunk.sel_chunks);
                    dataset->shared->cache.chunk.sel_chunks = NULL;
                } 

                
                if (dataset->shared->cache.chunk.single_space) {
                    (void)H5S_close(dataset->shared->cache.chunk.single_space);
                    dataset->shared->cache.chunk.single_space = NULL;
                } 

                
                if (dataset->shared->cache.chunk.single_piece_info) {
                    dataset->shared->cache.chunk.single_piece_info =
                        H5FL_FREE(H5D_piece_info_t, dataset->shared->cache.chunk.single_piece_info);
                    dataset->shared->cache.chunk.single_piece_info = NULL;
                } 
                break;

            case H5D_COMPACT:
            case H5D_VIRTUAL:
                
                break;

            case H5D_LAYOUT_ERROR:
            case H5D_NLAYOUTS:
            default:
                assert("not implemented yet" && 0);
#ifdef NDEBUG
                HGOTO_ERROR(H5E_IO, H5E_UNSUPPORTED, FAIL, "unsupported storage layout");
#endif
        } 

        
        if (dataset->shared->layout.ops->dest && (dataset->shared->layout.ops->dest)(dataset) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL, "unable to destroy layout info");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D_mult_refresh_reopen(H5D_t *dataset)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(dataset && dataset->oloc.file && dataset->shared);
    assert(dataset->shared->fo_count > 0);

    if (dataset->shared->fo_count > 1) {
        
        if (H5S_close(dataset->shared->space) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL, "unable to release dataspace");

        
        if (NULL == (dataset->shared->space = H5S_read(&(dataset->oloc))))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to load dataspace info from dataset header");

        
        if (H5D__cache_dataspace_info(dataset) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "can't cache dataspace info");

        
        if (H5O_msg_reset(H5O_LAYOUT_ID, &dataset->shared->layout) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTRESET, FAIL, "unable to reset layout info");

        
        if (NULL == H5O_msg_read(&(dataset->oloc), H5O_LAYOUT_ID, &(dataset->shared->layout)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to read data layout message");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5O_loc_t *
H5D_oloc(H5D_t *dataset)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    FUNC_LEAVE_NOAPI(dataset ? &(dataset->oloc) : (H5O_loc_t *)NULL)
} 

H5G_name_t *
H5D_nameof(H5D_t *dataset)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    FUNC_LEAVE_NOAPI(dataset ? &(dataset->path) : NULL)
} 

herr_t
H5D__alloc_storage(H5D_t *dset, H5D_time_alloc_t time_alloc, bool full_overwrite, hsize_t old_dim[])
{
    H5F_t        *f;                         
    H5O_layout_t *layout;                    
    bool          must_init_space = false;   
    bool          addr_set        = false;   
    herr_t        ret_value       = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset);
    f = dset->oloc.file;
    assert(f);

    
    if (!(0 == H5S_GET_EXTENT_NPOINTS(dset->shared->space) || dset->shared->dcpl_cache.efl.nused > 0)) {
        
        layout = &(dset->shared->layout);

        switch (layout->type) {
            case H5D_CONTIGUOUS:
                if (!(*dset->shared->layout.ops->is_space_alloc)(&dset->shared->layout.storage)) {
                    
                    if (layout->storage.u.contig.size > 0) {
                        
                        if (H5D__contig_alloc(f, &layout->storage.u.contig ) < 0)
                            HGOTO_ERROR(H5E_IO, H5E_CANTINIT, FAIL,
                                        "unable to initialize contiguous storage");

                        
                        must_init_space = true;
                    } 
                    else
                        layout->storage.u.contig.addr = HADDR_UNDEF;

                    
                    addr_set = true;
                } 
                break;

            case H5D_CHUNKED:
                if (!(*dset->shared->layout.ops->is_space_alloc)(&dset->shared->layout.storage)) {
                    
                    if (H5D__chunk_create(dset ) < 0)
                        HGOTO_ERROR(H5E_IO, H5E_CANTINIT, FAIL, "unable to initialize chunked storage");

                    
                    addr_set = true;

                    
                    must_init_space = true;
                } 

                
                if (dset->shared->dcpl_cache.fill.alloc_time == H5D_ALLOC_TIME_EARLY &&
                    time_alloc == H5D_ALLOC_EXTEND)
                    must_init_space = true;
                break;

            case H5D_COMPACT:
                
                if (NULL == layout->storage.u.compact.buf) {
                    
                    if (layout->storage.u.compact.size > 0) {
                        if (NULL ==
                            (layout->storage.u.compact.buf = H5MM_malloc(layout->storage.u.compact.size)))
                            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                                        "unable to allocate memory for compact dataset");
                        if (!full_overwrite)
                            memset(layout->storage.u.compact.buf, 0, layout->storage.u.compact.size);
                        layout->storage.u.compact.dirty = true;

                        
                        must_init_space = true;
                    }
                    else {
                        layout->storage.u.compact.dirty = false;
                        must_init_space                 = false;
                    }
                } 
                break;

            case H5D_VIRTUAL:
                
                break;

            case H5D_LAYOUT_ERROR:
            case H5D_NLAYOUTS:
            default:
                assert("not implemented yet" && 0);
#ifdef NDEBUG
                HGOTO_ERROR(H5E_IO, H5E_UNSUPPORTED, FAIL, "unsupported storage layout");
#endif
        } 

        
        if (must_init_space) {
            if (layout->type == H5D_CHUNKED) {
                
                if (!(dset->shared->dcpl_cache.fill.alloc_time == H5D_ALLOC_TIME_INCR &&
                      time_alloc == H5D_ALLOC_WRITE))
                    if (H5D__init_storage(dset, full_overwrite, old_dim) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL,
                                    "unable to initialize dataset with fill value");
            } 
            else {
                H5D_fill_value_t fill_status; 

                
                if (H5P_is_fill_value_defined(&dset->shared->dcpl_cache.fill, &fill_status) < 0)
                    HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't tell if fill value defined");

                
                if (dset->shared->dcpl_cache.fill.fill_time == H5D_FILL_TIME_ALLOC ||
                    (dset->shared->dcpl_cache.fill.fill_time == H5D_FILL_TIME_IFSET &&
                     fill_status == H5D_FILL_VALUE_USER_DEFINED))
                    if (H5D__init_storage(dset, full_overwrite, old_dim) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL,
                                    "unable to initialize dataset with fill value");
            } 
        }     

        
        
        if (time_alloc != H5D_ALLOC_CREATE && addr_set)
            
            if (H5D__mark(dset, H5D_MARK_LAYOUT) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "unable to mark dataspace as dirty");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__init_storage(H5D_t *dset, bool full_overwrite, hsize_t old_dim[])
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(dset);

    switch (dset->shared->layout.type) {
        case H5D_COMPACT:
            
            if (!full_overwrite) {
                
                if (H5D__compact_fill(dset) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL,
                                "unable to initialize compact dataset storage");
            } 
            break;

        case H5D_CONTIGUOUS:
            
            
            if ((dset->shared->dcpl_cache.efl.nused == 0 || dset->shared->dcpl_cache.fill.buf) &&
                !full_overwrite)
                if (H5D__contig_fill(dset) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to allocate all chunks of dataset");
            break;

        case H5D_CHUNKED:
            
            {
                hsize_t zero_dim[H5O_LAYOUT_NDIMS] = {0};

                
                if (old_dim == NULL)
                    old_dim = zero_dim;

                if (H5D__chunk_allocate(dset, full_overwrite, old_dim) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to allocate all chunks of dataset");
                break;
            } 

        case H5D_VIRTUAL:
            

        case H5D_LAYOUT_ERROR:
        case H5D_NLAYOUTS:
        default:
            assert("not implemented yet" && 0);
#ifdef NDEBUG
            HGOTO_ERROR(H5E_IO, H5E_UNSUPPORTED, FAIL, "unsupported storage layout");
#endif
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__get_storage_size(const H5D_t *dset, hsize_t *storage_size)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(dset->oloc.addr)

    switch (dset->shared->layout.type) {
        case H5D_CHUNKED:
            if ((*dset->shared->layout.ops->is_space_alloc)(&dset->shared->layout.storage)) {
                if (H5D__chunk_allocated(dset, storage_size) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL,
                                "can't retrieve chunked dataset allocated size");
            } 
            else
                *storage_size = 0;
            break;

        case H5D_CONTIGUOUS:
            
            if ((*dset->shared->layout.ops->is_space_alloc)(&dset->shared->layout.storage))
                *storage_size = dset->shared->layout.storage.u.contig.size;
            else
                *storage_size = 0;
            break;

        case H5D_COMPACT:
            *storage_size = dset->shared->layout.storage.u.compact.size;
            break;

        case H5D_VIRTUAL:
            
            *storage_size = 0;
            break;

        case H5D_LAYOUT_ERROR:
        case H5D_NLAYOUTS:
        default:
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a dataset type");
    }

done:
    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

haddr_t
H5D__get_offset(const H5D_t *dset)
{
    haddr_t ret_value = HADDR_UNDEF;

    FUNC_ENTER_PACKAGE

    assert(dset);

    switch (dset->shared->layout.type) {
        case H5D_VIRTUAL:
        case H5D_CHUNKED:
        case H5D_COMPACT:
            break;

        case H5D_CONTIGUOUS:
            
            if (dset->shared->dcpl_cache.efl.nused == 0 ||
                H5_addr_defined(dset->shared->layout.storage.u.contig.addr))
                
                ret_value = dset->shared->layout.storage.u.contig.addr + H5F_BASE_ADDR(dset->oloc.file);
            break;

        case H5D_LAYOUT_ERROR:
        case H5D_NLAYOUTS:
        default:
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, HADDR_UNDEF, "unknown dataset layout type");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5D__vlen_get_buf_size_alloc(size_t size, void *info)
{
    H5D_vlen_bufsize_common_t *vlen_bufsize_com = (H5D_vlen_bufsize_common_t *)info;
    void                      *ret_value        = NULL; 

    FUNC_ENTER_PACKAGE

    
    if (size > vlen_bufsize_com->vl_tbuf_size) {
        if (NULL ==
            (vlen_bufsize_com->vl_tbuf = H5FL_BLK_REALLOC(vlen_vl_buf, vlen_bufsize_com->vl_tbuf, size)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, NULL, "can't reallocate temporary VL data buffer");
        vlen_bufsize_com->vl_tbuf_size = size;
    } 

    
    vlen_bufsize_com->size += size;

    
    ret_value = vlen_bufsize_com->vl_tbuf;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__vlen_get_buf_size_cb(void H5_ATTR_UNUSED *elem, const H5T_t *type, unsigned H5_ATTR_UNUSED ndim,
                          const hsize_t *point, void *op_data)
{
    H5D_vlen_bufsize_native_t *vlen_bufsize = (H5D_vlen_bufsize_native_t *)op_data;
    H5D_dset_io_info_t         dset_info;                
    herr_t                     ret_value = H5_ITER_CONT; 

    FUNC_ENTER_PACKAGE

    
    assert(type);
    assert(point);
    assert(op_data);

    
    if (H5S_select_elements(vlen_bufsize->fspace, H5S_SELECT_SET, (size_t)1, point) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTCREATE, H5_ITER_ERROR, "can't select point");

    {
        dset_info.dset       = vlen_bufsize->dset;
        dset_info.mem_space  = vlen_bufsize->mspace;
        dset_info.file_space = vlen_bufsize->fspace;
        dset_info.buf.vp     = vlen_bufsize->common.fl_tbuf;
        dset_info.mem_type   = type;

        
        if (H5D__read(1, &dset_info) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "can't read data");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__vlen_get_buf_size(H5D_t *dset, hid_t type_id, hid_t space_id, hsize_t *size)
{
    H5D_vlen_bufsize_native_t vlen_bufsize = {NULL, NULL, NULL, {NULL, NULL, 0, 0}};
    H5S_t                    *fspace       = NULL; 
    H5S_t                    *mspace       = NULL; 
    char                      bogus;               
    H5S_t                    *space;               
    const H5T_t              *type;                
    H5S_sel_iter_op_t         dset_op;             
    herr_t                    ret_value = FAIL;    

    FUNC_ENTER_PACKAGE

    
    if (NULL == (type = (const H5T_t *)H5I_object(type_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not an valid base datatype");
    if (NULL == (space = (H5S_t *)H5I_object(space_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid dataspace");
    if (!(H5S_has_extent(space)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "dataspace does not have extent set");

    
    vlen_bufsize.dset = dset;

    
    if (NULL == (fspace = H5S_copy(dset->shared->space, false, true)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "unable to get dataspace");
    vlen_bufsize.fspace = fspace;

    
    if (NULL == (mspace = H5S_create(H5S_SCALAR)))
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTCREATE, FAIL, "can't create dataspace");
    vlen_bufsize.mspace = mspace;

    
    if (NULL == (vlen_bufsize.common.fl_tbuf = H5FL_BLK_MALLOC(vlen_fl_buf, H5T_get_size(type))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "no temporary buffers available");
    if (NULL == (vlen_bufsize.common.vl_tbuf = H5FL_BLK_MALLOC(vlen_vl_buf, (size_t)1)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "no temporary buffers available");
    vlen_bufsize.common.vl_tbuf_size = 1;

    
    if (H5CX_set_vlen_alloc_info(H5D__vlen_get_buf_size_alloc, &vlen_bufsize.common, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set VL data allocation routine");

    
    vlen_bufsize.common.size = 0;

    
    dset_op.op_type  = H5S_SEL_ITER_OP_LIB;
    dset_op.u.lib_op = H5D__vlen_get_buf_size_cb;

    ret_value = H5S_select_iterate(&bogus, type, space, &dset_op, &vlen_bufsize);

    
    if (ret_value >= 0)
        *size = vlen_bufsize.common.size;

done:
    if (fspace && H5S_close(fspace) < 0)
        HDONE_ERROR(H5E_DATASPACE, H5E_CANTRELEASE, FAIL, "unable to release dataspace");
    if (mspace && H5S_close(mspace) < 0)
        HDONE_ERROR(H5E_DATASPACE, H5E_CANTRELEASE, FAIL, "unable to release dataspace");
    if (vlen_bufsize.common.fl_tbuf != NULL)
        vlen_bufsize.common.fl_tbuf = H5FL_BLK_FREE(vlen_fl_buf, vlen_bufsize.common.fl_tbuf);
    if (vlen_bufsize.common.vl_tbuf != NULL)
        vlen_bufsize.common.vl_tbuf = H5FL_BLK_FREE(vlen_vl_buf, vlen_bufsize.common.vl_tbuf);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__vlen_get_buf_size_gen_cb(void H5_ATTR_UNUSED *elem, hid_t type_id, unsigned H5_ATTR_UNUSED ndim,
                              const hsize_t *point, void *op_data)
{
    H5D_vlen_bufsize_generic_t *vlen_bufsize = (H5D_vlen_bufsize_generic_t *)op_data;
    H5T_t                      *dt;                  
    void                       *vol_obj_data;        
    herr_t                      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(point);
    assert(op_data);

    
    if (NULL == (dt = (H5T_t *)H5I_object(type_id)))
        HGOTO_ERROR(H5E_DATASET, H5E_BADTYPE, FAIL, "not a datatype");

    
    if (NULL == (vlen_bufsize->common.fl_tbuf =
                     H5FL_BLK_REALLOC(vlen_fl_buf, vlen_bufsize->common.fl_tbuf, H5T_get_size(dt))))
        HGOTO_ERROR(H5E_DATASET, H5E_NOSPACE, FAIL, "can't resize tbuf");

    
    if (H5S_select_elements(vlen_bufsize->fspace, H5S_SELECT_SET, (size_t)1, point) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTCREATE, FAIL, "can't select point");

    
    vol_obj_data = H5VL_OBJ_DATA(vlen_bufsize->dset_vol_obj);
    if (H5VL_dataset_read(1, &vol_obj_data, H5VL_OBJ_CONNECTOR(vlen_bufsize->dset_vol_obj), &type_id,
                          &vlen_bufsize->mspace_id, &vlen_bufsize->fspace_id, vlen_bufsize->dxpl_id,
                          &vlen_bufsize->common.fl_tbuf, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "can't read point");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__vlen_get_buf_size_gen(H5VL_object_t *vol_obj, hid_t type_id, hid_t space_id, hsize_t *size)
{
    H5D_vlen_bufsize_generic_t vlen_bufsize = {
        NULL, H5I_INVALID_HID, NULL, H5I_INVALID_HID, H5I_INVALID_HID, {NULL, NULL, 0, 0}};
    H5P_genplist_t         *dxpl   = NULL;       
    H5S_t                  *mspace = NULL;       
    char                    bogus;               
    H5S_t                  *space;               
    const H5T_t            *type;                
    H5S_sel_iter_op_t       dset_op;             
    H5VL_dataset_get_args_t vol_cb_args;         
    herr_t                  ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == (type = (const H5T_t *)H5I_object(type_id)))
        HGOTO_ERROR(H5E_DATASET, H5E_BADTYPE, FAIL, "not an valid datatype");
    if (NULL == (space = (H5S_t *)H5I_object(space_id)))
        HGOTO_ERROR(H5E_DATASET, H5E_BADTYPE, FAIL, "invalid dataspace");
    if (!(H5S_has_extent(space)))
        HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "dataspace does not have extent set");

    
    vlen_bufsize.dset_vol_obj = vol_obj;

    
    vol_cb_args.op_type                 = H5VL_DATASET_GET_SPACE;
    vol_cb_args.args.get_space.space_id = H5I_INVALID_HID;

    
    if (H5VL_dataset_get(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get dataspace");
    vlen_bufsize.fspace_id = vol_cb_args.args.get_space.space_id;
    if (NULL == (vlen_bufsize.fspace = (H5S_t *)H5I_object(vlen_bufsize.fspace_id)))
        HGOTO_ERROR(H5E_DATASET, H5E_BADTYPE, FAIL, "not a dataspace");

    
    if (NULL == (mspace = H5S_create(H5S_SCALAR)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTCREATE, FAIL, "can't create dataspace");
    if ((vlen_bufsize.mspace_id = H5I_register(H5I_DATASPACE, mspace, true)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTREGISTER, FAIL, "unable to register dataspace ID");

    
    if (NULL == (vlen_bufsize.common.fl_tbuf = H5FL_BLK_MALLOC(vlen_fl_buf, H5T_get_size(type))))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "no temporary buffers available");
    if (NULL == (vlen_bufsize.common.vl_tbuf = H5FL_BLK_MALLOC(vlen_vl_buf, (size_t)1)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "no temporary buffers available");
    vlen_bufsize.common.vl_tbuf_size = 1;

    
    if (NULL == (dxpl = (H5P_genplist_t *)H5I_object(H5P_DATASET_XFER_DEFAULT)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get default DXPL");
    if ((vlen_bufsize.dxpl_id = H5P_copy_plist(dxpl, true)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "can't copy property list");
    if (NULL == (dxpl = (H5P_genplist_t *)H5I_object(vlen_bufsize.dxpl_id)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get copied DXPL");
    if (H5P_set_vlen_mem_manager(dxpl, H5D__vlen_get_buf_size_alloc, &vlen_bufsize.common, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set VL data allocation routine on DXPL");

    
    vlen_bufsize.common.size = 0;

    
    dset_op.op_type          = H5S_SEL_ITER_OP_APP;
    dset_op.u.app_op.op      = H5D__vlen_get_buf_size_gen_cb;
    dset_op.u.app_op.type_id = type_id;

    ret_value = H5S_select_iterate(&bogus, type, space, &dset_op, &vlen_bufsize);

    
    if (ret_value >= 0)
        *size = vlen_bufsize.common.size;

done:
    if (vlen_bufsize.fspace_id >= 0) {
        if (H5I_dec_app_ref(vlen_bufsize.fspace_id) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTDEC, FAIL, "problem freeing id");
        vlen_bufsize.fspace = NULL;
    } 
    if (vlen_bufsize.fspace && H5S_close(vlen_bufsize.fspace) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL, "unable to release dataspace");
    if (vlen_bufsize.mspace_id >= 0) {
        if (H5I_dec_app_ref(vlen_bufsize.mspace_id) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTDEC, FAIL, "problem freeing id");
        mspace = NULL;
    } 
    if (mspace && H5S_close(mspace) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL, "unable to release dataspace");
    if (vlen_bufsize.common.fl_tbuf != NULL)
        vlen_bufsize.common.fl_tbuf = H5FL_BLK_FREE(vlen_fl_buf, vlen_bufsize.common.fl_tbuf);
    if (vlen_bufsize.common.vl_tbuf != NULL)
        vlen_bufsize.common.vl_tbuf = H5FL_BLK_FREE(vlen_vl_buf, vlen_bufsize.common.vl_tbuf);
    if (vlen_bufsize.dxpl_id != H5I_INVALID_HID) {
        if (H5I_dec_app_ref(vlen_bufsize.dxpl_id) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTDEC, FAIL, "can't close property list");
        dxpl = NULL;
    } 
    if (dxpl && H5P_close(dxpl) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL, "unable to release DXPL");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__check_filters(H5D_t *dataset)
{
    H5O_fill_t *fill;                
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dataset);

    
    fill = &dataset->shared->dcpl_cache.fill;
    if (!dataset->shared->checked_filters) {
        H5D_fill_value_t fill_status; 

        
        if (H5P_is_fill_value_defined(fill, &fill_status) < 0)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "Couldn't retrieve fill value from dataset.");

        
        if (fill_status == H5D_FILL_VALUE_DEFAULT || fill_status == H5D_FILL_VALUE_USER_DEFINED) {
            if (fill->fill_time == H5D_FILL_TIME_ALLOC ||
                (fill->fill_time == H5D_FILL_TIME_IFSET && fill_status == H5D_FILL_VALUE_USER_DEFINED)) {
                
                if (H5D_flush_layout_to_dcpl(dataset) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "unable to flush layout");

                
                if (H5Z_can_apply(dataset->shared->dcpl_id, dataset->shared->type_id) < 0)
                    HGOTO_ERROR(H5E_PLINE, H5E_CANAPPLY, FAIL, "can't apply filters");

                dataset->shared->checked_filters = true;
            } 
        }     
    }         

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__set_extent(H5D_t *dset, const hsize_t *size)
{
    hsize_t  curr_dims[H5S_MAX_RANK]; 
    htri_t   changed;                 
    size_t   u, v;                    
    unsigned dim_idx;                 
    herr_t   ret_value = SUCCEED;     

    FUNC_ENTER_PACKAGE_TAG(dset->oloc.addr)

    
    assert(dset);
    assert(size);

    
    if (0 == (H5F_INTENT(dset->oloc.file) & H5F_ACC_RDWR))
        HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "no write intent on file");

    
    if (H5D_COMPACT == dset->shared->layout.type)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "dataset has compact storage");
    if (H5D_CONTIGUOUS == dset->shared->layout.type && 0 == dset->shared->dcpl_cache.efl.nused)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "dataset has contiguous storage");

    
    if (H5D__check_filters(dset) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't apply filters");

    
    HDcompile_assert(sizeof(curr_dims) == sizeof(dset->shared->curr_dims));
    H5MM_memcpy(curr_dims, dset->shared->curr_dims, H5S_MAX_RANK * sizeof(curr_dims[0]));

    
    if ((changed = H5S_set_extent(dset->shared->space, size)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to modify size of dataspace");

    
    if (changed) {
        bool shrink        = false; 
        bool expand        = false; 
        bool update_chunks = false; 

        
        for (dim_idx = 0; dim_idx < dset->shared->ndims; dim_idx++) {
            
            if (size[dim_idx] < curr_dims[dim_idx])
                shrink = true;
            if (size[dim_idx] > curr_dims[dim_idx])
                expand = true;

            
            if (H5D_CHUNKED == dset->shared->layout.type && dset->shared->ndims > 1) {
                hsize_t scaled; 

                
                if (dset->shared->layout.u.chunk.dim[dim_idx] == 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "chunk size must be > 0, dim = %u ",
                                dim_idx);

                scaled = size[dim_idx] / dset->shared->layout.u.chunk.dim[dim_idx];

                
                if (scaled != dset->shared->cache.chunk.scaled_dims[dim_idx]) {
                    hsize_t scaled_power2up; 

                    
                    dset->shared->cache.chunk.scaled_dims[dim_idx] = scaled;

                    
                    if ((scaled > dset->shared->cache.chunk.nslots &&
                         dset->shared->cache.chunk.scaled_dims[dim_idx] <=
                             dset->shared->cache.chunk.nslots) ||
                        (scaled <= dset->shared->cache.chunk.nslots &&
                         dset->shared->cache.chunk.scaled_dims[dim_idx] > dset->shared->cache.chunk.nslots))
                        update_chunks = true;

                    if (!(scaled_power2up = H5VM_power2up(scaled)))
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to get the next power of 2");

                    
                    if (dset->shared->cache.chunk.scaled_power2up[dim_idx] != scaled_power2up) {
                        
                        dset->shared->cache.chunk.scaled_power2up[dim_idx] = scaled_power2up;
                        dset->shared->cache.chunk.scaled_encode_bits[dim_idx] =
                            H5VM_log2_gen(scaled_power2up);

                        
                        update_chunks = true;
                    } 
                }     
            }         

            
            dset->shared->curr_dims[dim_idx] = size[dim_idx];
        } 

        
        
        if (H5D_CHUNKED == dset->shared->layout.type) {
            
            if (H5D__chunk_set_info(dset) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "unable to update # of chunks");

            
            if (update_chunks)
                
                if (H5D__chunk_update_cache(dset) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "unable to update cached chunk indices");
        } 

        
        if (H5D_VIRTUAL == dset->shared->layout.type) {
            
            if (H5D_virtual_check_min_dims(dset) < 0)
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL,
                            "virtual dataset dimensions not large enough to contain all limited dimensions "
                            "in all selections");

            
            for (u = 0; u < dset->shared->layout.storage.u.virt.list_nused; u++) {
                
                if (H5S_set_extent(dset->shared->layout.storage.u.virt.list[u].source_dset.virtual_select,
                                   size) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to modify size of dataspace");
                dset->shared->layout.storage.u.virt.list[u].virtual_space_status = H5O_VIRTUAL_STATUS_CORRECT;

                
                for (v = 0; v < dset->shared->layout.storage.u.virt.list[u].sub_dset_nused; v++)
                    if (H5S_set_extent(dset->shared->layout.storage.u.virt.list[u].sub_dset[v].virtual_select,
                                       size) < 0)
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to modify size of dataspace");
            } 

            
            dset->shared->layout.storage.u.virt.init = false;
        } 

        
        if (expand && dset->shared->dcpl_cache.fill.alloc_time == H5D_ALLOC_TIME_EARLY)
            if (H5D__alloc_storage(dset, H5D_ALLOC_EXTEND, false, curr_dims) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to extend dataset storage");

        
        if (H5D_CHUNKED == dset->shared->layout.type) {
            if (shrink && ((*dset->shared->layout.ops->is_space_alloc)(&dset->shared->layout.storage) ||
                           (dset->shared->layout.ops->is_data_cached &&
                            (*dset->shared->layout.ops->is_data_cached)(dset->shared))))
                
                if (H5D__chunk_prune_by_extent(dset, curr_dims) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "unable to remove chunks");

            
            if (expand &&
                (dset->shared->layout.u.chunk.flags & H5O_LAYOUT_CHUNK_DONT_FILTER_PARTIAL_BOUND_CHUNKS) &&
                (dset->shared->dcpl_cache.pline.nused > 0))
                if (H5D__chunk_update_old_edge_chunks(dset, curr_dims) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "unable to do update old edge chunks");
        } 

        
        if (H5D__mark(dset, H5D_MARK_SPACE) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "unable to mark dataspace as dirty");
    } 

done:
    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5D__flush_sieve_buf(H5D_t *dataset)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dataset);

    
    if (dataset->shared->cache.contig.sieve_buf && dataset->shared->cache.contig.sieve_dirty) {
        assert(dataset->shared->layout.type !=
               H5D_COMPACT); 

        
        if (H5F_shared_block_write(
                H5F_SHARED(dataset->oloc.file), H5FD_MEM_DRAW, dataset->shared->cache.contig.sieve_loc,
                dataset->shared->cache.contig.sieve_size, dataset->shared->cache.contig.sieve_buf) < 0)
            HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL, "block write failed");

        
        dataset->shared->cache.contig.sieve_dirty = false;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__flush_real(H5D_t *dataset)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(dataset->oloc.addr)

    
    assert(dataset);
    assert(dataset->shared);

    
    if (!dataset->shared->closing)
        
        if (dataset->shared->layout.ops->flush && (dataset->shared->layout.ops->flush)(dataset) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTFLUSH, FAIL, "unable to flush raw data");

done:
    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5D__flush(H5D_t *dset, hid_t dset_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset);
    assert(dset->shared);

    
    if (H5F_HAS_FEATURE(dset->oloc.file, H5FD_FEAT_HAS_MPI))
        HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL, "H5Oflush isn't supported for parallel");

    
    if (H5D__flush_real(dset) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTFLUSH, FAIL, "unable to flush cached dataset info");

    
    if (H5O_flush_common(&dset->oloc, dset_id) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTFLUSH, FAIL, "unable to flush dataset and object flush callback");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__format_convert(H5D_t *dataset)
{
    H5D_chk_idx_info_t new_idx_info;                
    H5D_chk_idx_info_t idx_info;                    
    H5O_layout_t      *newlayout         = NULL;    
    bool               init_new_index    = false;   
    bool               delete_old_layout = false;   
    bool               add_new_layout    = false;   
    herr_t             ret_value         = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(dataset->oloc.addr)

    
    assert(dataset);

    switch (dataset->shared->layout.type) {
        case H5D_CHUNKED:
            assert(dataset->shared->layout.u.chunk.idx_type != H5D_CHUNK_IDX_BTREE);

            if (NULL == (newlayout = (H5O_layout_t *)H5MM_calloc(sizeof(H5O_layout_t))))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "unable to allocate buffer");

            
            idx_info.f      = dataset->oloc.file;
            idx_info.pline  = &dataset->shared->dcpl_cache.pline;
            idx_info.layout = &dataset->shared->layout;

            
            H5MM_memcpy(newlayout, &dataset->shared->layout, sizeof(H5O_layout_t));

            
            newlayout->version                        = H5O_LAYOUT_VERSION_3;
            newlayout->storage.u.chunk.idx_type       = H5D_CHUNK_IDX_BTREE;
            newlayout->storage.u.chunk.idx_addr       = HADDR_UNDEF;
            newlayout->storage.u.chunk.ops            = H5D_COPS_BTREE;
            newlayout->storage.u.chunk.u.btree.shared = NULL;

            
            new_idx_info.f      = dataset->oloc.file;
            new_idx_info.pline  = &dataset->shared->dcpl_cache.pline;
            new_idx_info.layout = newlayout;

            
            if (new_idx_info.layout->storage.u.chunk.ops->init &&
                (new_idx_info.layout->storage.u.chunk.ops->init)(&new_idx_info, dataset->shared->space,
                                                                 dataset->oloc.addr) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't initialize indexing information");
            init_new_index = true;

            
            if (H5_addr_defined(idx_info.layout->storage.u.chunk.idx_addr)) {

                
                if ((new_idx_info.layout->storage.u.chunk.ops->create)(&new_idx_info) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't create chunk index");

                
                if (H5D__chunk_format_convert(dataset, &idx_info, &new_idx_info) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_BADITER, FAIL, "unable to iterate/convert chunk index");
            } 

            
            if (H5O_msg_remove(&dataset->oloc, H5O_LAYOUT_ID, H5O_ALL, false) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTDELETE, FAIL, "unable to delete layout message");

            delete_old_layout = true;

            
            if (H5O_msg_create(&dataset->oloc, H5O_LAYOUT_ID, 0, H5O_UPDATE_TIME, newlayout) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to update layout header message");

            add_new_layout = true;

            
            if (idx_info.layout->storage.u.chunk.ops->dest &&
                (idx_info.layout->storage.u.chunk.ops->dest)(&idx_info) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "unable to release chunk index info");

            
            H5MM_memcpy(&dataset->shared->layout, newlayout, sizeof(H5O_layout_t));

            break;

        case H5D_CONTIGUOUS:
        case H5D_COMPACT:
            assert(dataset->shared->layout.version > H5O_LAYOUT_VERSION_DEFAULT);
            dataset->shared->layout.version = H5O_LAYOUT_VERSION_DEFAULT;
            if (H5O_msg_write(&(dataset->oloc), H5O_LAYOUT_ID, 0, H5O_UPDATE_TIME,
                              &(dataset->shared->layout)) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTINIT, FAIL, "unable to update layout message");
            break;

        case H5D_VIRTUAL:
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "virtual dataset layout not supported");

        case H5D_LAYOUT_ERROR:
        case H5D_NLAYOUTS:
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid dataset layout type");

        default:
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "unknown dataset layout type");
    } 

done:
    if (ret_value < 0 && dataset->shared->layout.type == H5D_CHUNKED) {
        
        if (add_new_layout)
            if (H5O_msg_remove(&dataset->oloc, H5O_LAYOUT_ID, H5O_ALL, false) < 0)
                HDONE_ERROR(H5E_SYM, H5E_CANTDELETE, FAIL, "unable to delete layout message");

        
        if (delete_old_layout)
            if (H5O_msg_create(&dataset->oloc, H5O_LAYOUT_ID, 0, H5O_UPDATE_TIME, &dataset->shared->layout) <
                0)
                HDONE_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to add layout header message");

        
        if (init_new_index) {
            if (H5_addr_defined(new_idx_info.layout->storage.u.chunk.idx_addr)) {
                
                if (!H5_addr_defined(dataset->oloc.addr))
                    HDONE_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "address undefined");

                
                if (H5AC_expunge_tag_type_metadata(dataset->oloc.file, dataset->oloc.addr, H5AC_BT_ID,
                                                   H5AC__NO_FLAGS_SET))
                    HDONE_ERROR(H5E_DATASET, H5E_CANTEXPUNGE, FAIL, "unable to expunge index metadata");
            } 

            
            if (new_idx_info.layout->storage.u.chunk.ops->dest &&
                (new_idx_info.layout->storage.u.chunk.ops->dest)(&new_idx_info) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "unable to release chunk index info");
        } 
    }     

    if (newlayout != NULL)
        newlayout = (H5O_layout_t *)H5MM_xfree(newlayout);

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5D__mark(const H5D_t *dataset, unsigned flags)
{
    H5O_t *oh        = NULL;    
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dataset);
    assert(!(flags & (unsigned)~(H5D_MARK_SPACE | H5D_MARK_LAYOUT)));

    
    if (flags) {
        unsigned update_flags = H5O_UPDATE_TIME; 

        
        if (NULL == (oh = H5O_pin(&dataset->oloc)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTPIN, FAIL, "unable to pin dataset object header");

        
        if (flags & H5D_MARK_LAYOUT) {
            if (H5D__layout_oh_write(dataset, oh, update_flags) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "unable to update layout info");

            
            update_flags = 0;
        } 

        
        if (flags & H5D_MARK_SPACE) {
            if (H5S_write(dataset->oloc.file, oh, update_flags, dataset->shared->space) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "unable to update file with new dataspace");

            
            update_flags = 0;
        } 

        
        assert(update_flags == 0);
    } 

done:
    
    if (oh != NULL)
        if (H5O_unpin(oh) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTUNPIN, FAIL, "unable to unpin dataset object header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5D__flush_all_cb(void *_dataset, hid_t H5_ATTR_UNUSED id, void *_udata)
{
    H5D_t *dataset   = (H5D_t *)_dataset; 
    H5F_t *f         = (H5F_t *)_udata;   
    int    ret_value = H5_ITER_CONT;      

    FUNC_ENTER_PACKAGE

    
    assert(dataset);
    assert(f);

    
    if (f == dataset->oloc.file)
        
        if (H5D__flush_real(dataset) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, H5_ITER_ERROR, "unable to flush cached dataset info");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D_flush_all(H5F_t *f)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);

    
    if (H5I_iterate(H5I_DATASET, H5D__flush_all_cb, f, false) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_BADITER, FAIL, "unable to flush cached dataset info");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5D_get_create_plist(const H5D_t *dset)
{
    H5P_genplist_t *dcpl_plist;        
    H5P_genplist_t *new_plist;         
    H5O_layout_t    copied_layout;     
    H5O_fill_t      copied_fill = {0}; 
    H5O_efl_t       copied_efl;        
    H5T_t          *dst_type    = NULL;
    H5T_t          *tmp_type    = NULL;
    hid_t           new_dcpl_id = FAIL;
    hid_t           ret_value   = H5I_INVALID_HID; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (NULL == (dcpl_plist = (H5P_genplist_t *)H5I_object(dset->shared->dcpl_id)))
        HGOTO_ERROR(H5E_DATASET, H5E_BADTYPE, FAIL, "can't get property list");

    
    if (H5D_flush_layout_to_dcpl(dset) < 0) {
        HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't flush layout to DCPL");
    }

    
    if ((new_dcpl_id = H5P_copy_plist(dcpl_plist, true)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "unable to copy the creation property list");
    if (NULL == (new_plist = (H5P_genplist_t *)H5I_object(new_dcpl_id)))
        HGOTO_ERROR(H5E_DATASET, H5E_BADTYPE, FAIL, "can't get property list");

    
    if (H5O_get_create_plist(&dset->oloc, new_plist) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get object creation info");

    
    if (H5P_peek(new_plist, H5D_CRT_LAYOUT_NAME, &copied_layout) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get layout");

    
    copied_layout.ops = NULL;
    switch (copied_layout.type) {
        case H5D_COMPACT:
            copied_layout.storage.u.compact.buf = H5MM_xfree(copied_layout.storage.u.compact.buf);
            memset(&copied_layout.storage.u.compact, 0, sizeof(copied_layout.storage.u.compact));
            break;

        case H5D_CONTIGUOUS:
            copied_layout.storage.u.contig.addr = HADDR_UNDEF;
            copied_layout.storage.u.contig.size = 0;
            break;

        case H5D_CHUNKED:
            
            copied_layout.u.chunk.size = 0;

            
            if (copied_layout.storage.u.chunk.ops)
                
                if (H5D_chunk_idx_reset(&copied_layout.storage.u.chunk, true) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL,
                                "unable to reset chunked storage index in dest");

            
            copied_layout.storage.u.chunk.ops = NULL;
            break;

        case H5D_VIRTUAL:
            copied_layout.storage.u.virt.serial_list_hobjid.addr = HADDR_UNDEF;
            copied_layout.storage.u.virt.serial_list_hobjid.idx  = 0;
            break;

        case H5D_LAYOUT_ERROR:
        case H5D_NLAYOUTS:
        default:
            assert(0 && "Unknown layout type!");
    } 

    
    if (H5P_poke(new_plist, H5D_CRT_LAYOUT_NAME, &copied_layout) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "unable to set layout");

    
    if (H5P_peek(new_plist, H5D_CRT_FILL_VALUE_NAME, &copied_fill) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get fill value");

    
    if (copied_fill.buf != NULL && copied_fill.type == NULL) {
        H5T_path_t *tpath; 

        
        if (NULL == (copied_fill.type = H5T_copy(dset->shared->type, H5T_COPY_TRANSIENT)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to copy dataset datatype for fill value");

        
        if (NULL == (tpath = H5T_path_find(dset->shared->type, copied_fill.type)))
            HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL,
                        "unable to convert between src and dest data types");

        
        if (!H5T_path_noop(tpath)) {
            uint8_t *bkg_buf = NULL; 
            size_t   bkg_size;       

            dst_type = copied_fill.type;
            if (H5T_detect_class(dst_type, H5T_VLEN, false) > 0 ||
                H5T_detect_class(dst_type, H5T_REFERENCE, false) > 0) {
                if (NULL == (tmp_type = H5T_copy(dst_type, H5T_COPY_TRANSIENT)))
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "unable to copy fill value datatype");
                dst_type = tmp_type;
            }

            
            bkg_size = MAX(H5T_GET_SIZE(copied_fill.type), H5T_GET_SIZE(dset->shared->type));
            if (H5T_path_bkg(tpath) && NULL == (bkg_buf = H5FL_BLK_CALLOC(type_conv, bkg_size)))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "memory allocation failed");

            
            if (H5T_convert(tpath, dset->shared->type, dst_type, (size_t)1, (size_t)0, (size_t)0,
                            copied_fill.buf, bkg_buf) < 0) {
                if (bkg_buf)
                    bkg_buf = H5FL_BLK_FREE(type_conv, bkg_buf);
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCONVERT, FAIL, "datatype conversion failed");
            } 

            
            if (bkg_buf)
                bkg_buf = H5FL_BLK_FREE(type_conv, bkg_buf);
        } 
    }     

    
    if (H5P_poke(new_plist, H5D_CRT_FILL_VALUE_NAME, &copied_fill) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "unable to set fill value");

    
    if (H5P_peek(new_plist, H5D_CRT_EXT_FILE_LIST_NAME, &copied_efl) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get external file list");

    
    if (copied_efl.slot) {
        unsigned u;

        copied_efl.heap_addr = HADDR_UNDEF;
        for (u = 0; u < copied_efl.nused; u++)
            copied_efl.slot[u].name_offset = 0;
    } 

    
    if (H5P_poke(new_plist, H5D_CRT_EXT_FILE_LIST_NAME, &copied_efl) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "unable to set external file list");

    
    ret_value = new_dcpl_id;

done:
    if (tmp_type && (H5T_close(tmp_type) < 0))
        HDONE_ERROR(H5E_DATASET, H5E_CANTCLOSEOBJ, FAIL, "unable to close temporary datatype");

    if (ret_value < 0) {
        if (new_dcpl_id > 0)
            if (H5I_dec_app_ref(new_dcpl_id) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CANTDEC, FAIL, "unable to close temporary object");

        if (copied_fill.type && (H5T_close_real(copied_fill.type) < 0))
            HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "Can't free temporary datatype");
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5D_get_access_plist(const H5D_t *dset)
{
    H5P_genplist_t    *old_plist;                    
    H5P_genplist_t    *new_plist;                    
    H5P_genplist_t    *def_dapl              = NULL; 
    H5D_append_flush_t def_append_flush_info = {0};  
    H5D_rdcc_t         def_chunk_info;               
    H5D_vds_view_t     def_vds_view;                 
    hsize_t            def_vds_gap;                  
    hid_t              new_dapl_id = FAIL;
    hid_t              ret_value   = FAIL;

    FUNC_ENTER_NOAPI_NOINIT

    
    if (NULL == (old_plist = (H5P_genplist_t *)H5I_object(dset->shared->dapl_id)))
        HGOTO_ERROR(H5E_DATASET, H5E_BADTYPE, FAIL, "can't get property list");
    if ((new_dapl_id = H5P_copy_plist(old_plist, true)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINIT, FAIL, "can't copy dataset access property list");
    if (NULL == (new_plist = (H5P_genplist_t *)H5I_object(new_dapl_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a property list");

    
    if (dset->shared->layout.type == H5D_CHUNKED) {
        if (H5P_set(new_plist, H5D_ACS_DATA_CACHE_NUM_SLOTS_NAME, &(dset->shared->cache.chunk.nslots)) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set data cache number of slots");
        if (H5P_set(new_plist, H5D_ACS_DATA_CACHE_BYTE_SIZE_NAME, &(dset->shared->cache.chunk.nbytes_max)) <
            0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set data cache byte size");
        if (H5P_set(new_plist, H5D_ACS_PREEMPT_READ_CHUNKS_NAME, &(dset->shared->cache.chunk.w0)) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set preempt read chunks");
        if (H5P_set(new_plist, H5D_ACS_APPEND_FLUSH_NAME, &dset->shared->append_flush) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set append flush property");
    }
    else {
        
        if (NULL == (def_dapl = (H5P_genplist_t *)H5I_object(H5P_LST_DATASET_ACCESS_ID_g)))
            HGOTO_ERROR(H5E_DATASET, H5E_BADTYPE, FAIL, "not a property list");

        
        if (H5P_get(def_dapl, H5D_ACS_DATA_CACHE_NUM_SLOTS_NAME, &def_chunk_info.nslots) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get data number of slots");
        if (H5P_set(new_plist, H5D_ACS_DATA_CACHE_NUM_SLOTS_NAME, &def_chunk_info.nslots) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set data cache number of slots");

        
        if (H5P_get(def_dapl, H5D_ACS_DATA_CACHE_BYTE_SIZE_NAME, &def_chunk_info.nbytes_max) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get data cache byte size");
        if (H5P_set(new_plist, H5D_ACS_DATA_CACHE_BYTE_SIZE_NAME, &def_chunk_info.nbytes_max) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set data cache byte size");

        
        if (H5P_get(def_dapl, H5D_ACS_PREEMPT_READ_CHUNKS_NAME, &def_chunk_info.w0) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get preempt read chunks");
        if (H5P_set(new_plist, H5D_ACS_PREEMPT_READ_CHUNKS_NAME, &def_chunk_info.w0) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set preempt read chunks");

        
        if (H5P_set(new_plist, H5D_ACS_APPEND_FLUSH_NAME, &def_append_flush_info) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set append flush property");
    } 

    
    if (dset->shared->layout.type == H5D_VIRTUAL) {
        if (H5P_set(new_plist, H5D_ACS_VDS_VIEW_NAME, &(dset->shared->layout.storage.u.virt.view)) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set VDS view");
        if (H5P_set(new_plist, H5D_ACS_VDS_PRINTF_GAP_NAME,
                    &(dset->shared->layout.storage.u.virt.printf_gap)) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set VDS Rprintf gap");
    }
    else {
        
        if (!def_dapl && NULL == (def_dapl = (H5P_genplist_t *)H5I_object(H5P_LST_DATASET_ACCESS_ID_g)))
            HGOTO_ERROR(H5E_DATASET, H5E_BADTYPE, FAIL, "not a property list");

        
        if (H5P_get(def_dapl, H5D_ACS_VDS_VIEW_NAME, &def_vds_view) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get VDS view");
        if (H5P_set(new_plist, H5D_ACS_VDS_VIEW_NAME, &def_vds_view) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set VDS view");

        
        if (H5P_get(def_dapl, H5D_ACS_VDS_PRINTF_GAP_NAME, &def_vds_gap) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get VDS Rprintf gap");
        if (H5P_set(new_plist, H5D_ACS_VDS_PRINTF_GAP_NAME, &def_vds_gap) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set VDS Rprintf gap");
    }

    
    if (H5P_set(new_plist, H5D_ACS_VDS_PREFIX_NAME, &(dset->shared->vds_prefix)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set vds prefix");

    
    if (H5P_set(new_plist, H5D_ACS_EFILE_PREFIX_NAME, &(dset->shared->extfile_prefix)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set external file prefix");

    
    ret_value = new_dapl_id;

done:
    if (ret_value < 0)
        if (new_dapl_id > 0)
            if (H5I_dec_app_ref(new_dapl_id) < 0)
                HDONE_ERROR(H5E_SYM, H5E_CANTDEC, FAIL, "can't free");

    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5D__get_space(const H5D_t *dset)
{
    H5S_t *space     = NULL;
    hid_t  ret_value = H5I_INVALID_HID;

    FUNC_ENTER_PACKAGE

    
    if (dset->shared->layout.type == H5D_VIRTUAL)
        if (H5D__virtual_set_extent_unlim(dset) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to update virtual dataset extent");

    
    if (NULL == (space = H5S_copy(dset->shared->space, false, true)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to get dataspace");

    
    if ((ret_value = H5I_register(H5I_DATASPACE, space, true)) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTREGISTER, FAIL, "unable to register dataspace");

done:
    if (ret_value < 0)
        if (space != NULL)
            if (H5S_close(space) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "unable to release dataspace");

    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5D__get_type(const H5D_t *dset)
{
    H5T_t *dt        = NULL;
    hid_t  ret_value = FAIL;

    FUNC_ENTER_PACKAGE

    
    if (H5T_patch_file(dset->shared->type, dset->oloc.file) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to patch datatype's file pointer");

    
    if (NULL == (dt = H5T_copy_reopen(dset->shared->type)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to copy datatype");

    
    if (H5T_set_loc(dt, NULL, H5T_LOC_MEMORY) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "invalid datatype location");

    
    if (H5T_lock(dt, false) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to lock transient datatype");

    
    if (H5T_is_named(dt)) {
        
        if ((ret_value = H5VL_wrap_register(H5I_DATATYPE, dt, true)) < 0)
            HGOTO_ERROR(H5E_ID, H5E_CANTREGISTER, FAIL, "unable to register datatype");
    } 
    else if ((ret_value = H5I_register(H5I_DATATYPE, dt, true)) < 0)
        HGOTO_ERROR(H5E_ID, H5E_CANTREGISTER, FAIL, "unable to register datatype");

done:
    if (ret_value < 0)
        if (dt && H5T_close(dt) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "unable to release datatype");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__refresh(H5D_t *dset, hid_t dset_id)
{
    H5D_virtual_held_file_t *head            = NULL;    
    bool                     virt_dsets_held = false;   
    herr_t                   ret_value       = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dset);
    assert(dset->shared);

    
    if (dset->shared->layout.type == H5D_VIRTUAL) {
        
        if (H5D__virtual_hold_source_dset_files(dset, &head) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINC, FAIL, "unable to hold VDS source files open");
        virt_dsets_held = true;

        
        if (H5D__virtual_refresh_source_dsets(dset) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTFLUSH, FAIL, "unable to refresh VDS source datasets");
    } 

    
    if ((H5O_refresh_metadata(&dset->oloc, dset_id)) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTFLUSH, FAIL, "unable to refresh dataset");

done:
    
    if (virt_dsets_held)
        if (H5D__virtual_release_source_dset_files(head) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTDEC, FAIL, "can't release VDS source files held open");

    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5D_get_dcpl_id(const H5D_obj_create_t *d)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(d);

    FUNC_LEAVE_NOAPI(d->dcpl_id);
} 

herr_t
H5D_flush_layout_to_dcpl(const H5D_t *dset)
{
    herr_t          ret_value      = SUCCEED;
    H5P_genplist_t *dcpl           = NULL;
    bool            ndims_modified = false;

    FUNC_ENTER_NOAPI(FAIL)

    if ((dcpl = H5P_object_verify(dset->shared->dcpl_id, H5P_DATASET_CREATE, true)) == NULL) {
        HGOTO_ERROR(H5E_DATASET, H5E_BADID, FAIL, "invalid DCPL ID");
    }

    if (!dset->shared->layout_copied_to_dcpl) {
        
        if (H5P_is_default_plist(dset->shared->dcpl_id)) {
            HGOTO_DONE(ret_value);
        }

        
        if (H5D_CHUNKED == dset->shared->layout.type) {
            dset->shared->layout.u.chunk.ndims--;
            ndims_modified = true;
        }

        
        if (H5P_set(dcpl, H5D_CRT_LAYOUT_NAME, (void *)&dset->shared->layout) < 0) {
            HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "can't set layout property");
        }
    }

done:
    if (ret_value == SUCCEED)
        dset->shared->layout_copied_to_dcpl = true;

    if (ndims_modified)
        dset->shared->layout.u.chunk.ndims++;

    FUNC_LEAVE_NOAPI(ret_value);
} 
