/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Tmodule.h" 
#define H5F_FRIEND     
#define H5R_FRIEND     

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Eprivate.h"  
#include "H5Iprivate.h"  
#include "H5Fpkg.h"      
#include "H5MMprivate.h" 
#include "H5Rpkg.h"      
#include "H5Tpkg.h"      

#include "H5VLnative_private.h" 

#define H5T_REF_MEM_SIZE         (H5R_REF_BUF_SIZE)
#define H5T_REF_OBJ_MEM_SIZE     (H5R_OBJ_REF_BUF_SIZE)
#define H5T_REF_DSETREG_MEM_SIZE (H5R_DSET_REG_REF_BUF_SIZE)

#define H5T_REF_OBJ_DISK_SIZE(f)     (H5F_SIZEOF_ADDR(f))
#define H5T_REF_DSETREG_DISK_SIZE(f) (H5HG_HEAP_ID_SIZE(f))

// #define H5T_REF_DEBUG
#ifdef H5T_REF_DEBUG
#define H5T_REF_LOG_DEBUG(...)                                                                               \
    do {                                                                                                     \
        Rfprintf(Rstdout, " # %s(): ", __func__);                                                              \
        Rfprintf(Rstdout, __VA_ARGS__);                                                                        \
        Rfprintf(Rstdout, "\n");                                                                               \
        fflush(Rstdout);                                                                                      \
    } while (0)
#else
#define H5T_REF_LOG_DEBUG(...)                                                                               \
    do {                                                                                                     \
    } while (0)
#endif

struct H5Tref_dsetreg {
    H5O_token_t token; 
    H5S_t      *space; 
};

static herr_t H5T__ref_mem_isnull(const H5VL_object_t *src_file, const void *src_buf, bool *isnull);
static herr_t H5T__ref_mem_setnull(H5VL_object_t *dst_file, void *dst_buf, void *bg_buf);
static size_t H5T__ref_mem_getsize(H5VL_object_t *src_file, const void *src_buf, size_t src_size,
                                   H5VL_object_t *dst_file, bool *dst_copy);
static herr_t H5T__ref_mem_read(H5VL_object_t *src_file, const void *src_buf, size_t src_size,
                                H5VL_object_t *dst_file, void *dst_buf, size_t dst_size);
static herr_t H5T__ref_mem_write(H5VL_object_t *src_file, const void *src_buf, size_t src_size,
                                 H5R_type_t src_type, H5VL_object_t *dst_file, void *dst_buf, size_t dst_size,
                                 void *bg_buf);

static herr_t H5T__ref_disk_isnull(const H5VL_object_t *src_file, const void *src_buf, bool *isnull);
static herr_t H5T__ref_disk_setnull(H5VL_object_t *dst_file, void *dst_buf, void *bg_buf);
static size_t H5T__ref_disk_getsize(H5VL_object_t *src_file, const void *src_buf, size_t src_size,
                                    H5VL_object_t *dst_file, bool *dst_copy);
static herr_t H5T__ref_disk_read(H5VL_object_t *src_file, const void *src_buf, size_t src_size,
                                 H5VL_object_t *dst_file, void *dst_buf, size_t dst_size);
static herr_t H5T__ref_disk_write(H5VL_object_t *src_file, const void *src_buf, size_t src_size,
                                  H5R_type_t src_type, H5VL_object_t *dst_file, void *dst_buf,
                                  size_t dst_size, void *bg_buf);

static herr_t H5T__ref_obj_disk_isnull(const H5VL_object_t *src_file, const void *src_buf, bool *isnull);
static size_t H5T__ref_obj_disk_getsize(H5VL_object_t *src_file, const void *src_buf, size_t src_size,
                                        H5VL_object_t *dst_file, bool *dst_copy);
static herr_t H5T__ref_obj_disk_read(H5VL_object_t *src_file, const void *src_buf, size_t src_size,
                                     H5VL_object_t *dst_file, void *dst_buf, size_t dst_size);

static herr_t H5T__ref_dsetreg_disk_isnull(const H5VL_object_t *src_file, const void *src_buf, bool *isnull);
static size_t H5T__ref_dsetreg_disk_getsize(H5VL_object_t *src_file, const void *src_buf, size_t src_size,
                                            H5VL_object_t *dst_file, bool *dst_copy);
static herr_t H5T__ref_dsetreg_disk_read(H5VL_object_t *src_file, const void *src_buf, size_t src_size,
                                         H5VL_object_t *dst_file, void *dst_buf, size_t dst_size);

static const H5T_ref_class_t H5T_ref_mem_g = {
    H5T__ref_mem_isnull,  
    H5T__ref_mem_setnull, 
    H5T__ref_mem_getsize, 
    H5T__ref_mem_read,    
    H5T__ref_mem_write    
};

static const H5T_ref_class_t H5T_ref_disk_g = {
    H5T__ref_disk_isnull,  
    H5T__ref_disk_setnull, 
    H5T__ref_disk_getsize, 
    H5T__ref_disk_read,    
    H5T__ref_disk_write    
};

static const H5T_ref_class_t H5T_ref_obj_disk_g = {
    H5T__ref_obj_disk_isnull,  
    NULL,                      
    H5T__ref_obj_disk_getsize, 
    H5T__ref_obj_disk_read,    
    NULL                       
};

static const H5T_ref_class_t H5T_ref_dsetreg_disk_g = {
    H5T__ref_dsetreg_disk_isnull,  
    NULL,                          
    H5T__ref_dsetreg_disk_getsize, 
    H5T__ref_dsetreg_disk_read,    
    NULL                           
};

htri_t
H5T__ref_set_loc(H5T_t *dt, H5VL_object_t *file, H5T_loc_t loc)
{
    htri_t ret_value = false; 

    FUNC_ENTER_PACKAGE
    H5T_REF_LOG_DEBUG("loc=%d", (int)loc);

    assert(dt);
    
    assert(loc >= H5T_LOC_BADLOC && loc < H5T_LOC_MAXLOC);

    
    if (loc == dt->shared->u.atomic.u.r.loc && file == dt->shared->u.atomic.u.r.file)
        HGOTO_DONE(false);

    switch (loc) {
        case H5T_LOC_MEMORY: 

            

            
            dt->shared->u.atomic.u.r.loc = H5T_LOC_MEMORY;

            
            if (dt->shared->owned_vol_obj) {
                if (H5VL_free_object(dt->shared->owned_vol_obj) < 0)
                    HGOTO_ERROR(H5E_REFERENCE, H5E_CANTCLOSEOBJ, FAIL, "unable to close owned VOL object");
                dt->shared->owned_vol_obj = NULL;
            } 

            
            dt->shared->u.atomic.u.r.file = file; 

            if (dt->shared->u.atomic.u.r.opaque) {
                
                dt->shared->size          = H5T_REF_MEM_SIZE;
                dt->shared->u.atomic.prec = 8 * dt->shared->size;

                
                dt->shared->u.atomic.u.r.cls = &H5T_ref_mem_g;
            } 
            else if (dt->shared->u.atomic.u.r.rtype == H5R_OBJECT1) {
                
                dt->shared->size          = H5T_REF_OBJ_MEM_SIZE;
                dt->shared->u.atomic.prec = 8 * dt->shared->size;

                
                dt->shared->u.atomic.u.r.cls = NULL;
            } 
            else if (dt->shared->u.atomic.u.r.rtype == H5R_DATASET_REGION1) {
                
                dt->shared->size          = H5T_REF_DSETREG_MEM_SIZE;
                dt->shared->u.atomic.prec = 8 * dt->shared->size;

                
                dt->shared->u.atomic.u.r.cls = NULL;
            } 
            else
                HGOTO_ERROR(H5E_DATATYPE, H5E_BADTYPE, FAIL, "invalid location");
            break;

        case H5T_LOC_DISK: 
            assert(file);

            
            dt->shared->u.atomic.u.r.loc = H5T_LOC_DISK;

            
            dt->shared->u.atomic.u.r.file = file;

            
            if (H5T_own_vol_obj(dt, file) < 0)
                HGOTO_ERROR(H5E_REFERENCE, H5E_CANTINIT, FAIL, "can't give ownership of VOL object");

            if (dt->shared->u.atomic.u.r.rtype == H5R_OBJECT1) {
                H5F_t *f;

#ifndef NDEBUG
                {
                    bool is_native = false; 

                    
                    if (H5VL_object_is_native(file, &is_native) < 0)
                        HGOTO_ERROR(H5E_REFERENCE, H5E_CANTGET, FAIL,
                                    "can't query if file uses native VOL connector");

                    
                    assert(is_native);
                }
#endif 

                
                if (NULL == (f = (H5F_t *)H5VL_object_data(file)))
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid VOL object");

                
                dt->shared->size          = H5T_REF_OBJ_DISK_SIZE(f);
                dt->shared->u.atomic.prec = 8 * dt->shared->size;

                
                dt->shared->u.atomic.u.r.cls = &H5T_ref_obj_disk_g;
            } 
            else if (dt->shared->u.atomic.u.r.rtype == H5R_DATASET_REGION1) {
                H5F_t *f;

#ifndef NDEBUG
                {
                    bool is_native = false; 

                    
                    if (H5VL_object_is_native(file, &is_native) < 0)
                        HGOTO_ERROR(H5E_REFERENCE, H5E_CANTGET, FAIL,
                                    "can't query if file uses native VOL connector");

                    
                    assert(is_native);
                }
#endif 

                
                if (NULL == (f = (H5F_t *)H5VL_object_data(file)))
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid VOL object");

                
                dt->shared->size          = H5T_REF_DSETREG_DISK_SIZE(f);
                dt->shared->u.atomic.prec = 8 * dt->shared->size;

                
                dt->shared->u.atomic.u.r.cls = &H5T_ref_dsetreg_disk_g;
            } 
            else {
                H5VL_file_cont_info_t cont_info = {H5VL_CONTAINER_INFO_VERSION, 0, 0, 0};
                H5VL_file_get_args_t  vol_cb_args; 
                size_t                ref_encode_size;
                H5R_ref_priv_t        fixed_ref;

                
                vol_cb_args.op_type                 = H5VL_FILE_GET_CONT_INFO;
                vol_cb_args.args.get_cont_info.info = &cont_info;

                
                if (H5VL_file_get(file, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, H5_REQUEST_NULL) < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, FAIL, "unable to get container info");

                
                memset(&fixed_ref, 0, sizeof(fixed_ref));
                fixed_ref.type       = (int8_t)H5R_OBJECT2;
                fixed_ref.token_size = (uint8_t)cont_info.token_size;
                if (H5R__encode(NULL, &fixed_ref, NULL, &ref_encode_size, 0) < 0)
                    HGOTO_ERROR(H5E_REFERENCE, H5E_CANTGET, FAIL, "can't get encode size");

                
                dt->shared->size =
                    MAX(sizeof(uint32_t) + H5R_ENCODE_HEADER_SIZE + cont_info.blob_id_size, ref_encode_size);
                dt->shared->u.atomic.prec = 8 * dt->shared->size;

                
                dt->shared->u.atomic.u.r.cls = &H5T_ref_disk_g;
            }
            break;

        case H5T_LOC_BADLOC:
            
            dt->shared->u.atomic.u.r.loc = H5T_LOC_BADLOC;

            
            dt->shared->u.atomic.u.r.file = NULL;

            
            dt->shared->u.atomic.u.r.cls = NULL;

            break;

        case H5T_LOC_MAXLOC: 
        default:
            HGOTO_ERROR(H5E_DATATYPE, H5E_BADRANGE, FAIL, "invalid reference datatype location");
    } 

    
    ret_value = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__ref_mem_isnull(const H5VL_object_t H5_ATTR_UNUSED *src_file, const void *src_buf, bool *isnull)
{
    const unsigned char zeros[H5T_REF_MEM_SIZE] = {0};
    herr_t              ret_value               = SUCCEED;

    FUNC_ENTER_PACKAGE_NOERR
    H5T_REF_LOG_DEBUG("");

    
    assert(src_buf);
    assert(isnull);

    *isnull = (0 == memcmp(src_buf, zeros, H5T_REF_MEM_SIZE)) ? true : false;

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__ref_mem_setnull(H5VL_object_t H5_ATTR_UNUSED *dst_file, void *dst_buf, H5_ATTR_UNUSED void *bg_buf)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE_NOERR
    H5T_REF_LOG_DEBUG("");

    memset(dst_buf, 0, H5T_REF_MEM_SIZE);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5T__ref_mem_getsize(H5VL_object_t H5_ATTR_UNUSED *src_file, const void *src_buf,
                     size_t H5_ATTR_UNUSED src_size, H5VL_object_t *dst_file, bool *dst_copy)
{
    H5VL_object_t        *vol_obj = NULL; 
    const H5R_ref_priv_t *src_ref = (const H5R_ref_priv_t *)src_buf;
    char                 *file_name_buf_dyn =
        NULL; 
    unsigned flags     = 0; 
    size_t   ret_value = 0; 

    FUNC_ENTER_PACKAGE
    H5T_REF_LOG_DEBUG("");

    
    assert(src_buf);
    assert(src_size == H5T_REF_MEM_SIZE);

    if (NULL != dst_file) {
        bool files_equal = true; 

        
        if (NULL == (vol_obj = H5VL_vol_object(src_ref->loc_id)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, 0, "invalid location identifier");

        
        if (H5VL_file_is_same(vol_obj, dst_file, &files_equal) < 0)
            HGOTO_ERROR(H5E_REFERENCE, H5E_CANTCOMPARE, 0, "can't check if files are equal");
        flags |= !files_equal ? H5R_IS_EXTERNAL : 0;
    }

    
    if (flags || !src_ref->encode_size) {
        H5VL_file_get_args_t vol_cb_args;               
        char                *file_name = NULL;          
        char                 file_name_buf_static[256]; 
        size_t               file_name_len = 0;         

        
        if (src_ref->type == (int8_t)H5R_DATASET_REGION2) {
            bool is_native = false; 

            
            if (H5VL_object_is_native(dst_file, &is_native) < 0)
                HGOTO_ERROR(H5E_REFERENCE, H5E_CANTGET, 0, "can't query if file uses native VOL connector");

            
            if (is_native) {
                H5F_t *dst_f; 

                if (NULL == (dst_f = (H5F_t *)H5VL_object_data(dst_file)))
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, 0, "invalid VOL object");
                H5CX_set_libver_bounds(dst_f);
            } 
            else
                H5CX_set_libver_bounds(NULL);
        } 

        
        vol_cb_args.op_type                     = H5VL_FILE_GET_NAME;
        vol_cb_args.args.get_name.type          = H5I_FILE;
        vol_cb_args.args.get_name.buf_size      = sizeof(file_name_buf_static);
        vol_cb_args.args.get_name.buf           = file_name_buf_static;
        vol_cb_args.args.get_name.file_name_len = &file_name_len;

        
        if (H5VL_file_get(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, NULL) < 0)
            HGOTO_ERROR(H5E_REFERENCE, H5E_CANTGET, 0, "can't get file name");

        
        if (file_name_len >= sizeof(file_name_buf_static)) {
            
            if (NULL == (file_name_buf_dyn = H5MM_malloc(file_name_len + 1)))
                HGOTO_ERROR(H5E_REFERENCE, H5E_CANTALLOC, 0, "can't allocate space for file name");

            
            vol_cb_args.args.get_name.buf_size = file_name_len + 1;
            vol_cb_args.args.get_name.buf      = file_name_buf_dyn;

            
            if (H5VL_file_get(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, NULL) < 0)
                HGOTO_ERROR(H5E_REFERENCE, H5E_CANTGET, 0, "can't get file name");

            file_name = file_name_buf_dyn;
        } 
        else
            file_name = file_name_buf_static;

        
        if (H5R__encode(file_name, src_ref, NULL, &ret_value, flags) < 0)
            HGOTO_ERROR(H5E_REFERENCE, H5E_CANTENCODE, 0, "unable to determine encoding size");
    } 
    else {
        
        if (src_ref->type == (int8_t)H5R_OBJECT2)
            *dst_copy = true;

        
        ret_value = src_ref->encode_size;
    } 

done:
    H5MM_xfree(file_name_buf_dyn);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__ref_mem_read(H5VL_object_t H5_ATTR_UNUSED *src_file, const void *src_buf, size_t H5_ATTR_UNUSED src_size,
                  H5VL_object_t *dst_file, void *dst_buf, size_t dst_size)
{
    H5VL_object_t        *vol_obj; 
    const H5R_ref_priv_t *src_ref     = (const H5R_ref_priv_t *)src_buf;
    bool                  files_equal = true; 
    char                 *file_name   = NULL; 
    char                  file_name_buf_static[256] = {'\0'}; 
    char                 *file_name_buf_dyn =
        NULL; 
    unsigned flags     = 0;       
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE
    H5T_REF_LOG_DEBUG("");

    
    assert(src_buf);
    assert(src_size == H5T_REF_MEM_SIZE);
    assert(dst_buf);
    assert(dst_size);

    
    if (NULL == dst_file) {
        H5MM_memcpy(dst_buf, src_buf, dst_size);
        HGOTO_DONE(ret_value);
    }

    
    if (NULL == (vol_obj = H5VL_vol_object(src_ref->loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, 0, "invalid location identifier");

    
    if (H5VL_file_is_same(vol_obj, dst_file, &files_equal) < 0)
        HGOTO_ERROR(H5E_REFERENCE, H5E_CANTCOMPARE, FAIL, "can't check if files are equal");
    flags |= !files_equal ? H5R_IS_EXTERNAL : 0;

    
    if (src_ref->type == (int8_t)H5R_DATASET_REGION2) {
        bool is_native = false; 

        
        if (H5VL_object_is_native(dst_file, &is_native) < 0)
            HGOTO_ERROR(H5E_REFERENCE, H5E_CANTGET, 0, "can't query if file uses native VOL connector");

        
        if (is_native) {
            H5F_t *dst_f;

            if (NULL == (dst_f = (H5F_t *)H5VL_object_data(dst_file)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, 0, "invalid VOL object");
            H5CX_set_libver_bounds(dst_f);
        } 
        else
            H5CX_set_libver_bounds(NULL);
    } 

    
    if (flags) {
        H5VL_file_get_args_t vol_cb_args;       
        size_t               file_name_len = 0; 

        
        vol_cb_args.op_type                     = H5VL_FILE_GET_NAME;
        vol_cb_args.args.get_name.type          = H5I_FILE;
        vol_cb_args.args.get_name.buf_size      = sizeof(file_name_buf_static);
        vol_cb_args.args.get_name.buf           = file_name_buf_static;
        vol_cb_args.args.get_name.file_name_len = &file_name_len;

        
        if (H5VL_file_get(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, NULL) < 0)
            HGOTO_ERROR(H5E_REFERENCE, H5E_CANTGET, 0, "can't get file name");

        
        if (file_name_len >= sizeof(file_name_buf_static)) {
            
            if (NULL == (file_name_buf_dyn = H5MM_malloc(file_name_len + 1)))
                HGOTO_ERROR(H5E_REFERENCE, H5E_CANTALLOC, 0, "can't allocate space for file name");

            
            vol_cb_args.args.get_name.buf_size = file_name_len + 1;
            vol_cb_args.args.get_name.buf      = file_name_buf_dyn;

            
            if (H5VL_file_get(vol_obj, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, NULL) < 0)
                HGOTO_ERROR(H5E_REFERENCE, H5E_CANTGET, 0, "can't get file name");

            file_name = file_name_buf_dyn;
        } 
        else
            file_name = file_name_buf_static;
    } 

    
    if (H5R__encode(file_name, src_ref, (unsigned char *)dst_buf, &dst_size, flags) < 0)
        HGOTO_ERROR(H5E_REFERENCE, H5E_CANTENCODE, FAIL, "Cannot encode reference");

done:
    H5MM_xfree(file_name_buf_dyn);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__ref_mem_write(H5VL_object_t *src_file, const void *src_buf, size_t src_size, H5R_type_t src_type,
                   H5VL_object_t H5_ATTR_UNUSED *dst_file, void *dst_buf,
                   size_t H5_ATTR_NDEBUG_UNUSED dst_size, void H5_ATTR_UNUSED *bg_buf)
{
    H5F_t          *src_f   = NULL;
    hid_t           file_id = H5I_INVALID_HID;
    H5R_ref_priv_t *dst_ref = (H5R_ref_priv_t *)dst_buf;
    H5R_ref_priv_t  tmp_ref; 
    herr_t          ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE
    H5T_REF_LOG_DEBUG("");

    
    assert(src_buf);
    assert(src_size);
    assert(dst_buf);
    assert(dst_size == H5T_REF_MEM_SIZE);
    HDcompile_assert(sizeof(*dst_ref) == sizeof(tmp_ref));

    
    if (NULL == src_file) {
        H5MM_memcpy(dst_buf, src_buf, src_size);
        HGOTO_DONE(ret_value);
    }

#ifndef NDEBUG
    if ((src_type == H5R_OBJECT1) || (src_type == H5R_DATASET_REGION1)) {
        bool is_native = false; 

        
        if (H5VL_object_is_native(src_file, &is_native) < 0)
            HGOTO_ERROR(H5E_REFERENCE, H5E_CANTGET, FAIL, "can't query if file uses native VOL connector");

        
        assert(is_native);
    }
#endif 

    
    if (NULL == (src_f = (H5F_t *)H5VL_object_data(src_file)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid VOL object");

    
    memset(&tmp_ref, 0, sizeof(tmp_ref));

    switch (src_type) {
        case H5R_OBJECT1: {
            size_t token_size = H5F_SIZEOF_ADDR(src_f);

            if (H5R__create_object((const H5O_token_t *)src_buf, token_size, &tmp_ref) < 0)
                HGOTO_ERROR(H5E_REFERENCE, H5E_CANTCREATE, FAIL, "unable to create object reference");
        } break;

        case H5R_DATASET_REGION1: {
            const struct H5Tref_dsetreg *src_reg    = (const struct H5Tref_dsetreg *)src_buf;
            size_t                       token_size = H5F_SIZEOF_ADDR(src_f);

            if (H5R__create_region(&src_reg->token, token_size, src_reg->space, &tmp_ref) < 0)
                HGOTO_ERROR(H5E_REFERENCE, H5E_CANTCREATE, FAIL, "unable to create region reference");

            
            if (H5S_close(src_reg->space) < 0)
                HGOTO_ERROR(H5E_REFERENCE, H5E_CANTFREE, FAIL, "Cannot close dataspace");
        } break;

        case H5R_DATASET_REGION2:
            
            H5CX_set_libver_bounds(src_f);
            
            H5_ATTR_FALLTHROUGH
        case H5R_OBJECT2:
        case H5R_ATTR:
            
            if (H5R__decode((const unsigned char *)src_buf, &src_size, &tmp_ref) < 0)
                HGOTO_ERROR(H5E_REFERENCE, H5E_CANTDECODE, FAIL, "Cannot decode reference");
            break;

        case H5R_BADTYPE:
        case H5R_MAXTYPE:
        default:
            assert("unknown reference type" && 0);
            HGOTO_ERROR(H5E_REFERENCE, H5E_UNSUPPORTED, FAIL, "internal error (unknown reference type)");
    } 

    
    if (NULL == H5R_REF_FILENAME(&tmp_ref)) {
        
        if ((file_id = H5F_get_file_id(src_file, H5I_FILE, false)) < 0)
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file or file object");

        
        if (H5R__set_loc_id(&tmp_ref, file_id, true, true) < 0)
            HGOTO_ERROR(H5E_REFERENCE, H5E_CANTSET, FAIL, "unable to attach location id to reference");
    } 

    
    H5MM_memcpy(dst_ref, &tmp_ref, sizeof(tmp_ref));

done:
    if ((file_id != H5I_INVALID_HID) && (H5I_dec_ref(file_id) < 0))
        HDONE_ERROR(H5E_REFERENCE, H5E_CANTDEC, FAIL, "unable to decrement refcount on location id");
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5_WARN_CAST_AWAY_CONST_OFF
static herr_t
H5T__ref_disk_isnull(const H5VL_object_t *src_file, const void *src_buf, bool *isnull)
{
    const uint8_t *p = (const uint8_t *)src_buf;
    H5R_type_t     ref_type;
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE
    H5T_REF_LOG_DEBUG("");

    
    assert(src_file);
    assert(src_buf);
    assert(isnull);

    
    ref_type = (H5R_type_t)*p++;
    if (ref_type) {
        
        *isnull = false;
    }
    else {
        H5VL_blob_specific_args_t vol_cb_args; 

        
        p = (const uint8_t *)src_buf + H5R_ENCODE_HEADER_SIZE + sizeof(uint32_t);

        
        vol_cb_args.op_type             = H5VL_BLOB_ISNULL;
        vol_cb_args.args.is_null.isnull = isnull;

        
        if (H5VL_blob_specific(src_file, (void *)p, &vol_cb_args) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, FAIL, "unable to check if a blob ID is 'nil'");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
H5_WARN_CAST_AWAY_CONST_ON

static herr_t
H5T__ref_disk_setnull(H5VL_object_t *dst_file, void *dst_buf, void *bg_buf)
{
    H5VL_blob_specific_args_t vol_cb_args; 
    uint8_t                  *q         = (uint8_t *)dst_buf;
    uint8_t                  *p_bg      = (uint8_t *)bg_buf;
    herr_t                    ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE
    H5T_REF_LOG_DEBUG("");

    assert(dst_file);
    assert(dst_buf);

    
    if (p_bg) {
        
        p_bg += (sizeof(uint32_t) + H5R_ENCODE_HEADER_SIZE);

        
        vol_cb_args.op_type = H5VL_BLOB_DELETE;

        
        if (H5VL_blob_specific(dst_file, (void *)p_bg, &vol_cb_args) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREMOVE, FAIL, "unable to delete blob");
    } 

    
    memset(q, 0, H5R_ENCODE_HEADER_SIZE);
    q += H5R_ENCODE_HEADER_SIZE;

    
    UINT32ENCODE(q, 0);

    
    vol_cb_args.op_type = H5VL_BLOB_SETNULL;

    
    if (H5VL_blob_specific(dst_file, q, &vol_cb_args) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTSET, FAIL, "unable to set a blob ID to 'nil'");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5T__ref_disk_getsize(H5VL_object_t H5_ATTR_UNUSED *src_file, const void *src_buf, size_t src_size,
                      H5VL_object_t H5_ATTR_UNUSED *dst_file, bool *dst_copy)
{
    const uint8_t *p = (const uint8_t *)src_buf;
    unsigned       flags;
    H5R_type_t     ref_type;
    size_t         ret_value = 0;

    FUNC_ENTER_PACKAGE
    H5T_REF_LOG_DEBUG("");

    assert(src_buf);

    
    ref_type = (H5R_type_t)*p++;
    if (ref_type <= H5R_BADTYPE || ref_type >= H5R_MAXTYPE)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, 0, "invalid reference type");

    
    flags = (unsigned)*p++;

    if (!(flags & H5R_IS_EXTERNAL) && (ref_type == H5R_OBJECT2)) {
        
        *dst_copy = true;

        ret_value = src_size;
    } 
    else {
        
        UINT32DECODE(p, ret_value);

        
        ret_value += H5R_ENCODE_HEADER_SIZE;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__ref_disk_read(H5VL_object_t *src_file, const void *src_buf, size_t H5_ATTR_NDEBUG_UNUSED src_size,
                   H5VL_object_t H5_ATTR_UNUSED *dst_file, void *dst_buf, size_t dst_size)
{
    const uint8_t *p         = (const uint8_t *)src_buf;
    uint8_t       *q         = (uint8_t *)dst_buf;
    size_t         blob_size = dst_size;
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE
    H5T_REF_LOG_DEBUG("");

    assert(src_file);
    assert(src_buf);
    assert(dst_buf);
    assert(dst_size);

    
    H5MM_memcpy(q, p, H5R_ENCODE_HEADER_SIZE);
    p += H5R_ENCODE_HEADER_SIZE;
    q += H5R_ENCODE_HEADER_SIZE;
    blob_size -= H5R_ENCODE_HEADER_SIZE;

    
    p += sizeof(uint32_t);
    assert(src_size > (H5R_ENCODE_HEADER_SIZE + sizeof(uint32_t)));

    
    if (H5VL_blob_get(src_file, p, q, blob_size, NULL) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, FAIL, "unable to get blob");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__ref_disk_write(H5VL_object_t H5_ATTR_UNUSED *src_file, const void *src_buf, size_t src_size,
                    H5R_type_t H5_ATTR_UNUSED src_type, H5VL_object_t *dst_file, void *dst_buf,
                    size_t H5_ATTR_NDEBUG_UNUSED dst_size, void *bg_buf)
{
    const uint8_t *p         = (const uint8_t *)src_buf;
    uint8_t       *q         = (uint8_t *)dst_buf;
    uint8_t       *p_bg      = (uint8_t *)bg_buf;
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE
    H5T_REF_LOG_DEBUG("");

    assert(src_buf);
    assert(src_size);
    assert(dst_file);
    assert(dst_buf);

    
    if (p_bg) {
        H5VL_blob_specific_args_t vol_cb_args; 

        
        p_bg += (sizeof(uint32_t) + H5R_ENCODE_HEADER_SIZE);

#ifndef NDEBUG
        size_t p_buf_size_left = dst_size;
        assert(p_buf_size_left > (sizeof(uint32_t) + H5R_ENCODE_HEADER_SIZE));
#endif

        
        vol_cb_args.op_type = H5VL_BLOB_DELETE;

        
        if (H5VL_blob_specific(dst_file, (void *)p_bg, &vol_cb_args) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREMOVE, FAIL, "unable to delete blob");
    } 

    
    H5MM_memcpy(q, p, H5R_ENCODE_HEADER_SIZE);
    p += H5R_ENCODE_HEADER_SIZE;
    q += H5R_ENCODE_HEADER_SIZE;
    src_size -= H5R_ENCODE_HEADER_SIZE;

#ifndef NDEBUG
    {
        size_t buf_size_left = dst_size - sizeof(uint32_t);
        assert(buf_size_left > sizeof(uint32_t));
    }
#endif

    
    UINT32ENCODE(q, src_size);

    
    if (H5VL_blob_put(dst_file, p, src_size, q, NULL) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTSET, FAIL, "unable to put blob");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__ref_obj_disk_isnull(const H5VL_object_t *src_file, const void *src_buf, bool *isnull)
{
    H5F_t         *src_f;
    const uint8_t *p = (const uint8_t *)src_buf;
    haddr_t        addr;
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE
    H5T_REF_LOG_DEBUG("");

    
    assert(src_file);
    assert(src_buf);
    assert(isnull);

#ifndef NDEBUG
    {
        bool is_native = false; 

        
        if (H5VL_object_is_native(src_file, &is_native) < 0)
            HGOTO_ERROR(H5E_REFERENCE, H5E_CANTGET, FAIL, "can't query if file uses native VOL connector");

        
        assert(is_native);
    }
#endif 

    
    if (NULL == (src_f = (H5F_t *)H5VL_object_data(src_file)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid VOL object");

    
    H5F_addr_decode(src_f, &p, &addr);

    
    *isnull = (addr == 0) ? true : false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5T__ref_obj_disk_getsize(H5VL_object_t *src_file, const void H5_ATTR_UNUSED *src_buf,
                          size_t H5_ATTR_UNUSED src_size, H5VL_object_t H5_ATTR_UNUSED *dst_file,
                          bool H5_ATTR_UNUSED *dst_copy)
{
    H5F_t *src_f;
    size_t ret_value = 0;

    FUNC_ENTER_PACKAGE
    H5T_REF_LOG_DEBUG("");

    assert(src_file);
    assert(src_buf);

#ifndef NDEBUG
    {
        bool is_native = false; 

        
        if (H5VL_object_is_native(src_file, &is_native) < 0)
            HGOTO_ERROR(H5E_REFERENCE, H5E_CANTGET, 0, "can't query if file uses native VOL connector");

        
        assert(is_native);
    }
#endif 

    
    if (NULL == (src_f = (H5F_t *)H5VL_object_data(src_file)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, 0, "invalid VOL object");

    assert(src_size == H5T_REF_OBJ_DISK_SIZE(src_f));

    ret_value = H5T_REF_OBJ_DISK_SIZE(src_f);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__ref_obj_disk_read(H5VL_object_t *src_file, const void *src_buf, size_t src_size,
                       H5VL_object_t H5_ATTR_UNUSED *dst_file, void *dst_buf, size_t H5_ATTR_UNUSED dst_size)
{
    H5F_t *src_f;
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE
    H5T_REF_LOG_DEBUG("");

    assert(src_file);
    assert(src_buf);
    assert(dst_buf);

#ifndef NDEBUG
    {
        bool is_native = false; 

        
        if (H5VL_object_is_native(src_file, &is_native) < 0)
            HGOTO_ERROR(H5E_REFERENCE, H5E_CANTGET, FAIL, "can't query if file uses native VOL connector");

        
        assert(is_native);
    }
#endif 

    
    if (NULL == (src_f = (H5F_t *)H5VL_object_data(src_file)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid VOL object");

    assert(src_size == H5T_REF_OBJ_DISK_SIZE(src_f));
    assert(dst_size == H5F_SIZEOF_ADDR(src_f));

    
    if (H5R__decode_token_obj_compat((const unsigned char *)src_buf, &src_size, (H5O_token_t *)dst_buf,
                                     H5F_SIZEOF_ADDR(src_f)) < 0)
        HGOTO_ERROR(H5E_REFERENCE, H5E_CANTDECODE, FAIL, "unable to get object address");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__ref_dsetreg_disk_isnull(const H5VL_object_t *src_file, const void *src_buf, bool *isnull)
{
    H5F_t         *src_f;
    const uint8_t *p = (const uint8_t *)src_buf;
    haddr_t        addr;
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE
    H5T_REF_LOG_DEBUG("");

    
    assert(src_file);
    assert(src_buf);
    assert(isnull);

#ifndef NDEBUG
    {
        bool is_native = false; 

        
        if (H5VL_object_is_native(src_file, &is_native) < 0)
            HGOTO_ERROR(H5E_REFERENCE, H5E_CANTGET, FAIL, "can't query if file uses native VOL connector");

        
        assert(is_native);
    }
#endif 

    
    if (NULL == (src_f = (H5F_t *)H5VL_object_data(src_file)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid VOL object");

    
    H5F_addr_decode(src_f, &p, &addr);

    
    *isnull = (addr == 0) ? true : false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5T__ref_dsetreg_disk_getsize(H5VL_object_t H5_ATTR_UNUSED *src_file, const void H5_ATTR_UNUSED *src_buf,
                              size_t H5_ATTR_UNUSED src_size, H5VL_object_t H5_ATTR_UNUSED *dst_file,
                              bool H5_ATTR_UNUSED *dst_copy)
{
    size_t ret_value = sizeof(struct H5Tref_dsetreg);

#ifndef NDEBUG
    FUNC_ENTER_PACKAGE
#else
    FUNC_ENTER_PACKAGE_NOERR
#endif
    H5T_REF_LOG_DEBUG("");

    assert(src_buf);

#ifndef NDEBUG
    {
        H5F_t *src_f;
        bool   is_native = false; 

        
        if (H5VL_object_is_native(src_file, &is_native) < 0)
            HGOTO_ERROR(H5E_REFERENCE, H5E_CANTGET, 0, "can't query if file uses native VOL connector");

        
        assert(is_native);

        
        if (NULL == (src_f = (H5F_t *)H5VL_object_data(src_file)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, 0, "invalid VOL object");

        assert(src_size == H5T_REF_DSETREG_DISK_SIZE(src_f));
    }
#endif 

#ifndef NDEBUG
done:
#endif
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__ref_dsetreg_disk_read(H5VL_object_t *src_file, const void *src_buf, size_t src_size,
                           H5VL_object_t H5_ATTR_UNUSED *dst_file, void *dst_buf,
                           size_t H5_ATTR_UNUSED dst_size)
{
    H5F_t                 *src_f;
    struct H5Tref_dsetreg *dst_reg   = (struct H5Tref_dsetreg *)dst_buf;
    herr_t                 ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE
    H5T_REF_LOG_DEBUG("");

    assert(src_file);
    assert(src_buf);
    assert(dst_buf);
    assert(dst_size == sizeof(struct H5Tref_dsetreg));

#ifndef NDEBUG
    {
        bool is_native = false; 

        
        if (H5VL_object_is_native(src_file, &is_native) < 0)
            HGOTO_ERROR(H5E_REFERENCE, H5E_CANTGET, FAIL, "can't query if file uses native VOL connector");

        
        assert(is_native);
    }
#endif 

    
    if (NULL == (src_f = (H5F_t *)H5VL_object_data(src_file)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid VOL object");

    assert(src_size == H5T_REF_DSETREG_DISK_SIZE(src_f));

    
    if (H5R__decode_token_region_compat(src_f, (const unsigned char *)src_buf, &src_size, &dst_reg->token,
                                        H5F_SIZEOF_ADDR(src_f), &dst_reg->space) < 0)
        HGOTO_ERROR(H5E_REFERENCE, H5E_CANTDECODE, FAIL, "unable to get object address");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T__ref_reclaim(void *elem, const H5T_t *dt)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE
    H5T_REF_LOG_DEBUG("");

    
    assert(elem);
    assert(dt && (dt->shared->type == H5T_REFERENCE));

    if (dt->shared->u.atomic.u.r.opaque && H5R__destroy((H5R_ref_priv_t *)elem) < 0)
        HGOTO_ERROR(H5E_REFERENCE, H5E_CANTFREE, FAIL, "cannot free reference");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
