% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{hpaDist}
\alias{hpaDist}
\alias{dhpa}
\alias{phpa}
\alias{ihpa}
\alias{ehpa}
\alias{etrhpa}
\alias{dtrhpa}
\alias{itrhpa}
\alias{dhpaDiff}
\alias{ihpaDiff}
\title{Probabilities and Moments Hermite Polynomial Approximation}
\usage{
dhpa(
  x,
  pol_coefficients = numeric(0),
  pol_degrees = numeric(0),
  given_ind = logical(0),
  omit_ind = logical(0),
  mean = numeric(0),
  sd = numeric(0),
  is_parallel = FALSE,
  log = FALSE,
  is_validation = TRUE
)

phpa(
  x,
  pol_coefficients = numeric(0),
  pol_degrees = numeric(0),
  given_ind = logical(0),
  omit_ind = logical(0),
  mean = numeric(0),
  sd = numeric(0),
  is_parallel = FALSE,
  log = FALSE,
  is_validation = TRUE
)

ihpa(
  x_lower = matrix(1, 1),
  x_upper = matrix(1, 1),
  pol_coefficients = numeric(0),
  pol_degrees = numeric(0),
  given_ind = logical(0),
  omit_ind = logical(0),
  mean = numeric(0),
  sd = numeric(0),
  is_parallel = FALSE,
  log = FALSE,
  is_validation = TRUE
)

ehpa(
  x = matrix(1, 1),
  pol_coefficients = numeric(0),
  pol_degrees = numeric(0),
  given_ind = logical(0),
  omit_ind = logical(0),
  mean = numeric(0),
  sd = numeric(0),
  expectation_powers = numeric(0),
  is_parallel = FALSE,
  is_validation = TRUE
)

etrhpa(
  tr_left = matrix(1, 1),
  tr_right = matrix(1, 1),
  pol_coefficients = numeric(0),
  pol_degrees = numeric(0),
  mean = numeric(0),
  sd = numeric(0),
  expectation_powers = numeric(0),
  is_parallel = FALSE,
  is_validation = TRUE
)

dtrhpa(
  x,
  tr_left = matrix(),
  tr_right = matrix(),
  pol_coefficients = numeric(0),
  pol_degrees = numeric(0),
  given_ind = logical(0),
  omit_ind = logical(0),
  mean = numeric(0),
  sd = numeric(0),
  is_parallel = FALSE,
  log = FALSE,
  is_validation = TRUE
)

itrhpa(
  x_lower = matrix(1, 1),
  x_upper = matrix(1, 1),
  tr_left = matrix(1, 1),
  tr_right = matrix(1, 1),
  pol_coefficients = numeric(0),
  pol_degrees = numeric(0),
  given_ind = logical(0),
  omit_ind = logical(0),
  mean = numeric(0),
  sd = numeric(0),
  is_parallel = FALSE,
  log = FALSE,
  is_validation = TRUE
)

dhpaDiff(
  x,
  pol_coefficients = numeric(0),
  pol_degrees = numeric(0),
  given_ind = logical(0),
  omit_ind = logical(0),
  mean = numeric(0),
  sd = numeric(0),
  type = "pol_coefficients",
  is_parallel = FALSE,
  log = FALSE,
  is_validation = TRUE
)

ihpaDiff(
  x_lower = matrix(1, 1),
  x_upper = matrix(1, 1),
  pol_coefficients = numeric(0),
  pol_degrees = numeric(0),
  given_ind = logical(0),
  omit_ind = logical(0),
  mean = numeric(0),
  sd = numeric(0),
  type = "pol_coefficients",
  is_parallel = FALSE,
  log = FALSE,
  is_validation = TRUE
)
}
\arguments{
\item{x}{numeric matrix of density function arguments and
conditional values. Note that \code{x} rows are points (observations)
while random vectors components (variables) are columns.}

\item{pol_coefficients}{numeric vector of polynomial coefficients.}

\item{pol_degrees}{non-negative integer vector of polynomial 
degrees (orders).}

\item{given_ind}{logical vector indicating whether corresponding random
vector component is conditioned. By default it is a logical 
vector of \code{FALSE} values. If \code{give_ind[i]} is \code{TRUE}
then \code{i}-th column of \code{x} matrix will contain conditional values.}

\item{omit_ind}{logical vector indicating whether corresponding
component is omitted. By default it is a logical vector 
of \code{FALSE} values. If \code{omit_ind[i]} is \code{TRUE} then values in
\code{i}-th column of \code{x} matrix will be ignored.}

\item{mean}{numeric vector of expected values.}

\item{sd}{positive numeric vector of standard deviations.}

\item{is_parallel}{if \code{TRUE} then multiple cores will be
used for some calculations. It usually provides speed advantage for
large enough samples (about more than 1000 observations).}

\item{log}{logical; if \code{TRUE} then probabilities p are given as log(p)
or derivatives will be given respect to log(p)}

\item{is_validation}{logical value indicating whether function input 
arguments should be validated.  Set it to \code{FALSE} for slight
performance boost (default value is \code{TRUE}).}

\item{x_lower}{numeric matrix of lower integration limits.
Note that \code{x_lower} rows are observations while variables are columns.}

\item{x_upper}{numeric matrix of upper integration limits.
Note that \code{x_upper} rows are observations while variables
are columns.}

\item{expectation_powers}{integer vector of random vector components powers.}

\item{tr_left}{numeric matrix of left (lower) truncation limits.
Note that \code{tr_left} rows are observations while variables are columns.
If \code{tr_left} and \code{tr_right} are single row matrices then the same 
truncation limits will be applied to all observations that are determined 
by the first rows of these matrices.}

\item{tr_right}{numeric matrix of right (upper) truncation limits.
Note that \code{tr_right} rows are observations while variables are columns.
If \code{tr_left} and \code{tr_right} are single row matrices then the same 
truncation limits will be applied to all observations that are determined 
by the first rows of these matrices.}

\item{type}{determines the partial derivatives to be included into the
gradient. If \code{type="pol_coefficients"} then gradient will contain 
partial derivatives respect to polynomial coefficients listed in the
same order as \code{pol_coefficients}. Other available types are 
\code{type = "mean"} and \code{type = "sd"}.
For function \code{\link[hpa]{dhpaDiff}} it is possible to take
gradient respect to the x points setting \code{type="x"}.
For function \code{\link[hpa]{ihpaDiff}} it is possible to take
gradient respect to the x lower and upper points setting 
\code{type = "x_lower"} or \code{type = "upper"} correspondingly.
In order to get full gradient please set \code{type="all"}.}
}
\value{
Functions \code{\link[hpa]{dhpa}}, \code{\link[hpa]{phpa}} and 
\code{\link[hpa]{dtrhpa}} return vector of probabilities of length
\code{nrow(x)}. 

Functions \code{\link[hpa]{ihpa}} and 
\code{\link[hpa]{itrhpa}} return vector of probabilities of length
\code{nrow(x_upper)}.

If \code{x} argument has not been provided or is a single row
matrix then function 
\code{\link[hpa]{ehpa}} returns moment value. Otherwise it returns vector of 
length \code{nrow(x)} containing moments values.

If \code{tr_left} and \code{tr_right} arguments are single row matrices then
function \code{\link[hpa]{etrhpa}} returns moment value.
Otherwise it returns vector of length
\code{max(nrow(tr_left), nrow(tr_right))} containing moments values.

Functions \code{\link[hpa]{dhpaDiff}} and \code{\link[hpa]{ihpaDiff}} 
return Jacobin matrix. The number
of columns depends on \code{type} argument. The number of rows is
\code{nrow(x)} for \code{\link[hpa]{dhpaDiff}} and 
\code{nrow(x_upper)} for
\code{\link[hpa]{ihpaDiff}}

If \code{mean} or \code{sd} are not specified they assume the default 
values of \eqn{m}-dimensional vectors of 0 and 1, respectively. 
If \code{x_lower} is not specified then it is the matrix of the 
same size as \code{x_upper} containing negative infinity values only. If
\code{expectation_powers} is not specified then it is \eqn{m}-dimensional
vector of 0 values.

Please see 'Details' section for additional information.
}
\description{
Approximation of truncated, marginal and conditional densities,
moments and cumulative probabilities of multivariate distributions via
Hermite polynomial based approach proposed by Gallant and Nychka in 1987.

Density approximating function is scale adjusted product of two terms. 
The first one is squared multivariate polynomial of \code{pol_degrees}  
degrees with \code{pol_coefficients} coefficients vector. 
The second is product of independent normal random variables' densities with 
expected values and standard deviations given by \code{mean} and \code{sd} 
vectors correspondingly. Approximating function satisfies properties of 
density function thus generating a broad family of distributions.
Characteristics of these distributions 
(moments, quantiles, probabilities and so on) 
may provide accurate approximations to characteristic of other
distributions. Moreover it is usually possible to provide arbitrary close
approximation by the means of polynomial degrees increase.
}
\details{
It is possible to approximate densities 
\code{\link[hpa]{dhpa}}, cumulative probabilities
\code{\link[hpa]{phpa}}, \code{\link[hpa]{ihpa}}, moments
\code{\link[hpa]{ehpa}} as well as their truncated
\code{\link[hpa]{dtrhpa}}, \code{\link[hpa]{itrhpa}}, 
\code{\link[hpa]{etrhpa}} forms
and gradients \code{\link[hpa]{dhpaDiff}}, \code{\link[hpa]{ihpaDiff}}.
Note that \code{\link[hpa]{phpa}} is special of \code{\link[hpa]{ihpa}}
where \code{x}
corresponds to \code{x_upper} while \code{x_lower} is matrix of
negative infinity values. So  \code{\link[hpa]{phpa}} intended to approximate 
cumulative
distribution functions while \code{\link[hpa]{ihpa}} approximates 
probabilities that
random vector components will be between values determined by rows of 
\code{x_lower} and \code{x_upper} matrices. Further details are given below.


Since density approximating function is non-negative and integrates
to 1 it is density function for some \eqn{m}-variate 
random vector \eqn{\xi}. Approximating function \eqn{f_{\xi }(x)} 
has the following form:
\deqn{f_{\xi }(x) = f_{\xi }(x;\mu, \sigma, \alpha) =
\frac{1}{\psi }\prod\limits_{t=1}^{m}\phi 
({x}_{t};{\mu }_{t},{\sigma }_{t}){{\left( \sum\limits_{{i}_{1}=0}^{{K}_{1}}
{...}\sum\limits_{{i}_{m}=0}^{{K}_{m}}{{{\alpha }_{({{i}_{1}},...,{{i}_{m}})
}}\prod\limits_{r=1}^{m}x_{r}^{{{i}_{r}}}} \right)}^{2}}}
\deqn{\psi =\sum\limits_{{i}_{1}=0}^{{K}_{1}}{...}\sum
\limits_{{i}_{m}=0}^{{K}_{m}}{\sum\limits_{{j}_{1}=0}^{{K}_{1}}
{...}\sum\limits_{{j}_{m}=0}^{{K}_{m}}{{{\alpha }_{({i}_{1},
\cdots,{i}_{m})}}{{\alpha }_{({j}_{1},\cdots,{j}_{m})}}\prod
\limits_{r=1}^{m}\mathcal{M}({i}_{r}+{j}_{r};{{\mu }_{r}},{\sigma }_{r})}},}
where:

\eqn{x = (x_{1},...x_{m})} - is vector of arguments i.e. rows
 of \code{x} matrix in \code{\link[hpa]{dhpa}}.
 
\eqn{{\alpha }_{({i}_{1},\cdots,{i}_{m})}} - is polynomial coefficient
corresponding to \code{pol_coefficients[k]} element. In order to investigate
correspondence between \code{k} and \eqn{({i}_{1},\cdots,{i}_{m})} values 
please see 'Examples' section below or \code{\link[hpa]{polynomialIndex}} 
function 'Details', 'Value' and 'Examples' sections. Note that if \eqn{m=1}
then \code{pol_coefficients[k]} simply corresponds to \eqn{\alpha_{k-1}}.

\eqn{(K_{1},...,K_{m})} - are polynomial degrees (orders) provided via
\code{pol_degrees} argument so \code{pol_degrees[i]} determines \eqn{K_{i}}.

\eqn{\phi 
(.;{\mu }_{t},{\sigma }_{t})} - is normal random variable density function 
where \eqn{\mu_{t}} and \eqn{\sigma_{t}} are mean and standard deviation 
determined by \code{mean[t]} and \code{sd[t]} arguments values.

\eqn{\mathcal{M}(q;{{\mu }_{r}},{\sigma }_{r})} - is \eqn{q}-th order
moment of normal random variable with mean \eqn{{\mu }_{r}} and standard
deviation \eqn{{\sigma }_{r}}. Note that function 
\code{\link[hpa]{normalMoment}} allows to calculate and differentiate normal 
random variable's moments.

\eqn{\psi} - constant term insuring that \eqn{f_{\xi }(x)} is
density function.

Therefore \code{\link[hpa]{dhpa}} allows to calculate \eqn{f_{\xi}(x)} 
values at points
determined by rows of \code{x} matrix given polynomial 
degrees \code{pol_degrees} (\eqn{K}) as well as \code{mean} (\eqn{\mu}), 
\code{sd} (\eqn{\sigma}) and \code{pol_coefficients} (\eqn{\alpha}) 
parameters values. Note that \code{mean}, \code{sd} and \code{pol_degrees} are 
\eqn{m}-variate vectors while \code{pol_coefficients} has
\code{prod(pol_degrees + 1)} elements.

Cumulative probabilities could be approximated as follows:
\deqn{P\left(\underline{x}_{1}\leq\xi_{1}\leq\overline{x}_{1},...,
\underline{x}_{m}\leq\xi_{m}\leq\overline{x}_{m}\right) = }
\deqn{= \bar{F}_{\xi}(\underline{x},\bar{x}) = 
\bar{F}_{\xi}(\underline{x},\bar{x};\mu, \sigma, \alpha) =
\frac{1}{\psi }
\prod\limits_{t=1}^{m}(\Phi ({{{\bar{x}}}_{t}};{{\mu }_{t}},
{{\sigma }_{t}})-\Phi ({{{\underline{x}}}_{t}};{{\mu }_{t}},
{{\sigma }_{t}})) * }
\eqn{* \sum\limits_{{{i}_{1}}=0}^{{{K}_{1}}}{...}
\sum\limits_{{{i}_{m}}=0}^{{{K}_{m}}}{\sum\limits_{{{j}_{1}}=0}^{{{K}_{1}}}
{...}\sum\limits_{{{j}_{m}}=0}^{{{K}_{m}}}
{{{\alpha }_{({{i}_{1}},...,{{i}_{m}})}}{{\alpha }_{({{j}_{1}},...,{{j}_{m}})
}}}}\prod\limits_{r=1}^{m}\mathcal{M}_{TR}\left({i}_{r}+{j}_{r};
\underline{x}_{r},\overline{x}_{r},\mu_{r},\sigma_{r}\right)}

where:

\eqn{\Phi 
(.;{\mu }_{t},{\sigma }_{t})} - is normal random variable's cumulative 
distribution function where \eqn{\mu_{t}} and \eqn{\sigma_{t}} are mean and 
standard deviation determined by \code{mean[t]} and \code{sd[t]} arguments 
values.

\eqn{\mathcal{M}_{TR}(q;
\underline{x}_{r},\overline{x}_{r},\mu_{r},\sigma_{r})} - is 
\eqn{q}-th order
moment of truncated (from above by \eqn{\overline{x}_{r}} and from below by
\eqn{\underline{x}_{r}}) 
normal random variable with mean \eqn{{\mu }_{r}} and standard
deviation \eqn{{\sigma }_{r}}. Note that function 
\code{\link[hpa]{truncatedNormalMoment}} allows to calculate and 
differentiate truncated normal random variable's moments.

\eqn{\overline{x} = (\overline{x}_{1},...,\overline{x}_{m})} - 
vector of upper integration limits
i.e. rows of \code{x_upper} matrix in \code{\link[hpa]{ihpa}}.

\eqn{\underline{x} = (\underline{x}_{1},...,\underline{x}_{m})} - 
vector of lower integration limits
i.e. rows of \code{x_lower} matrix in \code{\link[hpa]{ihpa}}.

Therefore \code{\link[hpa]{ihpa}} allows to calculate interval distribution 
function \eqn{\bar{F}_{\xi}(\underline{x},\bar{x})}
values at points determined by rows of \code{x_lower} (\eqn{\underline{x}})
and \code{x_upper} (\eqn{\overline{x}}) matrices.
The rest of the arguments are similar to \code{dhpa}.

Expected value powered product approximation is as follows:

\deqn{E\left( \prod\limits_{t=1}^{m}\xi_{t}^{{{k}_{t}}} \right)=
\frac{1}{\psi }\sum\limits_{{{i}_{1}}=0}^{{{K}_{1}}}{...}
\sum\limits_{{{i}_{m}}=0}^{{{K}_{m}}}
{\sum\limits_{{{j}_{1}}=0}^{{{K}_{1}}}{...}
\sum\limits_{{{j}_{m}}=0}^{{{K}_{m}}}
{{{\alpha }_{({{i}_{1}},...,{{i}_{m}})}}
{{\alpha }_{({{j}_{1}},...,{{j}_{m}})}}}}
\prod\limits_{r=1}^{m}\mathcal{M}({{i}_{r}}+{{j}_{r}}+{{k}_{t}};
{{\mu }_{r}},{{\sigma }_{r}})}

where \eqn{(k_{1},...,k_{m})} are integer powers determined by
\code{expectation_powers} argument of \code{\link[hpa]{ehpa}} so
\code{expectation_powers[t]} assigns \eqn{k_{t}}. Note that argument \code{x}
in \code{\link[hpa]{ehpa}} allows to determined conditional values.

Expanding polynomial degrees \eqn{(K_{1},...,K_{m})} it is possible to 
provide arbitrary close approximation to density of some \eqn{m}-variate 
random vector \eqn{\xi^{\star}}. So actually \eqn{f_{\xi}(x)}
approximates \eqn{f_{\xi^{\star}}(x)}. Accurate approximation requires
appropriate \code{mean}, \code{sd} and \code{pol_coefficients} values
selection. In order to get sample estimates of these parameters please apply 
\code{\link[hpa]{hpaML}} function.

In order to perform calculation for marginal distribution of some 
\eqn{\xi} components please provide omitted 
components via \code{omit_ind} argument.
For examples if ones assume \eqn{m=5}-variate distribution
and wants to deal with \eqn{1}-st, \eqn{3}-rd, and \eqn{5}-th components 
only i.e. \eqn{(\xi_{1},\xi_{3},\xi_{5})} then set 
\code{omit_ind = c(FALSE, TRUE, FALSE, TRUE, FALSE)}
indicating that \eqn{\xi_{2}} and \eqn{\xi_{4}} should be 'omitted' from
\eqn{\xi} since \eqn{2}-nd and \eqn{4}-th values of \code{omit_ind} are
\code{TRUE}.
Then \code{x} still should be \eqn{5} column matrix but 
values in \eqn{2}-nd and \eqn{4}-th columns will not affect 
calculation results. Meanwhile note that marginal distribution of \code{t}
components of \eqn{\xi} usually do not coincide with any marginal
distribution generated by \code{t}-variate density approximating function.

In order to perform calculation for conditional distribution i.e. given 
fixed values for some \eqn{\xi} components please provide these
components via \code{given_ind} argument.
For example if ones assume \eqn{m=5}-variate distribution
and wants to deal with \eqn{1}-st, \eqn{3}-rd, and \eqn{5}-th components 
given fixed values (suppose 8 and 10) for the other two components i.e. 
\eqn{(\xi|\xi_{2} = 8, \xi_{4} = 10)} then set 
\code{given_ind = c(FALSE, TRUE, FALSE, TRUE, FALSE)} and
\code{x[2] = 8}, \code{x[4] = 10} where for simplicity it is assumed that
\code{x} is single row \eqn{5} column matrix; it is possible to provide  
different conditional values for the same components simply setting different  
values to different \code{x} rows.

Note that it is possible to combine \code{given_ind} and \code{omit_ind}
arguments. However it is wrong to set both \code{given_ind[i]} and 
\code{omit_ind[i]} to \code{TRUE}. Also at least one value should be
\code{FALSE} both for \code{given_ind} and \code{omit_ind}.

In order to consider truncated distribution of \eqn{\xi} i.e. 
\eqn{\left(\xi|\overline{a}_{1}\leq\xi_{1}\leq\overline{b}_{1},
\cdots,\overline{a}_{m}\leq\xi_{m}\leq\overline{b}_{m}\right)}
please set lower (left) truncation points \eqn{\overline{a}} and 
upper (right) truncation points \eqn{\overline{b}} via \code{tr_left} 
and \code{tr_right} arguments correspondingly. Note that if lower truncation
points are negative infinite and upper truncation points are positive
infinite then \code{\link[hpa]{dtrhpa}}, \code{\link[hpa]{itrhpa}} and 
\code{\link[hpa]{etrhpa}} are similar to \code{\link[hpa]{dhpa}},
\code{\link[hpa]{ihpa}} and \code{\link[hpa]{ehpa}} correspondingly.

In order to calculate Jacobian of \eqn{f_{\xi }(x;\mu, \sigma, \alpha)}
and \eqn{\bar{F}_{\xi}(\underline{x},\bar{x};\mu, \sigma, \alpha)} w.r.t
all ore some particular parameters please apply \code{\link[hpa]{dhpaDiff}}
and \code{\link[hpa]{ihpaDiff}} functions correspondingly specifying
parameters of interest via \code{type} argument. If \code{x} or
\code{x_lower} and \code{x_upper} are single row matrices then gradients
will be calculated.

For further information please see 'Examples' section. Note that examples
are given separately for each function.

Densities Hermite polynomial approximation approach has been
proposed by A. Gallant and D. W. Nychka in 1987. The main idea is to
approximate unknown distribution density with scaled Hermite polynomial.
For more information please refer to the literature listed below.
}
\examples{
## Example demonstrating dhpa function application.
## Let's approximate some three random variables (i.e. X1, X2 and X3) 
## joint density function at points  x = (0,1, 0.2, 0.3) and 
## y = (0.5, 0.8, 0.6) with Hermite polynomial of (1, 2, 3) degrees which 
## polynomial coefficients equal 1 except coefficient related  to x1*(x^3) 
## polynomial element which equals 2. Also suppose that normal density 
## related mean vector equals (1.1, 1.2, 1.3) while standard deviations 
## vector is (2.1, 2.2, 2.3).

# Prepare initial values
x <- matrix(c(0.1, 0.2, 0.3), nrow = 1)   # x point as a single row matrix
y <- matrix(c(0.5, 0.8, 0.6), nrow = 1)   # y point as a single row matrix
x_y <- rbind(x, y)                        # matrix which rows are x and y
mean <- c(1.1, 1.2, 1.3)
sd <- c(2.1, 2.2, 2.3)
pol_degrees <- c(1, 2, 3)

# Create polynomial powers and indexes correspondence matrix
pol_ind <- polynomialIndex(pol_degrees)
# Set all polynomial coefficients to 1
pol_coefficients <- rep(1, ncol(pol_ind))
pol_degrees_n <- length(pol_degrees)

# Assign coefficient 2 to the polynomial element (x1 ^ 1)*(x2 ^ 0)*(x3 ^ 2)
pol_coefficients[apply(pol_ind, 2, function(x) all(x == c(1, 0, 2)))] <- 2

# Visualize correspondence between polynomial 
# elements and their coefficients
as.data.frame(rbind(pol_ind, pol_coefficients),
   row.names = c("x1 power", "x2 power", "x3 power", "coefficients"),
   optional = TRUE)
printPolynomial(pol_degrees, pol_coefficients)

# Calculate density approximation 
   # at point x (note that x should be a matrix)
dhpa(x = x,
     pol_coefficients = pol_coefficients, 
     pol_degrees = pol_degrees,
     mean = mean, sd = sd)
   # at points x and y
dhpa(x = x_y,
     pol_coefficients = pol_coefficients, 
     pol_degrees = pol_degrees,
     mean = mean, sd = sd)

# Condition second component to be 0.5 i.e. X2 = 0.5.
# Substitute x and y second components with conditional value 0.5
x <- matrix(c(0.1, 0.5, 0.3), nrow = 1) # or simply x[2] <- 0.5
y <- matrix(c(0.4, 0.5, 0.6), nrow = 1) # or simply y[2] <- 0.5
x_y <- rbind(x, y) 
# Set TRUE to the second component indicating that it is conditioned
given_ind <- c(FALSE, TRUE, FALSE)

# Calculate conditional (on X2 = 0.5) density approximation 
   # at point x
dhpa(x = x,
     pol_coefficients = pol_coefficients, 
     pol_degrees = pol_degrees,
     mean = mean, sd = sd,
     given_ind = given_ind)
   # at points x and y
dhpa(x = x_y,
     pol_coefficients = pol_coefficients, 
     pol_degrees = pol_degrees,
     mean = mean, sd = sd,
     given_ind = given_ind)	
	
# Consider third component marginal distribution conditioned on the
# second component 0.5 value i.e. (X3 | X2 = 0.5).
# Set TRUE to the first component indicating that it is omitted
omit_ind <- c(TRUE, FALSE, FALSE)

# Calculate conditional (on x2 = 0.5) marginal (for x3) density approximation
   # at point x
dhpa(x = x,
     pol_coefficients = pol_coefficients, 
     pol_degrees = pol_degrees,
     mean = mean, sd = sd,
     given_ind = given_ind, 
     omit_ind = omit_ind)
   # at points x and y
dhpa(x = x_y,
     pol_coefficients = pol_coefficients, 
     pol_degrees = pol_degrees,
     mean = mean, sd = sd,
     given_ind = given_ind, 
     omit_ind = omit_ind)  
        
## Example demonstrating phpa function application.
## Let's approximate some three random variables (X1, X2, X3) 
## joint cumulative distribution function (cdf) at point (0,1, 0.2, 0.3)
## with Hermite polynomial of (1, 2, 3) degrees which polynomial 
## coefficients equal 1 except coefficient related to x1*(x^3) polynomial 
## element which equals 2. Also suppose that normal density related
## mean vector equals (1.1, 1.2, 1.3) while standard deviations
## vector is (2.1, 2.2, 2.3).

## Prepare initial values
x <- matrix(c(0.1, 0.2, 0.3), nrow = 1)
mean <- c(1.1, 1.2, 1.3)
sd <- c(2.1, 2.2, 2.3)
pol_degrees <- c(1, 2, 3)

# Create polynomial powers and indexes correspondence matrix
pol_ind <- polynomialIndex(pol_degrees)

# Set all polynomial coefficients to 1
pol_coefficients <- rep(1, ncol(pol_ind))
pol_degrees_n <- length(pol_degrees)

# Assign coefficient 2 to the polynomial element (x1 ^ 1)*(x2 ^ 0)*(x3 ^ 2)
pol_coefficients[apply(pol_ind, 2, function(x) all(x == c(1, 0, 2)))] <- 2

# Visualize correspondence between polynomial
# elements and their coefficients
as.data.frame(rbind(pol_ind, pol_coefficients),
	           row.names = c("x1 power", "x2 power", 
	                         "x3 power", "coefficients"),
          	 optional = TRUE)
printPolynomial(pol_degrees, pol_coefficients)

# Calculate cdf approximation at point x
phpa(x = x,
     pol_coefficients = pol_coefficients, 
     pol_degrees = pol_degrees,
     mean = mean, sd = sd)
	
# Condition second component to be 0.5
# Substitute x second component with conditional value 0.5
x <- matrix(c(0.1, 0.5, 0.3), nrow = 1) # or simply x[2] <- 0.5

# Set TRUE to the second component indicating that it is conditioned
given_ind <- c(FALSE, TRUE, FALSE)

# Calculate conditional (on X2 = 0.5) cdf approximation at point x
phpa(x = x,
     pol_coefficients = pol_coefficients, 
     pol_degrees = pol_degrees,
     mean = mean, sd = sd,
     given_ind = given_ind)
	
# Consider third component marginal distribution
# conditioned on the second component 0.5 value

# Set TRUE to the first component indicating that it is omitted
omit_ind <- c(TRUE, FALSE, FALSE)

# Calculate conditional (on X2 = 0.5) marginal (for X3) cdf 
# approximation at point x
phpa(x = x,
      pol_coefficients = pol_coefficients, 
      pol_degrees = pol_degrees,
      mean = mean, sd = sd,
      given_ind = given_ind, 
      omit_ind = omit_ind)

## Example demonstrating ihpa function application.
## Let's approximate some three random variables (X1, X2, X3) joint interval 
## distribution function (intdf) at lower and upper points (0,1, 0.2, 0.3) 
## and (0,4, 0.5, 0.6) correspondingly with Hermite polynomial of (1, 2, 3) 
## degrees which polynomial coefficients equal 1 except coefficient related 
## to x1*(x^3) polynomial element which equals 2. Also suppose that normal
## density related mean vector equals (1.1, 1.2, 1.3) while standard
## deviations vector is (2.1, 2.2, 2.3).

## Prepare initial values
x_lower <- matrix(c(0.1, 0.2, 0.3), nrow=1)
x_upper <- matrix(c(0.4, 0.5, 0.6), nrow=1)
mean <- c(1.1, 1.2, 1.3)
sd <- c(2.1, 2.2, 2.3)
pol_degrees <- c(1, 2, 3)

# Create polynomial powers and indexes correspondence matrix
pol_ind <- polynomialIndex(pol_degrees)

# Set all polynomial coefficients to 1
pol_coefficients <- rep(1, ncol(pol_ind))
pol_degrees_n <- length(pol_degrees)

# Assign coefficient 2 to the polynomial element (x1 ^ 1)*(x2 ^ 0)*(x3 ^ 2)
pol_coefficients[apply(pol_ind, 2, function(x) all(x == c(1, 0, 2)))] <- 2

# Visualize correspondence between polynomial 
# elements and their coefficients
as.data.frame(rbind(pol_ind, pol_coefficients),
	           row.names = c("x1 power", "x2 power", 
	                         "x3 power", "coefficients"),
              optional = TRUE)
printPolynomial(pol_degrees, pol_coefficients)

# Calculate intdf approximation at points x_lower and x_upper
ihpa(x_lower = x_lower, x_upper = x_upper, 
     pol_coefficients = pol_coefficients, 
     pol_degrees = pol_degrees,
     mean = mean, sd = sd)
	
# Condition second component to be 0.7
# Substitute x second component with conditional value 0.7
x_upper <- matrix(c(0.4, 0.7, 0.6), nrow = 1)

# Set TRUE to the second component indicating that it is conditioned
given_ind <- c(FALSE, TRUE, FALSE)

# Calculate conditional (on X2 = 0.5) intdf approximation 
# at points x_lower and x_upper
ihpa(x_lower = x_lower, x_upper = x_upper,
     pol_coefficients = pol_coefficients, 
     pol_degrees = pol_degrees,
     mean = mean, sd = sd,
     given_ind = given_ind)
	
# Consider third component marginal distribution
# conditioned on the second component 0.7 value
# Set TRUE to the first component indicating that it is omitted
omit_ind <- c(TRUE, FALSE, FALSE)

# Calculate conditional (on X2 = 0.5) marginal (for X3) 
# intdf approximation at points x_lower and x_upper
ihpa(x_lower = x_lower, x_upper = x_upper,
     pol_coefficients = pol_coefficients, 
     pol_degrees = pol_degrees,
     mean = mean, sd = sd,
     given_ind = given_ind, omit_ind = omit_ind)
	
## Example demonstrating ehpa function application.
## Let's approximate some three random variables (X1, X2, X3) powered product 
## expectation for powers (3, 2, 1) with Hermite polynomial of (1, 2, 3) 
## degrees which polynomial coefficients equal 1 except coefficient 
## related to x1*(x^3) polynomial element which equals 2.
## Also suppose that normal density related mean vector equals 
## (1.1, 1.2, 1.3) while standard deviations vector is (2.1, 2.2, 2.3).

# Prepare initial values
expectation_powers = c(3,2,1)
mean <- c(1.1, 1.2, 1.3)
sd <- c(2.1, 2.2, 2.3)
pol_degrees <- c(1, 2, 3)

# Create polynomial powers and indexes correspondence matrix
pol_ind <- polynomialIndex(pol_degrees)

# Set all polynomial coefficients to 1
pol_coefficients <- rep(1, ncol(pol_ind))
pol_degrees_n <- length(pol_degrees)

#Assign coefficient 2 to the polynomial element (x1 ^ 1)*(x2 ^ 0)*(x3 ^ 2)
pol_coefficients[apply(pol_ind, 2, function(x) all(x == c(1, 0, 2)))] <- 2

# Visualize correspondence between polynomial elements and their coefficients
as.data.frame(rbind(pol_ind, pol_coefficients),
              row.names = c("x1 power", "x2 power", 
                            "x3 power", "coefficients"),
              optional = TRUE)
printPolynomial(pol_degrees, pol_coefficients)

# Calculate expected powered product approximation
ehpa(pol_coefficients = pol_coefficients, 
     pol_degrees = pol_degrees,
     mean = mean, sd = sd, 
     expectation_powers = expectation_powers)
	
# Condition second component to be 0.5
# Substitute x second component with conditional value 0.5
x <- matrix(c(NA, 0.5, NA), nrow = 1)
#Set TRUE to the second component indicating that it is conditioned
given_ind <- c(FALSE, TRUE, FALSE)

# Calculate conditional (on X2 = 0.5) expected powered product approximation
ehpa(x = x,
     pol_coefficients = pol_coefficients, 
     pol_degrees = pol_degrees,
     mean = mean, sd = sd, 
     expectation_powers = expectation_powers,
     given_ind = given_ind)
	
# Consider third component marginal distribution
# conditioned on the second component 0.5 value
# Set TRUE to the first component indicating that it is omitted
omit_ind <- c(TRUE, FALSE, FALSE)

# Calculate conditional (on X2 = 0.5) marginal (for X3) expected powered 
# product approximation at points x_lower and x_upper
ehpa(x = x,
     pol_coefficients = pol_coefficients, 
     pol_degrees = pol_degrees,
     mean = mean, sd = sd, 
     expectation_powers = expectation_powers,
     given_ind = given_ind, 
     omit_ind = omit_ind)
	
## Example demonstrating etrhpa function application.
## Let's approximate some three truncated random variables (X1, X2, X3) 
## powered product expectation for powers (3, 2, 1) with Hermite polynomial 
## of (1,2,3) degrees which polynomial coefficients equal 1 except 
## coefficient related to x1*(x^3) polynomial element which equals 2. Also
## suppose that normal density related mean vector equals (1.1, 1.2, 1.3) 
## while standard deviations vector is (2.1, 2.2, 2.3). Suppose that lower  
## and upper truncation points are (-1.1,-1.2,-1.3) and (1.1,1.2,1.3) 
## correspondingly.

# Prepare initial values
expectation_powers = c(3,2,1)
tr_left = matrix(c(-1.1,-1.2,-1.3), nrow = 1)
tr_right = matrix(c(1.1,1.2,1.3), nrow = 1)
mean <- c(1.1, 1.2, 1.3)
sd <- c(2.1, 2.2, 2.3)
pol_degrees <- c(1, 2, 3)

# Create polynomial powers and indexes correspondence matrix
pol_ind <- polynomialIndex(pol_degrees)
# Set all polynomial coefficients to 1
pol_coefficients <- rep(1, ncol(pol_ind))
pol_degrees_n <- length(pol_degrees)

# Assign coefficient 2 to the polynomial element (x1 ^ 1)*(x2 ^ 0)*(x3 ^ 2)
pol_coefficients[apply(pol_ind, 2, function(x) all(x == c(1, 0, 2)))] <- 2

# Visualize correspondence between polynomial elements and their coefficients
as.data.frame(rbind(pol_ind, pol_coefficients),
              row.names = c("x1 power", "x2 power", 
                            "x3 power", "coefficients"),
              optional = TRUE)
printPolynomial(pol_degrees, pol_coefficients)

# Calculate expected powered product approximation for truncated distribution
etrhpa(pol_coefficients = pol_coefficients, 
       pol_degrees = pol_degrees,
       mean = mean, sd = sd, 
       expectation_powers = expectation_powers,
       tr_left = tr_left, tr_right = tr_right)
       
## Example demonstrating dtrhpa function application.
## Let's approximate some three random variables (X1, X2, X3) joint density 
## function at point (0,1, 0.2, 0.3) with Hermite polynomial of (1,2,3)  
## degrees which polynomial coefficients equal 1 except coefficient related 
## to x1*(x^3) polynomial element which equals 2. Also suppose that normal 
## density related mean vector equals (1.1, 1.2, 1.3) while standard 
## deviations vector is (2.1, 2.2, 2.3). Suppose that lower and upper 
## truncation points are (-1.1,-1.2,-1.3) and (1.1,1.2,1.3) correspondingly.

# Prepare initial values
x <- matrix(c(0.1, 0.2, 0.3), nrow=1)
tr_left = matrix(c(-1.1,-1.2,-1.3), nrow = 1)
tr_right = matrix(c(1.1,1.2,1.3), nrow = 1)
mean <- c(1.1, 1.2, 1.3)
sd <- c(2.1, 2.2, 2.3)
pol_degrees <- c(1, 2, 3)

# Create polynomial powers and indexes correspondence matrix
pol_ind <- polynomialIndex(pol_degrees)

# Set all polynomial coefficients to 1
pol_coefficients <- rep(1, ncol(pol_ind))
pol_degrees_n <- length(pol_degrees)

# Assign coefficient 2 to the polynomial element (x1 ^ 1)*(x2 ^ 0)*(x3 ^ 2)
pol_coefficients[apply(pol_ind, 2, function(x) all(x == c(1, 0, 2)))] <- 2

# Visualize correspondence between polynomial elements and their coefficients
as.data.frame(rbind(pol_ind, pol_coefficients),
              row.names = c("x1 power", "x2 power", 
                            "x3 power", "coefficients"),
              optional = TRUE)
printPolynomial(pol_degrees, pol_coefficients)

# Calculate density approximation at point x
dtrhpa(x = x,
       pol_coefficients = pol_coefficients, 
       pol_degrees = pol_degrees,
       mean = mean, sd = sd,
       tr_left = tr_left, 
       tr_right = tr_right)
	
# Condition second component to be 0.5
# Substitute x second component with conditional value 0.5
x <- matrix(c(0.1, 0.5, 0.3), nrow = 1)
# Set TRUE to the second component indicating that it is conditioned
given_ind <- c(FALSE, TRUE, FALSE)
# Calculate conditional (on x2 = 0.5) density approximation at point x
dtrhpa(x = x,
       pol_coefficients = pol_coefficients, 
       pol_degrees = pol_degrees,
       mean = mean, sd = sd,
       given_ind = given_ind,
       tr_left = tr_left, tr_right = tr_right)
	
# Consider third component marginal distribution
# conditioned on the second component 0.5 value
# Set TRUE to the first component indicating that it is omitted
omit_ind <- c(TRUE, FALSE, FALSE)

# Calculate conditional (on X2 = 0.5) marginal (for X3) 
# density approximation at point x
dtrhpa(x = x,
       pol_coefficients = pol_coefficients, 
       pol_degrees = pol_degrees,
       mean = mean, sd = sd,
       given_ind = given_ind, omit_ind = omit_ind,
       tr_left = tr_left, tr_right = tr_right)
       
## Example demonstrating itrhpa function application.
## Let's approximate some three truncated random variables (X1, X2, X3) joint 
## interval distribution function at lower and upper points (0,1, 0.2, 0.3) 
## and (0,4, 0.5, 0.6) correspondingly with Hermite polynomial of (1 ,2, 3) 
## degrees which polynomial coefficients equal 1 except coefficient
## related to x1*(x^3) polynomial element which equals 2. Also suppose 
## that normal density related mean vector equals (1.1, 1.2, 1.3) while 
## standard deviations vector is (2.1, 2.2, 2.3). Suppose that lower and 
## upper truncation are (-1.1,-1.2,-1.3) and (1.1,1.2,1.3) correspondingly.

# Prepare initial values
x_lower <- matrix(c(0.1, 0.2, 0.3), nrow=1)
x_upper <- matrix(c(0.4, 0.5, 0.6), nrow=1)
tr_left = matrix(c(-1.1,-1.2,-1.3), nrow = 1)
tr_right = matrix(c(1.1,1.2,1.3), nrow = 1)
mean <- c(1.1, 1.2, 1.3)
sd <- c(2.1, 2.2, 2.3)
pol_degrees <- c(1, 2, 3)

# Create polynomial powers and indexes correspondence matrix
pol_ind <- polynomialIndex(pol_degrees)
# Set all polynomial coefficients to 1
pol_coefficients <- rep(1, ncol(pol_ind))
pol_degrees_n <- length(pol_degrees)

# Assign coefficient 2 to the polynomial element (x1 ^ 1)*(x2 ^ 0)*(x3 ^ 2)
pol_coefficients[apply(pol_ind, 2, function(x) all(x == c(1, 0, 2)))] <- 2
# Visualize correspondence between polynomial 
# elements and their coefficients
as.data.frame(rbind(pol_ind, pol_coefficients),
              row.names = c("x1 power", "x2 power", 
                            "x3 power", "coefficients"),
              optional = TRUE)
printPolynomial(pol_degrees, pol_coefficients)

# Calculate intdf approximation at points x_lower and x_upper
itrhpa(x_lower = x_lower, x_upper = x_upper, 
       pol_coefficients = pol_coefficients, 
       pol_degrees = pol_degrees,
       mean = mean, sd = sd,
       tr_left = tr_left, tr_right = tr_right)
    
# Condition second component to be 0.7
# Substitute x second component with conditional value 0.7
x_upper <- matrix(c(0.4, 0.7, 0.6), nrow = 1)
# Set TRUE to the second component indicating that it is conditioned
given_ind <- c(FALSE, TRUE, FALSE)

# Calculate conditional (on X2 = 0.5) intdf 
# approximation at points x_lower and x_upper
itrhpa(x_lower = x_lower, x_upper = x_upper,
       pol_coefficients = pol_coefficients, 
       pol_degrees = pol_degrees,
       mean = mean, sd = sd,
       given_ind = given_ind,
       tr_left = tr_left, tr_right = tr_right)
    
# Consider third component marginal distribution
# conditioned on the second component 0.7 value
# Set TRUE to the first component indicating that it is omitted
omit_ind <- c(TRUE, FALSE, FALSE)

# Calculate conditional (on X2 = 0.5) marginal (for X3) intdf 
# approximation at points x_lower and x_upper
itrhpa(x_lower = x_lower, x_upper = x_upper,
       pol_coefficients = pol_coefficients, 
       pol_degrees = pol_degrees,
       mean = mean, sd = sd,
       given_ind = given_ind, omit_ind = omit_ind,
       tr_left = tr_left, tr_right = tr_right)
       
## Example demonstrating dhpaDiff function application.
## Let's approximate some three random variables (X1, X2, X3) joint density
## function at point (0,1, 0.2, 0.3) with Hermite polynomial of (1,2,3)
## degrees which polynomial coefficients equal 1 except coefficient related
## to x1*(x^3) polynomial element which equals 2. Also suppose that normal
## density related mean vector equals (1.1, 1.2, 1.3) while standard
## deviations vector is (2.1, 2.2, 2.3). In this example let's calculate
## density approximating function's gradient respect to various parameters

# Prepare initial values
x <- matrix(c(0.1, 0.2, 0.3), nrow = 1)
mean <- c(1.1, 1.2, 1.3)
sd <- c(2.1, 2.2, 2.3)
pol_degrees <- c(1, 2, 3)

# Create polynomial powers and indexes correspondence matrix
pol_ind <- polynomialIndex(pol_degrees)
# Set all polynomial coefficients to 1
pol_coefficients <- rep(1, ncol(pol_ind))
pol_degrees_n <- length(pol_degrees)

# Assign coefficient 2 to the polynomial element (x1 ^ 1)*(x2 ^ 0)*(x3 ^ 2)
pol_coefficients[apply(pol_ind, 2, function(x) all(x == c(1, 0, 2)))] <- 2

# Visualize correspondence between polynomial
# elements and their coefficients
as.data.frame(rbind(pol_ind, pol_coefficients),
              row.names = c("x1 power", "x2 power", 
                            "x3 power", "coefficients"),
              optional = TRUE)
printPolynomial(pol_degrees, pol_coefficients)

# Calculate density approximation gradient 
# respect to polynomial coefficients at point x
dhpaDiff(x = x,
         pol_coefficients = pol_coefficients, 
         pol_degrees = pol_degrees,
         mean = mean, sd = sd)
	
# Condition second component to be 0.5
# Substitute x second component with conditional value 0.5
x <- matrix(c(0.1, 0.5, 0.3), nrow = 1)
# Set TRUE to the second component indicating that it is conditioned
given_ind <- c(FALSE, TRUE, FALSE)

# Calculate conditional (on x2 = 0.5) density approximation's 
# gradient respect to polynomial coefficients at point x
dhpaDiff(x = x,
         pol_coefficients = pol_coefficients, 
         pol_degrees = pol_degrees,
         mean = mean, sd = sd,
         given_ind = given_ind)
	
# Consider third component marginal distribution
# conditioned on the second component 0.5 value
# Set TRUE to the first component indicating that it is omitted
omit_ind <- c(TRUE, FALSE, FALSE)

# Calculate conditional (on X2 = 0.5) marginal (for X3) density 
# approximation's gradient respect to: 
  # polynomial coefficients
dhpaDiff(x = x,
         pol_coefficients = pol_coefficients, 
         pol_degrees = pol_degrees,
         mean = mean, sd = sd,
         given_ind = given_ind, 
         omit_ind = omit_ind)
  # mean
dhpaDiff(x = x,
         pol_coefficients = pol_coefficients, 
         pol_degrees = pol_degrees,
         mean = mean, sd = sd,
         given_ind = given_ind, 
         omit_ind = omit_ind,
         type = "mean")
  # sd
dhpaDiff(x = x,
         pol_coefficients = pol_coefficients, 
         pol_degrees = pol_degrees,
         mean = mean, sd = sd,
         given_ind = given_ind, 
         omit_ind = omit_ind,
         type = "sd")
 # x
dhpaDiff(x = x,
         pol_coefficients = pol_coefficients, 
         pol_degrees = pol_degrees,
         mean = mean, sd = sd,
         given_ind = given_ind, 
         omit_ind = omit_ind,
         type = "x")
         
## Example demonstrating ihpaDiff function application.
## Let's approximate some three random variables (X1, X2, X3 ) joint interval 
## distribution function (intdf) at lower and upper points (0,1, 0.2, 0.3) 
## and (0,4, 0.5, 0.6) correspondingly with Hermite polynomial of (1, 2, 3) 
## degrees which polynomial coefficients equal 1 except coefficient 
## related to x1*(x^3) polynomial element which equals 2.
## Also suppose that normal density related mean vector equals 
## (1.1, 1.2, 1.3) while standard deviations vector is (2.1, 2.2, 2.3).
## In this example let's calculate interval distribution approximating 
## function gradient respect to polynomial coefficients.

# Prepare initial values
x_lower <- matrix(c(0.1, 0.2, 0.3), nrow=1)
x_upper <- matrix(c(0.4, 0.5, 0.6), nrow=1)
mean <- c(1.1, 1.2, 1.3)
sd <- c(2.1, 2.2, 2.3)
pol_degrees <- c(1, 2, 3)

# Create polynomial powers and indexes correspondence matrix
pol_ind <- polynomialIndex(pol_degrees)

# Set all polynomial coefficients to 1
pol_coefficients <- rep(1, ncol(pol_ind))
pol_degrees_n <- length(pol_degrees)

# Assign coefficient 2 to the polynomial element (x1 ^ 1)*(x2 ^ 0)*(x3 ^ 2)
pol_coefficients[apply(pol_ind, 2, function(x) all(x == c(1, 0, 2)))] <- 2

# Visualize correspondence between polynomial 
# elements and their coefficients
as.data.frame(rbind(pol_ind, pol_coefficients),
              row.names = c("x1 power", "x2 power", 
                            "x3 power", "coefficients"),
              optional = TRUE)
printPolynomial(pol_degrees, pol_coefficients)

# Calculate intdf approximation gradient respect to 
# polynomial coefficients at points x_lower and x_upper
ihpaDiff(x_lower = x_lower, x_upper = x_upper, 
         pol_coefficients = pol_coefficients, 
         pol_degrees = pol_degrees,
         mean = mean, sd = sd)
	
# Condition second component to be 0.7
# Substitute x second component with conditional value 0.7
x_upper <- matrix(c(0.4, 0.7, 0.6), nrow = 1)

# Set TRUE to the second component indicating that it is conditioned
given_ind <- c(FALSE, TRUE, FALSE)

# Calculate conditional (on X2 = 0.5) intdf approximation
# respect to polynomial coefficients at points x_lower and x_upper
ihpaDiff(x_lower = x_lower, x_upper = x_upper,
         pol_coefficients = pol_coefficients, 
         pol_degrees = pol_degrees,
         mean = mean, sd = sd,
         given_ind = given_ind)
	
# Consider third component marginal distribution
# conditioned on the second component 0.7 value
# Set TRUE to the first component indicating that it is omitted
omit_ind <- c(TRUE, FALSE, FALSE)

# Calculate conditional (on X2 = 0.5) marginal (for X3) intdf approximation
# respect to:
  # polynomial coefficients
ihpaDiff(x_lower = x_lower, x_upper = x_upper,
         pol_coefficients = pol_coefficients, 
         pol_degrees = pol_degrees,
         mean = mean, sd = sd,
         given_ind = given_ind, omit_ind = omit_ind)
  # mean
ihpaDiff(x_lower = x_lower, x_upper = x_upper,
         pol_coefficients = pol_coefficients, 
         pol_degrees = pol_degrees,
         mean = mean, sd = sd,
         given_ind = given_ind, omit_ind = omit_ind,
         type = "mean")
  # sd
ihpaDiff(x_lower = x_lower, x_upper = x_upper,
         pol_coefficients = pol_coefficients, 
         pol_degrees = pol_degrees,
         mean = mean, sd = sd,
         given_ind = given_ind, omit_ind = omit_ind,
         type = "sd")
  # x_lower
ihpaDiff(x_lower = x_lower, x_upper = x_upper,
         pol_coefficients = pol_coefficients, 
         pol_degrees = pol_degrees,
         mean = mean, sd = sd,
         given_ind = given_ind, omit_ind = omit_ind,
         type = "x_lower")
  # x_upper
ihpaDiff(x_lower = x_lower, x_upper = x_upper,
          pol_coefficients = pol_coefficients, 
          pol_degrees = pol_degrees,
          mean = mean, sd = sd,
          given_ind = given_ind, omit_ind = omit_ind,
          type = "x_upper")
          
}
\references{
A. Gallant and D. W. Nychka (1987) <doi:10.2307/1913241>
}
