context("comorbidities")

test_that("try to induce c++ segfault bug", {
  #trigdat <- data.frame(visitId = c("visit1", "visit1"), icd9 = c("39891", "09320"))

  devnull <- icd9Comorbid(ahrqTestDat, ahrqComorbid, isShort = TRUE) # seems to run okay now
})

test_that("ahrq make sure all the children
            are listed in the saved data.", {
              for (i in ahrqComorbid) expect_equal(icd9ChildrenShort(i, onlyReal = FALSE), sort(i))
            })

test_that("Elixhauser make sure all the children
            are listed in the saved data.", {
              for (i in elixComorbid) expect_equal(icd9ChildrenShort(i, onlyReal = FALSE), sort(i))
            })

test_that("Quan Charlson make sure all the children
            are listed in the saved data.", {
              for (i in quanDeyoComorbid) expect_equal(icd9ChildrenShort(i, onlyReal = FALSE), sort(i))
            })

test_that("Quan Elixhauser make sure all the children
            are listed in the saved data.", {
              for (i in quanElixComorbid) expect_equal(icd9ChildrenShort(i, onlyReal = FALSE), sort(i))
            })

test_that("icd9 comorbidities are created correctly,
            and logical to binary conversion ok", {
              ptdf <- icd9Comorbid(icd9df = patientData, isShort = TRUE,
                                   icd9Mapping = ahrqComorbid,
                                   visitId = "visitId", return.df = TRUE)

              expect_equal(names(ptdf), c("visitId", names(ahrqComorbid)))

              expect_true(all(sapply(names(ahrqComorbid),
                                     function(x)
                                       class(ptdf[, x])) == "logical"))
              ptdflogical <- logicalToBinary(ptdf)
              expect_true(all(sapply(names(ahrqComorbid),
                                     function(x)
                                       class(ptdflogical[, x])) == "integer"))
              # do not expect all the rest of patient data to be returned - we
              # aren't responsible for aggregating other fields by visitId!
              expect_equal(dim(ptdf),
                           c(length(unique(patientData[["visitId"]])),
                             1 + length(ahrqComorbid)))
              expect_true(
                setequal(names(ptdf), c("visitId", names(ahrqComorbid))))
              expect_true(
                setequal(names(ptdflogical), c("visitId", names(ahrqComorbid))))

              expect_equal(
                logicalToBinary(data.frame(a = c("jack", "hayley"),
                                           b = c(TRUE, FALSE),
                                           f = c(TRUE, TRUE))),
                data.frame(a = c("jack", "hayley"),
                           b = c(1, 0),
                           f = c(1, 1))
              )
            })

test_that("ahrq icd9 mappings generated from the current generation code", {
  # same but from source data. Should be absolutely identical.
  expect_equal(ahrqComorbid,
               parseAhrqSas(condense = FALSE,
                            save = FALSE, returnAll = FALSE))
  # same but from source data. Should be absolutely identical.
  expect_identical(ahrqComorbidAll,
                   parseAhrqSas(condense = FALSE,
                                save = FALSE, returnAll = TRUE))
  expect_equivalent(icd9GetInvalidMappingShort(ahrqComorbid), list())
})

test_that("Quan Charlson icd9 mappings are all
            generated from the current generation code", {
              expect_identical(quanDeyoComorbid,
                               parseQuanDeyoSas(condense = FALSE, save = FALSE))
              expect_equivalent(icd9GetInvalidMappingShort(quanDeyoComorbid), list())
            })
test_that("Quan Elixhauser icd9 mappings are all
            generated from the current generation code", {
              expect_identical(quanElixComorbid,
                               parseQuanElix(condense = FALSE, save = FALSE))
              expect_equivalent(icd9GetInvalidMappingShort(quanElixComorbid), list())
            })
test_that("Elixhauser icd9 mappings are all
            generated from the current generation code", {

              expect_identical(elixComorbid,
                               parseElix(condense = FALSE, save = FALSE))
              expect_equivalent(icd9GetInvalidMappingShort(elixComorbid), list())
            })

test_that("can condense the big lists of comorbidities without errors", {
  # this is a useful test because the data weren't generated by just expanding
  # base ranges (which is how the condense works in reverse)
  for (onlyReal in c(TRUE, FALSE)) {
    ahrq <- lapply(ahrqComorbid, icd9CondenseToMajorShort, onlyReal)
    quanDeyo <- lapply(quanDeyoComorbid, icd9CondenseToMajorShort, onlyReal)
    quanElix <- lapply(quanElixComorbid,
                       icd9CondenseToMajorShort, onlyReal)
    elix <- lapply(elixComorbid, icd9CondenseToMajorShort, onlyReal)
    expect_is(ahrq, class = "list")
    expect_is(elix, class = "list")
    expect_is(quanDeyo, class = "list")
    expect_is(quanElix, class = "list")
    # the comorbidity mappings save in \code{data} should not be condensed.
    expect_that(ahrq,
                testthat::not(testthat::equals(ahrqComorbid)))
    expect_that(elix,
                testthat::not(testthat::equals(elixComorbid)))
    expect_that(quanDeyo,
                testthat::not(testthat::equals(quanDeyoComorbid)))
    expect_that(quanElix,
                testthat::not(testthat::equals(quanElixComorbid)))
  }
})


test_that("icd9Hierarchy as saved in data can be recreated", {
  skip("this is a very slow test with web lookup - 10 mins for one assertion")
  expect_equal(icd9GetChaptersHierarchy(save = FALSE),
               icd9::icd9Hierarchy)
})

# the following test is dependent on availability and consistency of
# http://www.icd9data.com because there is no machine readable CDC or CMS file
# with this data.
test_that("icd9Chapters, etc. as saved in data can be recreated", {
  #skip_on_cran() # and/or skip_on_travis()
  skip("does web look-ups - don't want to hammer the target web site")
  res <- parseIcd9Chapters(year = "2014", save = FALSE)
  expect_equal(res$icd9Chapters, icd9::icd9Chapters)
  expect_equal(res$icd9ChaptersSub, icd9::icd9ChaptersSub)
  expect_equal(res$icd9ChaptersMajor, icd9::icd9ChaptersMajor)
})

test_that("condense an ICD-9 code set to minimal group", {
  skip("TODO:  this test breaks because %i9s% now includes the last major, even if not all its child.")
  expect_equal(sort(icd9CondenseToMajorShort("98799" %i9s% "98901",
                                             onlyReal = FALSE)),
               sort(c("98799", "988", "98900", "98901")))
  expect_equal(sort(icd9CondenseToMajorShort("98799" %i9s% "98901",
                                             onlyReal = TRUE,
                                             dropNonReal = TRUE)),
               "988")
  expect_equal(
    sort(icd9CondenseToMajorShort("98799" %i9s% "98901",
                                  onlyReal = TRUE,
                                  dropNonReal = FALSE)),
    sort(
      # this list is the full range with these missing:
      #  c("9880", "9881", "9882", "9888", "9889")

      c("988", "98799", "988", "98800", "98801", "98802", "98803",
        "98804", "98805", "98806", "98807", "98808", "98809", "98810",
        "98811", "98812", "98813", "98814", "98815", "98816", "98817",
        "98818", "98819", "98820", "98821", "98822", "98823", "98824",
        "98825", "98826", "98827", "98828", "98829", "9883", "98830",
        "98831", "98832", "98833", "98834", "98835", "98836", "98837",
        "98838", "98839", "9884", "98840", "98841", "98842", "98843",
        "98844", "98845", "98846", "98847", "98848", "98849", "9885",
        "98850", "98851", "98852", "98853", "98854", "98855", "98856",
        "98857", "98858", "98859", "9886", "98860", "98861", "98862",
        "98863", "98864", "98865", "98866", "98867", "98868", "98869",
        "9887", "98870", "98871", "98872", "98873", "98874", "98875",
        "98876", "98877", "98878", "98879", "98880", "98881", "98882",
        "98883", "98884", "98885", "98886", "98887", "98888", "98889",
        "98890", "98891", "98892", "98893", "98894", "98895", "98896",
        "98897", "98898", "98899", "98900", "98901")
    ))
  # TODO: more tests
})


test_that("AHRQ interpretation at least returns something reasonable", {
  skip("defer")
  result <- parseAhrqSas(sasPath = system.file("extdata",
                                               "comformat2012-2013.txt", package="icd9"), save = FALSE)
  expect_that(result, is_a("list"))
  expect_true(length(result) > 10)
})

test_that("HTN subgroups all worked", {
  # pick one subcategory
  expect_true(all(ahrqComorbidAll$HTNPREG %in% ahrqComorbid[["HTNcx"]]))

  # and we didn't drop any:
  expect_true(all(ahrqComorbidAll$HTNCX %in% ahrqComorbid[["HTNcx"]]))
  expect_true(all(ahrqComorbidAll$CHF %in% ahrqComorbid[["CHF"]]))
  expect_true(all(ahrqComorbidAll$RENLFAIL %in% ahrqComorbid[["Renal"]]))

})

test_that("Charlson Deyo doesn't double count disease with two severities", {
  expect_false(any(quanDeyoComorbid[["Mild Liver Disease"]] %in%
                     quanDeyoComorbid[["Moderate or Severe Liver Disease"]] ))
  expect_false(any(quanDeyoComorbid[["Cancer"]] %in%
                     quanDeyoComorbid[["Metastatic Carcinoma"]] ))
  expect_false(any(quanDeyoComorbid[["Diabetes without complications"]] %in%
                     quanDeyoComorbid[["Diabetes with complications"]] ))
})

test_that("Elixhauser doesn't double count disease with multiple severities", {
  expect_false(any(quanElixComorbid[["dm.uncomp"]] %in%
                     quanElixComorbid[["dm.comp"]] ))
  expect_false(any(quanElixComorbid[["solid.tumor"]] %in%
                     quanElixComorbid[["mets"]] ))
  expect_false(any(elixComorbid[["dm.uncomp"]] %in%
                     elixComorbid[["dm.comp"]] ))
  expect_false(any(elixComorbid[["solid.tumor"]] %in%
                     elixComorbid[["mets"]] ))
  expect_false(any(ahrqComorbid[["DM"]] %in% ahrqComorbid[["DMCX"]] ))
  expect_false(any(ahrqComorbid[["TUMOR"]] %in% ahrqComorbid[["METS"]] ))
})

test_that("ICD-9 codes from the original sources do appear in my data", {
  # these tests demonstrate that the interpreted data is correctly transcribed
  # in cases where the data is structured differently, and also affirms that
  # 'child' codes are included in the RData mappings in the package. E.g. if the
  # mapping specifies "044", we do expect 111 total codes to be in the mapping
  # 0440 04400 04401 etc. Ahrq
  expect_true("3337" %in% ahrqComorbid$NeuroOther) # single value
  expect_true("33370" %in% ahrqComorbid$NeuroOther) # single value sub-code
  expect_true("494" %in% ahrqComorbid$Pulmonary) # top-level at start of range
  expect_true("4940" %in% ahrqComorbid$Pulmonary) # value within range
  expect_true("49400" %in% ahrqComorbid$Pulmonary) # sub-value within range

  # Quan Deyo Charlson
  # top level single value
  expect_true("410" %in% quanDeyoComorbid$MI)
  # this is not included (410 and 412 defined)
  expect_false("411" %in% quanDeyoComorbid$MI)
  # this is not included (410 and 412 defined)
  expect_false("41199" %in% quanDeyoComorbid$MI)
  # midlevel value, not from range
  expect_true("4100" %in% quanDeyoComorbid$MI)
  # lower-level value, not from range
  expect_true("41001" %in% quanDeyoComorbid$MI)
  # midlevel definition
  expect_true("2504" %in% quanDeyoComorbid$DMcx)
  # midlevel definition lower-level code
  expect_true("25041" %in% quanDeyoComorbid$DMcx)

})

test_that("sample of ICD-9 codes from manually specified mappings do appear", {
  # the following tests cover the mappings in which there was no source SAS
  # data, but the numbers were transcribed manually. This is therefore testing a
  # little of the transcription, and also the elobration of codes definied in
  # ranges
  expect_true("2500" %in% quanElixComorbid$DM)
  expect_true("2501" %in% quanElixComorbid$DM)
  expect_true("25011" %in% quanElixComorbid$DM)
  expect_true("276" %in% quanElixComorbid$FluidsLytes)
  expect_true("2761" %in% quanElixComorbid$FluidsLytes)
  expect_true("27612" %in% quanElixComorbid$FluidsLytes)
  # top level should not be included automatically
  expect_false("710" %in% quanElixComorbid$FluidsLytes)
  expect_true("09320" %in% elixComorbid$Valvular)
  expect_true("3971" %in% elixComorbid$Valvular)
  expect_true("V560" %in% elixComorbid$Renal)
  expect_true("V1090" %in% elixComorbid$Tumor) # child at end of a V range
})

test_that("github #34 - short and long custom map give different results", {
  mydf <- data.frame(visitId = c("a","b","b","c"),
                     icd9 = c("1","010","10","20"))

  mymaps <- list(jack = c("1", "2", "3"), alf = c("010", "20"))
  mymapd <- lapply(mymaps, icd9ShortToDecimal)

  expect_identical(
    icd9Comorbid(mydf, icd9Mapping = mymaps, isShort = TRUE),
    icd9Comorbid(mydf, icd9Mapping = mymapd, isShort = FALSE))

})

test_that("no NA values in the co-morbidity lists", {
  expect_false(anyNA(unlist(unname(ahrqComorbid))))
  expect_false(anyNA(unlist(unname(ahrqComorbidAll))))
  expect_false(anyNA(unlist(unname(quanDeyoComorbid))))
  expect_false(anyNA(unlist(unname(quanElixComorbid))))
  expect_false(anyNA(unlist(unname(elixComorbid))))
})

test_that("built-in icd9 to comorbidity mappings are all valid", {
  expect_true(icd9IsValidMappingShort(ahrqComorbid))
  expect_true(icd9IsValidMappingShort(quanDeyoComorbid))
  expect_true(icd9IsValidMappingShort(quanElixComorbid))
  expect_true(icd9IsValidMappingShort(elixComorbid))
})

test_that("unordered visit ids", {
  pts <- data.frame(visitId = c("2", "1", "2", "3", "3"),
                    icd9 = c("39891", "40110", "09322", "41514", "39891"))
  icd9ComorbidShort(pts, ahrqComorbid)
})

test_that("diff comorbid works", {
  expect_error(icd9DiffComorbid(bad_input)) # list, but not list of character vectors
  expect_error(icd9DiffComorbid(bad_input, bad_input))

  # no warning or error for good data
  expect_that(res <- icd9DiffComorbid(ahrqComorbid, elixComorbid, show = FALSE), testthat::not(gives_warning()))
  expect_true(all(names(res) %in% c(
    "CHF", "Valvular", "PHTN", "PVD", "HTN", "HTNcx", "Paralysis",
    "NeuroOther", "Pulmonary", "DM", "DMcx", "Hypothyroid", "Renal",
    "Liver", "PUD", "HIV", "Lymphoma", "Mets", "Tumor", "Rheumatic",
    "Coagulopathy", "Obesity", "WeightLoss", "FluidsLytes", "BloodLoss",
    "Anemia", "Alcohol", "Drugs", "Psychoses", "Depression")))
  # one side diff
  expect_identical(res$Lymphoma[["only.x"]], character(0))
  # other side diff
  expect_identical(res$Drugs[["only.y"]], character(0))
  # match
  expect_identical(res$Depression[[2]], character(0))
  expect_identical(res$Depression[[3]], character(0))

  # both, also with elements in either side set diff
  expect_equal(res$PUD$both, c("53170", "53270", "53370", "53470"))

  expect_that(resq <- icd9DiffComorbid(quanElixComorbid, quanDeyoComorbid, show = TRUE),
              testthat::not(gives_warning()))
  capture.output(resq <- icd9DiffComorbid(quanElixComorbid, quanDeyoComorbid, show = TRUE))

})

twoPts <- data.frame(visitId = c("v01", "v01", "v02", "v02"),
                     icd9 = c("040", "000", "100", "000"),
                     stringsAsFactors = FALSE)
twoMap <- list("malady" = c("100", "2000"),
               "ailment" = c("003", "040"))
twoPtsFac <- data.frame(visitId = c("v01", "v01", "v02", "v02"),
                        icd9 = c("040", "000", "100", "000"),
                        stringsAsFactors = TRUE)
twoMapFac <- as.list(data.frame("malady" = c("100", "2000"),
                                "ailment" = c("003", "040"),
                                stringsAsFactors = TRUE))

test_that("comorbid quick test", {
  testres <- icd9Comorbid(twoPts, twoMap, return.df = TRUE)
  trueres <- data.frame("visitId" = c("v01", "v02"),
                        "malady" = c(FALSE, TRUE),
                        "ailment" = c(TRUE, FALSE),
                        stringsAsFactors = FALSE)
  expect_equal(testres, trueres)

  testmat <- icd9Comorbid(twoPts, twoMap, return.df = FALSE)
  truemat <- matrix(c(FALSE, TRUE, TRUE, FALSE), nrow=2,
                    dimnames = list(c("v01", "v02"), c("malady", "ailment")))
  expect_equal(testmat, truemat)

  testresfac <- icd9Comorbid(twoPtsFac, twoMapFac, return.df = TRUE)
  trueresfac <- data.frame("visitId" = c("v01", "v02"),
                           "malady" = c(FALSE, TRUE),
                           "ailment" = c(TRUE, FALSE),
                           stringsAsFactors = TRUE)
  expect_equal(testresfac, trueresfac)
  expect_equal(icd9Comorbid(twoPtsFac, twoMapFac), truemat)

})

test_that("control params don't affect result of comorbid calc", {
  pts <- randomPatients(101, 13)
  pts$visitId <- asCharacterNoWarn(pts$visitId)
  pts$icd9 <- as.factor(pts$icd9)
  upts <- length(unique(pts$visitId))
  ac <-  lapply(ahrqComorbid, function(x) {
    f <- factor(x, levels(pts[["icd9"]]))
    f[!is.na(f)]
  })
  expect_identical(
    icd9ComorbidShortCpp(pts, ac, threads = 1, chunkSize=32),
    icd9ComorbidShortCpp(pts, ac, threads = 3, chunkSize=32)
  )
  expect_identical(
    icd9ComorbidShortCpp(pts, ac, threads = 2, chunkSize=32),
    icd9ComorbidShortCpp(pts, ac, threads = 5, chunkSize=32)
  )
  expect_identical(
    icd9ComorbidShortCpp(pts, ac, threads = 3, chunkSize=1),
    icd9ComorbidShortCpp(pts, ac, threads = 3, chunkSize=32)
  )
  expect_identical(
    icd9ComorbidShortCpp(pts, ac, threads = 4, chunkSize = upts - 1),
    icd9ComorbidShortCpp(pts, ac, threads = 4, chunkSize=upts)
  )
  expect_identical(
    icd9ComorbidShortCpp(pts, ac, threads = 4, chunkSize = upts - 1),
    icd9ComorbidShortCpp(pts, ac, threads = 4, chunkSize = upts + 1)
  )
  expect_identical(
    icd9ComorbidShortCpp(pts, ac, threads = 4, chunkSize = upts + 1),
    icd9ComorbidShortCpp(pts, ac, threads = 4, chunkSize=upts)
  )
  expect_identical(
    icd9ComorbidShortCpp(pts, ac, threads = 3, chunkSize=upts - 2, ompChunkSize = 1),
    icd9ComorbidShortCpp(pts, ac, threads = 3, chunkSize=upts + 2, ompChunkSize = 1)
  )
  expect_identical(
    icd9ComorbidShortCpp(pts, ac, threads = 3, chunkSize=upts - 2, ompChunkSize = 11),
    icd9ComorbidShortCpp(pts, ac, threads = 3, chunkSize=upts + 2, ompChunkSize = 11)
  )
  expect_identical(
    icd9ComorbidShortCpp(pts, ac, threads = 3, chunkSize=upts, ompChunkSize = 1),
    icd9ComorbidShortCpp(pts, ac, threads = 3, chunkSize=upts, ompChunkSize = 11)
  )
  expect_identical(
    icd9ComorbidShortCpp(pts, ac),
    icd9ComorbidShortCpp(pts, ac, threads = 3, chunkSize=3, ompChunkSize = 5) # primes < unique visits
  )
})

test_that("failing example", {
  mydf <- data.frame(visitId = c("a", "b", "c"),
                     icd9 = c("441", "412.93", "044.9"))
  cmb <- icd9ComorbidQuanDeyo(mydf, isShort = FALSE, applyHierarchy = TRUE)
  expect_false("names" %in% names(attributes(cmb)))
  icd9Charlson(mydf, isShort = FALSE)
  icd9Charlson(mydf, isShort = FALSE, return.df = TRUE)
  icd9CharlsonComorbid(cmb)
})

test_that("disordered visitIds works by default", {
  set.seed(1441)
  dat <- transform(testTwenty, visitId = sample(visitId))
  cmp <-
    tres <- icd9Comorbid(dat, ahrqComorbid, icd9Field = "icd9Code")
  cres <- icd9Comorbid(testTwenty, ahrqComorbid, icd9Field = "icd9Code")
  expect_equal(dim(tres), dim(cres))
  expect_equal(sum(tres), sum(cres))
  expect_true(setequal(rownames(tres), rownames(cres)))
  expect_equal(colnames(tres), colnames(cres))

})
