% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Grid2Polygons.R
\name{Grid2Polygons}
\alias{Grid2Polygons}
\title{Convert Spatial Grids to Polygons}
\usage{
Grid2Polygons(grd, zcol = 1, level = FALSE, at, cuts = 20,
  pretty = FALSE, xlim = NULL, ylim = NULL, ply = NULL)
}
\arguments{
\item{grd}{'SpatialGridDataFrame'.
Spatial grid data frame}

\item{zcol}{'character' or 'integer'.
Attribute name or column number in attribute table.}

\item{level}{'logical'.
If true, a set of levels is used to partition the range of \code{z}, its default is false.}

\item{at}{'numeric'.
A vector giving breakpoints along the range of \code{z}.}

\item{cuts}{'integer'.
Number of levels the range of \code{z} would be divided into.}

\item{pretty}{'logical'.
Whether to use pretty cut locations.}

\item{xlim}{'numeric'.
Vector of length 2 giving left and right limits of the spatial grid, data outside these limits is excluded.}

\item{ylim}{'numeric'.
Vector of length 2 giving lower and upper limits of the spatial grid, data outside these limits is excluded.}

\item{ply}{'SpatialPolygons', or 'SpatialGridDataFrame'.
Cropping polygon}
}
\value{
Returns an object of 'SpatialPolygonsDataFrame'.
  The objects \code{data} slot is a data frame, number of rows equal to
  the number of \code{Polygons} objects and a single column containing values of \code{z}.
  If \code{level} is true, \code{z} values are set equal to the midpoint between breakpoints.
  The status of the polygon as a hole or an island is taken from the ring direction,
  with clockwise meaning island, and counter-clockwise meaning hole.
}
\description{
This function converts \pkg{sp} spatial objects from class '\code{\link{SpatialGridDataFrame}}' to '\code{\link{SpatialPolygonsDataFrame}}'.
Spatial polygons can then be transformed to a different projection or datum with \code{spTransform} in package \pkg{rgdal}.
Image files created with spatial polygons are reduced in size and result in a much "cleaner" version of your image.
}
\note{
The traditional R graphics model does not draw polygon holes correctly,
  holes overpaint their containing 'Polygon' object using a user defined background color (white by default).
  Polygon holes are now rendered correctly using the \code{plot} method for
  spatial polygons (\code{\link{SpatialPolygons-class}}), see \code{\link{polypath}} for more details.
  The Trellis graphics model appears to rely on the traditional method so
  use caution when plotting with \code{\link[sp]{spplot}}.

As an alternative, consider using the \code{\link[raster]{rasterToPolygons}} function
  in the \pkg{raster} package setting \code{dissolve = TRUE}.
}
\examples{
# Example 1

z <- c(1.1,  1.5,  4.2,  4.1,  4.3,  4.7,
       1.2,  1.4,  4.8,  4.8,   NA,  4.1,
       1.7,  4.2,  1.4,  4.8,  4.0,  4.4,
       1.1,  1.3,  1.2,  4.8,  1.6,   NA,
       3.3,  2.9,   NA,  4.1,  1.0,  4.0)
m <- 5
n <- 6
x <- rep(0:n, m + 1)
y <- rep(0:m, each = n + 1)
xc <- c(rep(seq(0.5, n - 0.5, by = 1), m))
yc <- rep(rev(seq(0.5, m - 0.5, by = 1)), each = n)
grd <- data.frame(z = z, xc = xc, yc = yc)
sp::coordinates(grd) <- ~ xc + yc
sp::gridded(grd) <- TRUE
grd <- as(grd, "SpatialGridDataFrame")
image(grd, col = gray.colors(30), axes = TRUE)
grid(col = "black", lty = 1)
points(x = x, y = y, pch = 16)
text(cbind(xc, yc), labels = z)
text(cbind(x = x + 0.1, y = rev(y + 0.1)), labels = 1:((m + 1) * (n + 1)), cex = 0.6)

at <- 1:ceiling(max(z, na.rm = TRUE))
plys <- Grid2Polygons(grd, level = TRUE, at = at)
cols <- rainbow(length(plys), alpha = 0.3)
sp::plot(plys, add = TRUE, col = cols)
zz <- plys[[1]]
legend("top", legend = zz, fill = cols, bty = "n", xpd = TRUE,
       inset = c(0, -0.1), ncol = length(plys))

p1 <- sp::Polygon(rbind(c(1.2, 0.5), c(5.8, 1.7), c(2.5, 5.1), c(1.2, 0.5)),
                  hole = FALSE)
p2 <- sp::Polygon(rbind(c(2.5, 2.5), c(3.4, 1.8), c(3.7, 3.1), c(2.5, 2.5)),
                  hole = TRUE)
p3 <- sp::Polygon(rbind(c(-0.3, 3.3), c(1.7, 5.1), c(-1.0, 7.0), c(-0.3, 3.3)),
                  hole = FALSE)
p <- sp::SpatialPolygons(list(sp::Polygons(list(p1, p2, p3), 1)))
plys <- Grid2Polygons(grd, level = TRUE, at = at, ply = p)
cols <- rainbow(length(zz), alpha = 0.6)[zz \%in\% plys[[1]]]
sp::plot(plys, col = cols, add = TRUE)

# Example 2

data(meuse.grid, package = "sp")
sp::coordinates(meuse.grid) <- ~ x + y
sp::gridded(meuse.grid) <- TRUE
meuse.grid <- as(meuse.grid, "SpatialGridDataFrame")
meuse.plys <- Grid2Polygons(meuse.grid, "dist", level = FALSE)
op <- par(mfrow = c(1, 2), oma = rep(0, 4), mar = rep(0, 4))
sp::plot(meuse.plys, col = heat.colors(length(meuse.plys)))
title("level = FALSE", line = -7)

meuse.plys.lev <- Grid2Polygons(meuse.grid, "dist", level = TRUE)
sp::plot(meuse.plys.lev, col = heat.colors(length(meuse.plys.lev)))
title("level = TRUE", line = -7)
par(op)

}
\references{
A general explanation of the algorithm provided
  \href{https://stackoverflow.com/questions/643995/algorithm-to-merge-adjacent-rectangles-into-polygon}{here};
  inspiration provided \href{https://menugget.blogspot.com/2012/04/create-polygons-from-matrix.html}{here}.
}
\seealso{
\code{\link{SpatialPolygons}}
}
\author{
J.C. Fisher, U.S. Geological Survey, Idaho Water Science Center
}
\keyword{manip}
