#' Output the result of Category level intensity analysis as csv.
#' @details Gets the output of \code{CIA} function and the path variable and generate a csv report called "CategoryLevelIntensityAnalysis.csv". The output will be stored in "CSVOutput" directory in path direction.
#' @param CIA.output Output list generated by \code{CIA} function.
#' @param time.points a charachter vector showing the time point of each raster layer in chronological order.
#' @param categories A charachter vector showing the categories in the map. Order of categories decided bases on the equivalent IDs in the raster attribute table.
#' @param filename A charachter variable including an optional path and a required filename to where the user wants to store the csv output. If only the name of the file is provided, it will be stored in the working directory.
#' @return The output is a CSV file.
#' @importFrom utils write.table
#' @importFrom stats na.omit
#' @export
#' @examples
#' raster_2005 <- raster::raster(system.file("external/RASTER_2005.RST", package="intensity.analysis"))
#' raster_2010 <- raster::raster(system.file("external/RASTER_2010.RST", package="intensity.analysis"))
#' raster_2012 <- raster::raster(system.file("external/RASTER_2012.RST", package="intensity.analysis"))
#' raster.layers <- list(raster_2005, raster_2010, raster_2012)
#' time.points <- c("2005","2010","2012")
#' categories <- c("Water","Trees","Impervious")
#' crosstabulation <- multicrosstab(raster.layers, time.points, categories)
#' CIA.output <- CIA(crosstabulation, time.points, categories)
#' filename <- file.path(normalizePath(tempdir(), winslash = "/"), "CIA.csv")
#' CIA2csv(CIA.output,time.points, categories, filename)



CIA2csv <- function(CIA.output,time.points, categories, filename){

  parameters <- reqpar(time.points)

  output <- chkfilename(filename, expand = TRUE)


  if (file.exists(output)){
    suppressWarnings(file.remove(output, showWarnings = FALSE ))
  }

  for (i in 1: as.integer(parameters$number.of.intervals)){
    # Making a data frame for output of category level analysis for each interval
    out.dataframe.1 <- cbind(CIA.output$Annual.Gross.Loss[[i]], CIA.output$Annual.Gross.Gain[[i]], CIA.output$Annual.Loss.Intensity[[i]], CIA.output$Annual.Gain.Intensity[[i]],
                             CIA.output$Uniform.Category.Intensity[i], CIA.output$Loss.Behavior[[i]], CIA.output$Gain.Behavior[[i]])

    colnames(out.dataframe.1) <- c("Gross Loss", "Gross Gain","Loss Intensity","Gain Intensity",
                                   "Uniform Category Intensity","Loss Behavior","Gain Behavior")

    rownames(out.dataframe.1) <- paste(categories)
    category.level.header <- paste("Category level Intensity Analysis for interval: ", as.character(parameters$initial.times[i]),
                                   as.character("-"),as.character(parameters$subsequent.times[i]),sep =" ")

    cat(category.level.header, "\n", file = output ,append = TRUE)
    write.table(out.dataframe.1, file = output , append = TRUE, col.names=NA ,quote = TRUE, sep = ",")
    cat("\n", file = output ,append = TRUE)
  }

}
