// Copyright (c) 2023 The InterpretML Contributors
// Licensed under the MIT license.
// Author: Paul Koch <code@koch.ninja>

#include "pch.hpp"

#include <stddef.h> // size_t, ptrdiff_t
#include <limits> // std::numeric_limits
#include <string.h> // strcpy, strchr, memmove, memcpy

#include "libebm.h"
#include "logging.h" // EBM_ASSERT
#include "unzoned.h" // LIKELY

#define ZONE_main
#include "zones.h"

#include "common.hpp" // IsConvertError

#include "ebm_internal.hpp" // FloatTickIncrement

namespace DEFINED_ZONE_NAME {
#ifndef DEFINED_ZONE_NAME
#error DEFINED_ZONE_NAME must be defined
#endif // DEFINED_ZONE_NAME

static constexpr double k_percentageDeviationFromEndpointForInterpretableNumbers = double{0.25};

INLINE_ALWAYS constexpr static size_t CountBase10CharactersAbs(int n) noexcept {
   // this works for negative numbers too
   return int{0} == n / int{10} ? size_t{1} : size_t{1} + CountBase10CharactersAbs(n / int{10});
}

// According to the C++ documentation, std::numeric_limits<double>::max_digits10 - 1 digits
// are required after the period in +9.1234567890123456e-301 notation, so for a double, the values would be
// 17 == std::numeric_limits<double>::max_digits10, and printf format specifier "%.16e"
static constexpr size_t k_cDigitsAfterPeriod = size_t{std::numeric_limits<double>::max_digits10} - size_t{1};

// Unfortunately, min_exponent10 doesn't seem to include subnormal numbers, so although it's the true
// minimum exponent in terms of the floating point exponential representation, it isn't the true minimum exponent
// when considering numbers converted into text.  To counter this, we add 1 extra digit.  For double numbers
// the largest exponent (+308), the smallest exponent for normal (-308), and the smallest exponent for subnormal (-324)
// all have 3 digits, but in the more general scenario we might go from N to N+1 digits, but I think
// it's really unlikely to go from N to N+2, since in the simplest case that would be a factor of 10 in the
// exponential term (if the low number was almost N and the high number was just a bit above N+2), and
// subnormal numbers shouldn't increase the exponent by that much ever.
static constexpr size_t k_cExponentMaxTextDigits =
      CountBase10CharactersAbs(std::numeric_limits<double>::max_exponent10);
static constexpr size_t k_cExponentMinTextDigits =
      CountBase10CharactersAbs(std::numeric_limits<double>::min_exponent10) + size_t{1};
static constexpr size_t k_cExponentTextDigits =
      k_cExponentMaxTextDigits < k_cExponentMinTextDigits ? k_cExponentMinTextDigits : k_cExponentMaxTextDigits;

// we have a function that ensures our output is exactly in the format that we require.  That format is:
// "+9.1234567890123456e-301" (this is when 16 == cDigitsAfterPeriod, the value for doubles)
// the exponential term can have some variation.  It can be any number of digits and the '+' isn't required
// our text float handling code handles these conditions without requiring modification.
// 3 characters for "+9."
// cDigitsAfterPeriod characters for the mantissa text
// 2 characters for "e-"
// cExponentTextDigits characters for the exponent text
// 1 character for null terminator
static constexpr size_t k_iExp = size_t{3} + k_cDigitsAfterPeriod;
static constexpr size_t k_cCharsFloatPrint = k_iExp + size_t{2} + k_cExponentTextDigits + size_t{1};

extern double ArithmeticMean(const double low, const double high) noexcept {
   // nan values represent missing, and are filtered out from our data prior to discretization
   EBM_ASSERT(!std::isnan(low));
   EBM_ASSERT(!std::isnan(high));

   // -infinity is converted to std::numeric_limits<double>::lowest() and
   // +infinity is converted to std::numeric_limits<double>::max() in our data prior to discretization
   EBM_ASSERT(!std::isinf(low));
   EBM_ASSERT(!std::isinf(high));

   EBM_ASSERT(low < high); // if two numbers were equal, we wouldn't put a cut point between them

   static_assert(std::numeric_limits<double>::is_iec559,
         "IEEE 754 gives us certain guarantees for floating point results that we use below");

   // this multiplication before addition format avoid overflows/underflows at the cost of a little more work.
   // IEEE 754 guarantees that 0.5 is representable as 2^(-1), so it has an exact representation.
   // IEEE 754 guarantees that division and addition give exactly rounded results.  Since we're multiplying by 0.5,
   // the internal representation will have the same mantissa, but will decrement the exponent, unless it underflows
   // to zero, or we have a subnormal number, which also works for reasons described below.
   // Fundamentally, the average can be equal to low if high is one epsilon tick above low.  If low is zero and
   // high is the smallest number, then both numbers divided by two are zero and the average is zero.
   // If low is the smallest number and high is one tick above that, low will go to zero on the division, but
   // high will become the smallest number since it uses powers of two, so the avg is again the
   // low value in this case.
   double avg = low * double{0.5} + high * double{0.5};

   EBM_ASSERT(!std::isnan(avg)); // in no reasonable implementation should this result in NaN

   // in theory, EBM_ASSERT(!std::isinf(avg)); should be ok, but there are bad IEEE 754 implementations that might
   // do the addition before the multiplication, which could result in overflow if done that way.

   // these should be correct in IEEE 754, even with floating point inexactness, due to "correct rounding"
   // in theory, EBM_ASSERT(low <= avg); AND EBM_ASSERT(avg < high); should be ok, but there are bad IEEE 754
   // implementations that might incorrectly implement "correct rounding", like the Intel x87 instructions

   // if our result is equal to low, then high should be guaranteed to be the next highest floating point number
   // in theory, EBM_ASSERT(low < avg || low == avg && std::nextafter(low, high) == high); should be ok, but
   // this depends on "correct rounding" which isn't true of all compilers

   if(UNLIKELY(avg <= low)) {
      // This check is required to handle the case where high is one epsilon higher than low, which means the average
      // could be low (the average could also be higher than low, but we don't need to handle that)
      // In that case, our only option is to make our cut equal to high, since we use lower bound inclusive semantics
      //
      // this check has the added benefit that if we have a compiler/platform that isn't truely IEEE 754 compliant,
      // which is sadly common due to double rounding and other issues, then we'd return high,
      // which is a legal value for us to cut on, and if we have values this close, it's appropriate to just return
      // high instead of doing a more exhaustive examination
      //
      // this check has the added advantage of checking for -infinity
      avg = high;
   }
   if(UNLIKELY(high < avg)) {
      // because so many compilers claim to be IEEE 754, but are not, we have this fallback to prevent us
      // from crashing due to unexpected outputs.  high is a legal value to return since we use lower bound
      // inclusivity.  I don't see how, even in a bad compiler/platform, we'd get a NaN result I'm not including it
      // here.  Some non-compliant platforms might get to +-infinity if they do the addition first then multiply
      // so that's one possibility to be wary about
      //
      // this check has the added advantage of checking for +infinity
      avg = high;
   }
   return avg;
}

INLINE_RELEASE_UNTEMPLATED static double GeometricMeanPositives(const double low, const double high) noexcept {
   // nan values represent missing, and are filtered out from our data prior to discretization
   EBM_ASSERT(!std::isnan(low));
   EBM_ASSERT(!std::isnan(high));

   // -infinity is converted to min_float and +infinity is converted to max_float in our data prior to discretization
   EBM_ASSERT(!std::isinf(low));
   EBM_ASSERT(!std::isinf(high));

   // we handle zeros outside of this function
   EBM_ASSERT(double{0} < low);
   EBM_ASSERT(double{0} < high);

   EBM_ASSERT(low < high);

   // in a reasonable world, with both low and high being non-zero, non-nan, non-infinity, and
   // positive values before calling log, log should return a non-overflowing or non-underflowing
   // value since all floating point values from -min to +max for floats give us reasonable log values.
   // Since our logs should average to a number that is between them, the exp value should result in a value
   // between them in almost all cases, so it shouldn't overflow or underflow either.  BUT, with floating
   // point jitter, we might get any of these scenarios.  This is a real corner case that we can presume
   // is very very very rare.

   double result = std::exp((std::log(low) + std::log(high)) * double{0.5});

   // IEEE 754 doesn't give us a lot of guarantees about log and exp.  They don't have have "correct rounding"
   // guarantees, unlike basic operators, so we could obtain results outside of our bounds, or perhaps
   // even overflow or underflow in a way that would lead to infinities.  I can't think of a way to get NaN
   // but who knows what's happening inside log, which would get NaN for zero and in a bad implementation
   // perhaps might return that for subnormal floats.
   //
   // If our result is not between low and high, then low and high should be very close and we can use the
   // arithmatic mean.  In the spirit of not trusting log and exp, we'll check for bad outputs and
   // switch to arithmatic mean.  In the case that we have nan or +-infinity, we aren't guaranteed that
   // low and high are close, so we can't really use an approach were we move small epsilon values in
   // our floats, so the artithmetic mean is really our only viable falllback in that case.
   //
   // Even in the fully compliant IEEE 754 case, result could be equal to low, so we do need to handle that
   // since we can't return the low value given we use lower bound inclusivity for cut points

   // checking the bounds also checks for +-infinity
   if(std::isnan(result) || result <= low || high < result) {
      result = ArithmeticMean(low, high);
   }
   return result;
}

static bool FloatToFullString(const double val, char* const str) noexcept {
   EBM_ASSERT(!std::isnan(val));
   EBM_ASSERT(!std::isinf(val));
   EBM_ASSERT(double{0} <= val);
   EBM_ASSERT(nullptr != str);

   // NOTE: str must be a buffer with k_cCharsFloatPrint characters available

   // the C++ standard is pretty good about harmonizing the "e" format.  There is some openess to what happens
   // in the exponent (2 or 3 digits with or without the leading sign character, etc).  If there is ever any
   // implementation observed that differs, this function should convert all formats to a common standard that
   // we use for string manipulation to find interpretable cut points, so we need all strings to have a common format

   // snprintf says to use the buffer size for the "n" term, but in alternate unicode versions it says # of characters
   // with the null terminator as one of the characters, so a string of 5 characters plus a null terminator would be 6.
   // For char strings, the number of bytes and the number of characters is the same.  I use number of characters for
   // future-proofing the n term to unicode versions, so n-1 characters other than the null terminator can fill
   // the buffer.  According to the docs, snprintf returns the number of characters that would have been written MINUS
   // the null terminator.

   static constexpr char g_pPrintfForRoundTrip[] = "%+.*le";

   const int cCharsWithoutNullTerminator =
         snprintf(str, k_cCharsFloatPrint, g_pPrintfForRoundTrip, int{k_cDigitsAfterPeriod}, val);
   if(cCharsWithoutNullTerminator < int{k_iExp + size_t{2}} || int{k_cCharsFloatPrint} <= cCharsWithoutNullTerminator) {
      // cCharsWithoutNullTerminator < iExp + 2 checks for both negative values returned and strings that are too short
      // we need the 'e' and at least one digit, so +2 is legal, and anything less is illegal
      return true;
   }
   char ch;
   ch = str[0];
   if('+' != ch) {
      return true;
   }
   ch = str[1];
   if(ch < '0' || '9' < ch) {
      return true;
   }
   ch = str[2];
   if('.' != ch) {
      return true;
   }
   char* pch = &str[3];
   char* pE = &str[k_iExp];
   do {
      ch = *pch;
      if(ch < '0' || '9' < ch) {
         return true;
      }
      ++pch;
   } while(pch != pE);
   ch = *pch;
   if('e' != ch && 'E' != ch) {
      return true;
   }

   // use strtol instead of atol in case we have a bad input.  atol has undefined behavior if the
   // number isn't representable as an int.  strtol returns a 0 with bad inputs, or LONG_MAX, or LONG_MIN,
   // on overflow or underflow.  The C++ standard makes clear though that on error strtol sets endptr
   // equal to str, so we can use that

   ++pch;
   char* endptr = pch; // set it to the error value so that even if the function doesn't set it we get an error
   // we use endptr to detect failure, so ignore the return value.  Use the (void)! trick to eliminate WARNINGs
   (void)!strtol(pch, &endptr, 10);
   if(endptr <= pch) {
      return true;
   }
   return false;
}

#if 0

// TODO: this entire section below!

// We have a problem in that converting from a float to a string has many possible legal outputs.
// IEEE-754 requires that if we output 17 to 20 digits for a double that the result can be converted from the
// string back to the original double, but no guarantees are made that 16 digits or 21 digits will work,
// and some language implementations output shorter strings like 0.25 which have exact representations in IEEE-754
// or strings like 0.1 when 0.1 converts back to the original float representation in their representation
// There are odd corner cases that rely on the type of rounding (IEEE-754 requires bankers' rounding for strings
// I believe).  There are legals and illegal ways to format IEEE-754 in text specifically:
// ISO 6093:1985 -> https://www.titanwolf.org/Network/q/4d680399-6711-4742-9900-74a42ad9f5d7/y
// 
// We desire cross-language identical results, so when we have cut points or categorical strings
// of floats we want these to be identical between languages for both converting to strings and when strings
// are converted back to floats.  This also applies to serialization to JSON and other text.  To support this
// we implement our own converters that are guararanteed to be identical between languages.  
// 
// The gold standard for float/string conversion is this: https://www.netlib.org/fp/dtoa.c
// It is used in python: https://github.com/python/cpython/blob/main/Python/dtoa.c
// Microsoft Edge for Android uses it: https://www.microsoft.com/en-us/legal/products/notices/msedgeandroid
// Java uses a port of this made to the Java language
// Other languages also use this code, but not any C++ built-in libraries yet.
// Some languages don't round trip properly with less than 17 digits.
// Some languages are buggy and don't correctly round when outputting 17 digits.
//
// This implementation will progressively shorten the string until it reaches the point where the conversion
// back won't be identical which is a stronger guarantee than IEEE-754.
// 
// Outputting 17 digits should be guaranteed to have a unique conversion back to float, but when shortening
// to 16 digits there are oddities where incrementing the 16th digit upwards yields a good result but
// chopping the 17th digit doesn't work.  I assume there are numbers where both chopping the 17th digit and
// either moving the 16th digit up or keeping it the same yield the same result, so we need a consistent
// policy with regards to the 16th digit.  For the 15th digit we can always chop since there are no numbers
// where moving the 15th digit up yields the same number. One example is:
// 2e-44 which is 5.684341886080801486968994140625e-14.  Rounded to 15 digits (5.68434188608080e-14) doesn't work.
// Rounding down to 16 digits down doesn't work (5.684341886080801e-14), but rounding up to (5.684341886080802e-14) does work.
// as described in: https://www.exploringbinary.com/the-shortest-decimal-string-that-round-trips-may-not-be-the-nearest/

// for cut points we should always use float64 values since that gives us the best resolution, and our caller
// could have float64 values or float32 values and cuts points that are float64 can work on both of them
// Also, float64 is the most cross-language compatible format, and in JSON it's the only option.

// for scores we should always use float64 values.  Unlike cut points, we'll be using float32 internally within
// the booster, BUT we only need to turn scores into text for serialization to JSON, and JSON only supports
// float64 values, so we need to output that.  We should get 100% reproducibility by turning float32 scores
// into float64, then text, then back to float64, then back to float32, so this is fine.  The only thing
// we loose is a bit of simplicity since our JSON scores will have more digits, but we don't have the equivalent
// of rounded cuts anyways, so the scores will have as many decimals as we get via boosting anywyas

// lastly, since there are no integers in JSON (everything is a double), we should eliminate the difference
// between float64 and integers when numbers can be represented as unique integers.  WE should convert the float
// 4.0 therefore to "4" for any number that meets the criteria: "floor(x) == x && abs(x) <= SAFE_FLOAT64_AS_INT64_MAX"

extern IntEbm GetCountCharactersPerFloat() {
   // for calling FloatsToStrings the caller needs to allocate this many bytes per float in the string buffer
   // after every float is either a space separator or a null-terminator
   return k_cCharsFloatPrint;
}

extern ErrorEbm FloatsToString(IntEbm count, const double * vals, char * str) {
   // TODO: implement this:
   // 
   // This code takes an array of floats and converts them to a single string separated by spaces and a null-terminator
   // at the end
}

extern ErrorEbm StringToFloats(const char * str, double * vals) {
   // TODO: implement this:
   //
   // This code takes a single string with the floats separated by spaces and a null-terminator at the end
   // and converts these into an array of floats.  The caller had better be carefull in allocating, but they
   // should know how many float values they put into the string so they should know how many values they'll
   // get back and therefore how big to make the buffer
}

#endif

INLINE_RELEASE_UNTEMPLATED static long GetExponent(const char* const str) noexcept {
   // we previously checked that this converted to a long in FloatToFullString
   return strtol(&str[k_iExp + size_t{1}], nullptr, int{10});
}

static double StringToFloatWithFixup(const char* const str, const size_t iIdenticalCharsRequired) noexcept {
   // TODO: this is misguided... python shortens floating point numbers to the shorted string when printing numbers
   // and using nextafter to get to all zeros makes python, and other languages that do the same have a bunch of
   // zeros and a long string.  So, if 1.6999999999999994 rounds to 1.7 nicely in python, don't use nextafter
   // below to convert it to 1.7000000000000009 since then python will need to output the long string
   // to avoid ambiguity since 1.7 will convert to 1.6999999999999994 and not 1.7000000000000009
   // see: https://www.exploringbinary.com/the-shortest-decimal-string-that-round-trips-may-not-be-the-nearest/

   char strRehydrate[k_cCharsFloatPrint];

   // we only convert str values that we've verified to conform, OR chopped versions of these which we know to be legal
   // If the chopped representations underflow (possible on chopping to lower) or
   // overflow (possible when we increment from the lower chopped value), then strtod gives
   // us enough information to convert these

   // the documentation says that if we have an underflow or overflow, strtod returns us +-HUGE_VAL, which is
   // +-infinity for at least some implementations.  We can't really take a ratio from those numbers, so convert
   // this to the lowest and max values

   // TODO: switch over to using our better ConvertStringToFloat function now!
   double ret = strtod(str, nullptr);

   // this is a check for -infinity/-HUGE_VAL, without the -infinity value since some compilers make that illegal
   // even so far as to make isinf always FALSE with some compiler flags
   // include the equals case so that the compiler is less likely to optimize that out
   ret = ret <= std::numeric_limits<double>::lowest() ? std::numeric_limits<double>::lowest() : ret;
   // this is a check for +infinity/HUGE_VAL, without the +infinity value since some compilers make that illegal
   // even so far as to make isinf always FALSE with some compiler flags
   // include the equals case so that the compiler is less likely to optimize that out
   ret = std::numeric_limits<double>::max() <= ret ? std::numeric_limits<double>::max() : ret;

   if(FloatToFullString(ret, strRehydrate)) {
      return ret;
   }

   if(0 == memcmp(str, strRehydrate, iIdenticalCharsRequired * sizeof(*str))) {
      return ret;
   }

   EBM_ASSERT('+' == str[0]);

   if(std::numeric_limits<double>::max() != ret) {
      ret = FloatTickIncrement(ret);
   }

   return ret;
}

INLINE_ALWAYS static char* strcpy_NO_WARNINGS(char* const dest, const char* const src) EBM_NOEXCEPT {
   StopClangAnalysis();
   return strcpy(dest, src);
}

static bool StringToFloatChopped(const char* const pStr,
      size_t iTruncateMantissaTextDigitsAfterFirstDigit,
      double* const pLowChopOut,
      double* const pHighChopOut) noexcept {
   // the lowChopOut returned can be equal to highChopOut if pStr is an overflow

   // when iTruncateMantissaTextDigitsAfterFirstDigit is zero we chop anything after the first digit, so
   // 3.456789*10^4 -> 3*10^4 when iTruncateMantissaTextDigitsAfterFirstDigit == 0
   // 3.456789*10^4 -> 3.4*10^4 when iTruncateMantissaTextDigitsAfterFirstDigit == 1

   EBM_ASSERT(nullptr != pStr);
   EBM_ASSERT('+' == pStr[0]);
   // don't pass us a non-truncated string, since we should handle anything that gets to that level differently
   EBM_ASSERT(iTruncateMantissaTextDigitsAfterFirstDigit < k_cDigitsAfterPeriod);

   char strTruncated[k_cCharsFloatPrint];

   // eg: "+9.1234567890123456e-301"
   size_t iTruncateTextAfter = size_t{0} == iTruncateMantissaTextDigitsAfterFirstDigit ?
         size_t{2} :
         iTruncateMantissaTextDigitsAfterFirstDigit + size_t{3};

   memcpy(strTruncated, pStr, iTruncateTextAfter * sizeof(*pStr));
   strcpy_NO_WARNINGS(&strTruncated[iTruncateTextAfter], &pStr[k_iExp]);

   if(PREDICTABLE(nullptr != pLowChopOut)) {
      *pLowChopOut = StringToFloatWithFixup(strTruncated, iTruncateTextAfter);
   }
   if(PREDICTABLE(nullptr != pHighChopOut)) {
      char* pDigit = &strTruncated[iTruncateTextAfter - size_t{1}];
      char ch;
      if(size_t{2} == iTruncateTextAfter) {
         goto start_at_top;
      }
      while(true) {
         ch = *pDigit;
         if('.' == ch) {
            --pDigit;
         start_at_top:;
            EBM_ASSERT(strTruncated + size_t{1} == pDigit);
            ch = *pDigit;
            if('9' == ch) {
               // oh, great.  now we need to increment our exponential
               int exponent = GetExponent(pStr) + int{1};
               *pDigit = '1';
               *(pDigit + size_t{1}) = 'e';

               static constexpr char g_pPrintfLongInt[] = "%+d";
               // for the size -> one for the '+' or '-' sign, k_cExponentTextDigits for the digits, 1 for null
               // terminator
               int cCharsWithoutNullTerminator = snprintf(
                     pDigit + size_t{2}, size_t{1} + k_cExponentTextDigits + size_t{1}, g_pPrintfLongInt, exponent);
               if(cCharsWithoutNullTerminator <= int{1} ||
                     int{size_t{1} + k_cExponentTextDigits} < cCharsWithoutNullTerminator) {
                  return true;
               }
               // we don't have all those '9' characters anymore to check.  we just need the 1
               iTruncateTextAfter = size_t{2};
            } else {
               EBM_ASSERT('0' <= ch && ch <= '8');
               *pDigit = ch + char{1};
            }
            break;
         } else if('9' == ch) {
            *pDigit = '0';
            --pDigit;
         } else {
            EBM_ASSERT('0' <= ch && ch <= '8');
            *pDigit = ch + char{1};
            break;
         }
      }
      *pHighChopOut = StringToFloatWithFixup(strTruncated, iTruncateTextAfter);
   }
   return false;
}

extern double GetInterpretableCutPointFloat(double low, double high) noexcept {
   // TODO : add logs or asserts here when we find a condition we didn't think was possible, but that occurs

   // nan values represent missing, and are filtered out from our data prior to discretization
   EBM_ASSERT(!std::isnan(low));
   EBM_ASSERT(!std::isnan(high));

   // -infinity is converted to std::numeric_limits<double>::lowest() and
   // +infinity is converted to std::numeric_limits<double>::max() in our data prior to discretization
   EBM_ASSERT(!std::isinf(low));
   EBM_ASSERT(!std::isinf(high));

   EBM_ASSERT(low < high); // if two numbers were equal, we wouldn't put a cut point between them
   EBM_ASSERT(low < std::numeric_limits<double>::max());
   EBM_ASSERT(std::numeric_limits<double>::lowest() < high);

   // if our numbers pass the asserts above, all combinations of low and high values can get a legal cut point,
   // since we can always return the high value given that our binning is lower bound inclusive

   double lowChop;
   double highChop;
   char strAvg[k_cCharsFloatPrint];
   double ret;

   bool bNegative = false;
   if(UNLIKELY(low <= double{0})) {
      if(UNLIKELY(double{0} == low)) {
         EBM_ASSERT(double{0} < high);
         // half of any number should give us something with sufficient distance.  For instance probably the worse
         // number would be something like 1.999999999999*10^1 where the division by two might round up to
         // 1.000000000000*10^1.  In that case though, we'll find that 1*10^1 is closest to the average, and we'll
         // choose that instead of the much farther away 2.000*10^1

         const double avg = high * double{0.5};
         EBM_ASSERT(!std::isnan(avg));
         EBM_ASSERT(!std::isinf(avg));
         EBM_ASSERT(double{0} <= avg);
         ret = high;
         // check for underflow
         if(LIKELY(double{0} != avg)) {
            ret = avg;
            if(LIKELY(!FloatToFullString(ret, strAvg)) &&
                  LIKELY(!StringToFloatChopped(strAvg, 0, &lowChop, &highChop))) {
               EBM_ASSERT(!std::isnan(lowChop));
               EBM_ASSERT(!std::isinf(lowChop));
               // it's possible we could have chopped off digits such that we round down to zero
               EBM_ASSERT(double{0} <= lowChop);
               EBM_ASSERT(lowChop <= ret);
               // check for underflow from digit chopping.  If this happens avg/high must be pretty close to zero
               if(LIKELY(double{0} != lowChop)) {
                  EBM_ASSERT(!std::isnan(highChop));
                  EBM_ASSERT(!std::isinf(highChop));
                  EBM_ASSERT(double{0} < highChop);
                  EBM_ASSERT(ret <= highChop);

                  const double highDistance = highChop - ret;
                  EBM_ASSERT(!std::isnan(highDistance));
                  EBM_ASSERT(!std::isinf(highDistance));
                  EBM_ASSERT(double{0} <= highDistance);
                  const double lowDistance = ret - lowChop;
                  EBM_ASSERT(!std::isnan(lowDistance));
                  EBM_ASSERT(!std::isinf(lowDistance));
                  EBM_ASSERT(double{0} <= lowDistance);

                  ret = UNPREDICTABLE(highDistance <= lowDistance) ? highChop : lowChop;
               }
            }
         }

         EBM_ASSERT(!std::isnan(ret));
         EBM_ASSERT(!std::isinf(ret));
         EBM_ASSERT(low < ret);
         EBM_ASSERT(ret <= high);

         return ret;
      }

      if(UNLIKELY(double{0} <= high)) {
         // if low is negative and high is zero or positive, a natural cut point is zero.  Also, this solves the issue
         // that we can't take the geometric mean of mixed positive/negative numbers.  This works since we use
         // lower bound inclusivity, so a cut point of 0 will include the number 0 in the upper bin.  Normally we try
         // to avoid putting a cut directly on one of the numbers, but in the case of zero it seems appropriate.
         ret = double{0};
         if(UNLIKELY(double{0} == high)) {
            // half of any number should give us something with sufficient distance.  For instance probably the worse
            // number would be something like 1.999999999999*10^1 where the division by two might round up to
            // 1.000000000000*10^1.  In that case though, we'll find that 1*10^1 is closest to the average, and we'll
            // choose that instead of the much farther away 2.000*10^1

            ret = low * double{-0.5};
            EBM_ASSERT(!std::isnan(ret));
            EBM_ASSERT(!std::isinf(ret));
            EBM_ASSERT(double{0} <= ret);

            if(LIKELY(!FloatToFullString(ret, strAvg)) &&
                  LIKELY(!StringToFloatChopped(strAvg, 0, &lowChop, &highChop))) {
               EBM_ASSERT(!std::isnan(lowChop));
               EBM_ASSERT(!std::isinf(lowChop));
               // it's possible we could have chopped off digits such that we round down to zero
               EBM_ASSERT(double{0} <= lowChop);
               EBM_ASSERT(lowChop <= ret);

               EBM_ASSERT(!std::isnan(highChop));
               EBM_ASSERT(!std::isinf(highChop));
               EBM_ASSERT(double{0} < highChop);
               EBM_ASSERT(ret <= highChop);

               const double highDistance = highChop - ret;
               EBM_ASSERT(!std::isnan(highDistance));
               EBM_ASSERT(!std::isinf(highDistance));
               EBM_ASSERT(double{0} <= highDistance);
               const double lowDistance = ret - lowChop;
               EBM_ASSERT(!std::isnan(lowDistance));
               EBM_ASSERT(!std::isinf(lowDistance));
               EBM_ASSERT(double{0} <= lowDistance);

               ret = UNPREDICTABLE(highDistance <= lowDistance) ? highChop : lowChop;
            }
            ret = -ret;
         }

         EBM_ASSERT(!std::isnan(ret));
         EBM_ASSERT(!std::isinf(ret));
         EBM_ASSERT(low < ret);
         EBM_ASSERT(ret <= high);

         return ret;
      }

      const double tmpLow = low;
      low = -high;
      high = -tmpLow;
      bNegative = true;
   } else {
      EBM_ASSERT(double{0} < high);
   }

   EBM_ASSERT(double{0} < low);
   EBM_ASSERT(double{0} < high);
   EBM_ASSERT(low < high);
   EBM_ASSERT(low < std::numeric_limits<double>::max());
   EBM_ASSERT(high <= std::numeric_limits<double>::max());

   // divide by high since it's guaranteed to be bigger than low, so we can't blow up to infinity
   const double ratio = low / high;
   EBM_ASSERT(!std::isnan(ratio));
   EBM_ASSERT(!std::isinf(ratio));
   EBM_ASSERT(ratio <= double{1});
   EBM_ASSERT(double{0} <= ratio);

   // don't transition on a perfect 1000 ratio from arithmetic to geometric mean since many of our numbers
   // are probably going to be whole numbers and we don't want floating point inexactness to dictate the
   // transition, so choose a number just slightly lower than 1000, in this case 996.18959224497322090157279627358
   if(ratio < double{0.001003824982498}) {
      ret = GeometricMeanPositives(low, high);
      EBM_ASSERT(!std::isnan(ret));
      EBM_ASSERT(!std::isinf(ret));
      EBM_ASSERT(low < ret);
      EBM_ASSERT(ret <= high);

      if(LIKELY(LIKELY(!FloatToFullString(ret, strAvg)) &&
               LIKELY(!StringToFloatChopped(strAvg, size_t{0}, &lowChop, &highChop)))) {
         // avg / low == high / avg (approximately) since it's the geometric mean
         // the lowChop or highChop side that is closest to the average will be farthest away
         // from it's corresponding low/high value
         // since we don't want infinties, we divide the smaller number by the bigger one
         // the smallest number means it has the longest distance from the low/high value, hense it's closer
         // to the average

         EBM_ASSERT(low < lowChop);
         const double lowRatio = low / lowChop;
         EBM_ASSERT(!std::isnan(lowRatio));
         EBM_ASSERT(!std::isinf(lowRatio));
         EBM_ASSERT(lowRatio <= double{1});
         EBM_ASSERT(double{0} <= lowRatio);

         EBM_ASSERT(highChop < high);
         const double highRatio = highChop / high;
         EBM_ASSERT(!std::isnan(highRatio));
         EBM_ASSERT(!std::isinf(highRatio));
         EBM_ASSERT(highRatio <= double{1});
         EBM_ASSERT(double{0} <= highRatio);

         ret = UNPREDICTABLE(lowRatio <= highRatio) ? lowChop : highChop;
      }
   } else {
      ret = ArithmeticMean(low, high);
      EBM_ASSERT(!std::isnan(ret));
      EBM_ASSERT(!std::isinf(ret));
      EBM_ASSERT(low < ret);
      EBM_ASSERT(ret <= high);

      char strLow[k_cCharsFloatPrint];
      char strHigh[k_cCharsFloatPrint];
      if(LIKELY(LIKELY(!FloatToFullString(low, strLow)) && LIKELY(!FloatToFullString(high, strHigh)) &&
               LIKELY(!FloatToFullString(ret, strAvg)))) {
         size_t iTruncateMantissa = size_t{0};
         do {
            double lowHigh;
            double avgLow;
            double avgHigh;
            double highLow;

            if(UNLIKELY(StringToFloatChopped(strLow, iTruncateMantissa, nullptr, &lowHigh))) {
               break;
            }
            if(UNLIKELY(StringToFloatChopped(strAvg, iTruncateMantissa, &avgLow, &avgHigh))) {
               break;
            }
            if(UNLIKELY(StringToFloatChopped(strHigh, iTruncateMantissa, &highLow, nullptr))) {
               break;
            }

            if(lowHigh < avgLow && avgLow < highLow && low < avgLow && avgLow <= high) {
               // avgLow is a possibility
               if(lowHigh < avgHigh && avgHigh < highLow && low < avgHigh && avgHigh <= high) {
                  // avgHigh is a possibility
                  const double lowDistanceToAverage = ret - avgLow;
                  const double highDistanceToAverage = avgHigh - ret;
                  EBM_ASSERT(-0.000001 < lowDistanceToAverage);
                  EBM_ASSERT(-0.000001 < highDistanceToAverage);
                  if(UNPREDICTABLE(highDistanceToAverage < lowDistanceToAverage)) {
                     ret = avgHigh;
                     break;
                  }
               }
               ret = avgLow;
               break;
            } else {
               if(lowHigh < avgHigh && avgHigh < highLow && low < avgHigh && avgHigh <= high) {
                  // avgHigh works!
                  ret = avgHigh;
                  break;
               }
            }

            ++iTruncateMantissa;
         } while(k_cDigitsAfterPeriod != iTruncateMantissa);
      }
   }
   if(PREDICTABLE(bNegative)) {
      ret = -ret;
   }
   return ret;
}

extern double GetInterpretableEndpoint(const double center, const double movementFromEnds) noexcept {
   // TODO : add logs or asserts here when we find a condition we didn't think was possible, but that occurs

   // center can be -infinity OR +infinity
   // movementFromEnds can be +infinity

   EBM_ASSERT(!std::isnan(center));
   EBM_ASSERT(!std::isnan(movementFromEnds));
   EBM_ASSERT(double{0} <= movementFromEnds);

   double ret = center;
   // if the center is +-infinity then we'll always be farter away than the end cut points which can't be +-infinity
   // so return +-infinity so that our alternative cut point is rejected
   if(LIKELY(!std::isinf(ret))) {
      // we use movementFromEnds to compute center, so if movementFromEnd was an infinity, then center would be
      // an infinity value.  We filter out infinity values for center above though, so movementFromEnds can't be
      // infinity here, even though the
      EBM_ASSERT(!std::isinf(movementFromEnds));

      const double distance = k_percentageDeviationFromEndpointForInterpretableNumbers * movementFromEnds;
      EBM_ASSERT(!std::isnan(distance));
      EBM_ASSERT(!std::isinf(distance));
      EBM_ASSERT(double{0} <= distance);

      bool bNegative = false;
      if(PREDICTABLE(ret < double{0})) {
         ret = -ret;
         bNegative = true;
      }

      const double lowBound = ret - distance;
      EBM_ASSERT(!std::isnan(lowBound));
      // lowBound can be a negative number, but can't be +-infinity because we subtract from a positive number
      // and we use IEEE 754
      EBM_ASSERT(!std::isinf(lowBound));

      const double highBound = ret + distance;
      // highBound can be +infinity, but can't be negative
      EBM_ASSERT(!std::isnan(highBound));
      EBM_ASSERT(double{0} <= highBound);

      char str[k_cCharsFloatPrint];
      if(LIKELY(!FloatToFullString(ret, str))) {
         size_t iTruncateMantissa = size_t{0};
         do {
            double lowChop;
            double highChop;

            if(UNLIKELY(StringToFloatChopped(str, iTruncateMantissa, &lowChop, &highChop))) {
               break;
            }

            // these comparisons works even if lowBound is negative or highBound is +infinity
            EBM_ASSERT(!std::isinf(lowChop));
            EBM_ASSERT(!std::isinf(highChop));
            if(lowBound <= lowChop && lowChop <= highBound) {
               // lowChop is a possibility
               if(lowBound <= highChop && highChop <= highBound) {
                  // highChop is a possibility
                  const double lowDistanceToAverage = ret - lowChop;
                  const double highDistanceToAverage = highChop - ret;
                  EBM_ASSERT(-0.000001 < lowDistanceToAverage);
                  EBM_ASSERT(-0.000001 < highDistanceToAverage);
                  if(UNPREDICTABLE(highDistanceToAverage < lowDistanceToAverage)) {
                     ret = highChop;
                     break;
                  }
               }
               ret = lowChop;
               break;
            } else {
               if(lowBound <= highChop && highChop <= highBound) {
                  // highChop works!
                  ret = highChop;
                  break;
               }
            }

            ++iTruncateMantissa;
         } while(k_cDigitsAfterPeriod != iTruncateMantissa);
      }
      if(bNegative) {
         ret = -ret;
      }
   }
   return ret;
}

extern size_t RemoveMissingValsAndReplaceInfinities(const size_t cSamples, double* const aVals) noexcept {
   EBM_ASSERT(size_t{1} <= cSamples);
   EBM_ASSERT(nullptr != aVals);

   // In most cases we believe that for graphing the caller should only need the bin cuts that we'll eventually
   // return, and they'll want to position the graph to include the first and last cuts, and have a little bit of
   // space both above and below those cuts.  In most cases they shouldn't need the non-infinity min/max values or know
   // whether or not there is +-infinity in the data, BUT on the margins of choosing graphing it might be useful.
   // For example, if the first cut was at 0.1 it might be reasonable to think that the low boundary should be 0,
   // and that would be reasonable if the lowest true value was 0.01, but if the lowest value was actually -0.1,
   // then we might want to instead make our graph start at -1.  Likewise, knowing if there were +-infinity
   // values in the data probably won't affect the bounds shown, but perhaps the graphing code might want to
   // somehow indicate the existance of +-infinity values.  The user might write custom graphing code, so we should
   // just return all this information and let the user choose what they want.

   // we really don't want to have cut points that are either -infinity or +infinity because these values are
   // problematic for serialization, cross language compatibility, human understantability, graphing, etc.
   // In some cases though, +-infinity might carry some information that we do want to capture.  In almost all
   // cases though we can put a cut point between -infinity and the smallest value or +infinity and the largest
   // value.  One corner case is if our data has both max_float and +infinity values.  Our binning uses
   // lower inclusive bounds, so a cut value of max_float will include both max_float and +infinity, so if
   // our algorithm decides to put a cut there we'd be in trouble.  We don't want to make the cut +infinity
   // since that violates our no infinity cut point rule above.  A good compromise is to turn +infinity
   // into max_float.  If we do it here, our cutting algorithm won't need to deal with the odd case of indicating
   // a cut and removing it later.  In theory we could separate -infinity and min_float, since a cut value of
   // min_float would separate the two, but we convert -infinity to min_float here for symmetry with the positive
   // case and for simplicity.

   // when +-infinity values and min_float/max_float values are present, they usually don't represent real values,
   // since it's exceedingly unlikley that min_float or max_float represents a natural value that just happened
   // to not overflow.  When picking our cut points later between values, we should care more about the highest
   // or lowest value that is not min_float/max_float/+-infinity.  So, we convert +-infinity to min_float/max_float
   // here and disregard that value when choosing bin cut points.  We put the bin cut closer to the other value
   // A good point to put the cut is the value that has the same exponent, but increments the top value, so for
   // example, (7.84222e22, +infinity) should have a bin cut value of 8e22).

   // all of this infrastructure gives the user back the maximum amount of information possible, while also avoiding
   // +-infinity values in either the cut points, or the min/max values, which is good since serialization of
   // +-infinity isn't very standardized accross languages.  It's a problem in JSON especially.

   double* pCopyFrom = aVals;
   double* pCopyTo = aVals;
   const double* const pValsEnd = aVals + cSamples;
   do {
      double val = *pCopyFrom;
      if(PREDICTABLE(!std::isnan(val))) {
         val = UNPREDICTABLE(std::numeric_limits<double>::infinity() == val) ? std::numeric_limits<double>::max() : val;
         val = UNPREDICTABLE(-std::numeric_limits<double>::infinity() == val) ? std::numeric_limits<double>::lowest() :
                                                                                val;
         *pCopyTo = val;
         ++pCopyTo;
      }
      ++pCopyFrom;
   } while(LIKELY(pValsEnd != pCopyFrom));
   const size_t cSamplesWithoutMissing = pCopyTo - aVals;
   EBM_ASSERT(cSamplesWithoutMissing <= cSamples);
   return cSamplesWithoutMissing;
}

EBM_API_BODY ErrorEbm EBM_CALLING_CONVENTION SuggestGraphBounds(IntEbm countCuts,
      double lowestCut,
      double highestCut,
      double minFeatureVal,
      double maxFeatureVal,
      double* lowGraphBoundOut,
      double* highGraphBoundOut) {
   // lowGraphBoundOut and highGraphBoundOut will never legally return NaN
   // lowGraphBoundOut can be -inf if minFeatureVal was -inf, or if our bounds get pushed into -inf
   // lowGraphBoundOut can be +inf if there are no cuts, and both minFeatureVal and maxFeatureVal are +inf (if all data
   // is +inf) highGraphBoundOut can also be any of these values

   // TODO: review these comments below now that things have changed:
   // There are a lot of complexities in choosing the graphing bounds.  Let's start from the beginning:
   // - cuts occur on floating point values.  We need to make a choice whether features that are the exact value
   //   of the cut point go into the upper or lower bounds
   // - we choose lower bound inclusivity so that if a cut is at 5, then the numbers 5.0 and 5.1 will be in the same
   // bound
   // - unfortunately, this means that -1.0 is NOT in the same bounds as -1.1, but negative numbers should be rarer
   //   and in general we shouldn't be putting cuts on whole numbers anyways.  Cuts at 0.5, 1.5, etc are better
   //   and this is where the algorithm will tend to put cuts anyways for whole numbers.
   // - if we wanted -1.0 to be in the same bin as -1.1 AND also have +1.0 be in the same bin as +1.1 there is
   //   an alternative of making 0 it's own bin and using lower bound inclusive for positives, and upper bound
   //   inclusive for negatives.  We do however want other people to be able to write EBM evaluators and this kind
   //   of oddity is just a bit too odd and non-standard.  It's also hard to optimize this in the binary search.
   // - With lower bound inclusivity, no cut should ever be -inf, since -inf will be included in the upper bound
   //   and nothing can be lower than -inf (NaN means missing)
   // - In theory, +inf cut points might have a use to separate max and +inf feature values, but it's kind of weird
   //   to disallow cuts at -inf but allow them at +inf, so we declare +inf to be an illegal cut point as well.
   //   The consequence of this is that the highest legal cut point at max can't separate max from +inf values,
   //   but we can live with that.  lowest can however separate -inf from lowest, but that's just the consequence
   //   of our decisions above.
   // - NaN cut points don't make any sense, so in conclusion cuts can be any normal floating point number.
   // - To preserve interpretability, our graphs should initially be displayed over the range from the min value in
   //   the data to the maximum value in the data.  This is a strong indicator of issues in the data if the range
   //   is too wide, and the user can rapidly zoom into the main area if they need to see more detail.  We could
   //   later add some kind of button to zoom to a point that keeps the outer most bounds in view as an option.
   // - Annother important aspect is that all bins and their scores should be visible in principle on the graphs.
   //   We can't prevent a bin though from being vanishingly small such that it wouldn't be effectively visible, but
   //   it should at least be within the observable range on the graphs.
   // - Under normal circumstances when the user doesn't edit the graphs or provide custom cut points we can offer
   //   an absolute guarantee that when the graphing view goes from the min to the max value that it includes all cut
   //   points IF the following are true:
   //   - the user has not stripped the min and max value information from the model.  It would be nice to allow
   //     the user to do this as an option though if they want to for privacy reasons since the min and max are
   //     potentially egregious violations of privacy (eg: the max net worth in the dataset is $100,000,000,000).
   //   - The end user doesn't edit the graphs after the fact.  Our automatic binning never put cuts automatically
   //     above the max or below the min, however if the user later edits graphs they could want to put cuts outside
   //     of the min/max range.  If the user wants to put a range between 999 and 1001 with a value of +10 and the
   //     max value in the natural data was 100, then we'll have a cut point outside of the normally displayed
   //     min -> max range.  The scenario of wanting a previously unseen bin might happen if the data changes after
   //     training and the user wants to correct the model to handle these cases (eg: a sensor fails and the user wants
   //     to give all values scores of 0 in some range where they were previously +3.1), so I believe we should support
   //     these kinds of scenarios of adding bins outside the natural data range.
   //   - The end user didn't supply user defined cut points upfront before fitting.  If the user supplies user
   //     defined cut points of 1,2,3 and no data is larger than 2.5, then the 3 value cut is above the max
   //   - There is a corner case where the max value is for example 10 and there is a lot of data at 10, but also
   //     there is data one floating point tick below 10 (9.9999999 as a close example).  Since we use lower bound
   //     inclusivity, the cut point will be at 10, and the max value will be at 10, so if our graph goes from the
   //     min value to 10, then the score in the bin ABOVE 10 isn't visible on the graph.  The worse case of this
   //     would occur if one third of the data was at the max float minus one float tick, one third of the data
   //     was at the max, and one third of the data was at +inf.  The only valid cut would be at max, with 1/3 of the
   //     data on the left and 2/3 of the data on the right and the graph bounds ending at max value with no
   //     possibility to show the upper score bin unless the graph shows beyond the max value and in fact shows
   //     beyond the max float value.
   //   - There is an odd case, but one that'll happen with regularity on real data where all the feature values
   //     are the same.  For instance a sensor that has never worked and always reports 0.  In this case the
   //     min and max value are both 0, and there are no cuts (for non-editied or pre-specified cuts), but the
   //     interesting aspect is that the range has zero mass since (0 - 0) = 0 and thus the graph doesn't have
   //     a range that is representable as a 2D graph.  I would recommend putting the single value on a graph with
   //     one tick at the specific value and showing no other tick marks on the graph leaving the impression that
   //     the graph has infinite resolution, which it does.
   //   - To handle the scenario where the max and the highest next data value is max minus one tick AND to handle
   //     the more likely scenario where there is only one value, Slicer needs to be able to handle zero width
   //     regions where the upper and lower bound is the same number.  Other than these two special cases though
   //     since cut points should always increase, it should not be possible for the lower and upper bound to be
   //     identical
   // - we could disallow Slicer from having zero width slices (low bound and high bound identical) if we were
   //   willing to do the following:
   //   - if all the data is identical (eg: all values are 5.0), then we could choose some arbitrary zone to graph
   //     by showing for instance 4.9999999 to 5.0000001 which would be purposely narrow to show that there is
   //     only 1 value in the data, and then we have the width on the graph to show the score in the "bin"
   //   - if we encounter the scenario where the top value and next lower value are separated by a float tick,
   //     we can move the graph bound outwards a bit.  This violates our rule that the graph should initially show
   //     from min to max, but this is an exceptional circumstance.
   //   - alternatively, we could simply remove zero width bins at the top and just accept that this is a truely
   //     exceptional scenario that just doesn't get graphed.  In this case since we want to avoid the zero width
   //     zone in the programming interface it's not just a UI change but we need to filter it out when genereating
   //     the explanation
   //   - If we choose to increment up one tick, since we can't increment up from max, we need to disallow cuts on
   //     exactly max, so we should therefore throw exceptions if the user specifies max as a cut point and force
   //     cuts to not choose max on automated cut selection
   //   - disallowing NaN and infinities for cut points is expected, but I think many users would be surprised to
   //     recieve an exception if they specify a cut point at max.  For this reason, I prefer allowing zero
   //     width intervals in Slicer and detecting handling this scenario in the graphs.  I also don't like removing
   //     one Slicer zone if the end slices are zero width since this will be surprizing behaviour to anyone
   //     using our interface and even if they don't get an exception it could lead to a crash bug or worse.
   // - In general, I think that we should allow the user to edit graphs and add bins beyond the original graphing
   //   range because there are valid reasons why the user might want to do this:
   //   - adding special casing after the fact, like if a sensor stops working and the returned value of 1000 should
   //     be ignored, yet this particular odd value never appears in the training data.
   //   - the user will get a surprising to them exception if we disallow editing outside of the min/max range.  Model
   //     evaluation could work during initial model building, but then fail later in a model building pipeline if one
   //     of the upper bins is rare.  For example if it's rare for data to be above 1000, but the user still wants
   //     to special case edit models to handle that differently, if in a production environment they just happen to
   //     get a dataset that doesn't have values above 1000 in it.  I think it shouldn't fail since then the user
   //     needs to carefully check their data for all kinds of rare exceptional events which they won't know about
   //     beforehand unless they read our code very carefully to know all the failure cases.  A warning would be a
   //     far nicer option in these circumstances, which we have enough information to do.
   // - if we allow the user to edit graphs to put cut points beyond the min/max value, then we need to break one of
   //   our two cardinal rules above in those exceptoinal cirumstances where the user edits features:
   //   - If we choose to continue to show the graph between the min and max then
   //     there will be cuts and scores not visible on our graphs, and the data inside Slicer will look very odd
   //     with the last bin having negative width where the lower bound would be the last cut and the upper bound
   //     be the maximum, which is now lower than the upper bound cut
   //   - if we choose to expand the graph viewing area beyond the min and max value, then we can now show the entire
   //     range of valid scores.  We need to though expand BEYOND the outer cut points in this case because there
   //     is a range that extends from the highest cut point until infinity and from the lowest cut point to negative
   //     infinity and those score values won't be shown if our graph terminates at the lowest and highest cut points
   // - Of these two bad options, probably expanding the view of the graph is the least worse since it at least
   //   notifies the user of an odd/unusual situation with that feature and it allows them to see all values which
   //   wouldn't be possible, and it avoids the issues of really super odd negative width ranges in Slicer.
   // - Under this scenario there are 3 types of values:
   //   - cut points
   //   - min/max value of the data (storing this as is is nice to keep in the model file, and maybe indicate on the
   //   graphs)
   //   - the graph low and grpah high values.  We can use these values in Slicer intervals.  The graph low and graph
   //   high values
   //     can be made to guarantee that they are always beyond the outside cut points AND either equal to or beyond
   //     the min/max values
   // - slicer should only include the cut points and the graph bounds in the ranges (never the min or max values!).
   //   We should keep the true min and max values as separate data fields outside of the Slicer ranges since
   //   otherwise we'll get negative range widths in some bad cases
   // - We don't need to store the graph min and graph max in the model.  We can generate these when we generate
   //   an explanation object since they derive precicely from the cut points and min/max values which we do store
   // - if the user chooses to remove the min/max value for privacy reasons (or any other reason), then we'd use
   //   the same algorithm of putting the graph view just a bit outside of the range of the lower and upper cut points
   // - There is one more wrinkle however.  What if the min is -inf or the max is +inf:
   //   - infinity isn't a JSON compatible value, so we should avoid serializing it as JSON
   //   - it's impossible to make a graph that starts on +-infinity since then all other features are 1/+-inf and
   //     then you can't even zoom into them, even in theory since your range is infinite to begin with
   //   - our options are:
   //     - make +inf as max and -inf as lowest
   //     - write out non-JSON compliant +-inf values and special case the graphs
   //     - store the non-infinity min and non-infinity max AND also keep bools to know if the true min and max were
   //       inf values.  (we should store this as -1/0/+1 for each since the max can be -inf if all data values
   //       are -inf OR if all data values are +inf then the min can be +inf!
   //     - I like the latter option since it might be nice for debugging to know what the min/max values were other
   //       than the +-inf values
   //   - If we choose the latter though of storing the non-infinity min/max, then we need to be careful when we
   //     we choose automatic cut points.  If we replaced +inf with max, and -inf with min for the purposes of
   //     cut point calculation, then we can have a cut point that is outside the min/max range for non-edited and
   //     non-pre-specified cut points.  We can avoid this scenario though by not allowing the automatic cut point
   //     algorithm to select cut points between the non-infinity min/max values and min/lowest.  We essentially
   //     ignore the +inf and -inf values beside using their counts.  This does however resolve another kind of
   //     issue that we can get huge unnatural values if we attempt to put cuts between the
   //     non-inf min/max and +-inf or max/lowest
   //   - as a nicety, we can avoid the scenario where the upper cut point and the max are the same value for the
   //     automatic binning code by disallowing a cut at the max value.  So if the data consisted of:
   //     4.999999, 5.000000, 5.000001 (assuming these are 1 float tick apart), then 5.000001 would be the max value
   //     and the cutting algorithm would tend to put a cut at 5.000001 in order to separate 5.000000 from 5.000001
   //     but then we'd get a zero width Slicer interval and we wouldn't be able to show the score value for
   //     5.000001 and beyond since our graph ends at 5.000001, but if we disallow an automatic cut there then
   //     the only cut allowed here would be at 5.000000 with our graph going from 4.999999 to 5.000001, thus
   //     we'd show some region for both the lower and upper bin (exactly 1 float point tick's worth).
   // - consider two clarifying scenarios:
   //   - you have values between -5 and +10 and a single +inf value
   //     - obviously the cut should be below +10, but what should the max be?  We can't really show a range
   //       of +inf since graphs don't work that way.  We could show max_float but that's confusing to end users
   //       and brittle if the number is changed between float/double
   //     - probably using 10 is the right max here.. what else makes sense?
   //   - you have values between -5 and +10 and 1/100 of the data is +inf
   //     - if we set the max to +inf then we can't really make a graph and we can't write to JSON
   //     - if we set the maxFeatureVal to max_float then people won't really undrestand why the graph goes unitl
   //       3.402823466e+38
   //     - if we set the upper cut below 10, then we get to keep it less than the max, but then +10 will be bunched
   //       with +inf and that seems like a bad clusting
   //     - PROBABLY, choosing something like 1000 times the range of the data beyond the max is the right thing
   //       to do.  If someone said the data ranged from 0 to 1 with +inf values too, but then you get a value
   //       at 10,000, you might wonder if it should go into the 0 to 1 bin or the +inf bin.
   // - since we can't graph to +-inf anyways, we might want to include a field to indicate if there are infinities
   //   at either the min or the max, but it gets complicated if all values are +inf for instance then the min
   //   value is +inf and that makes it confusing.  You'd need to have -1/0/+1 to cover all the -inf/no_inf/+inf cases
   //   so I tend to think it's not worth preserving the information that there are +inf or -inf values in the dataset
   // - generally, max_float and min_float are problematic to graph or reason about.  If we set the bounds to
   //   min_float to max_float then the distance between them goes to infinity since max_float - (min_float) = +inf
   //   Also, people don't generally know that max_float is 3.402823466e+38 so it just confuses them.
   //   Also, if the user tries to zoom past max_float, then the graphing software might fail, and we do want to
   //   allow people to use graphing software other than ours.

   //-we want to get the max and min non - infinity values to write in JSON and to graph
   //- BUT, if there are a lot of + inf or -inf values we have to decide where to put cut points if automatic cutting is
   // selected
   //- the graph is going to go from min to max then it's tempting to want to put the cut points slightly below the max
   // and slightly above the min by bunching the max and min values with the + inf or -inf values, but + inf or -inf
   // might be truely special scenarios so we'd rather keep them in their own bins if there is enough data for them so
   // we'd rather put the cut point above max and below min to separate the max and min
   //- but where to put the cut points above max or min.In theory they have a huge range(to infinity!)
   // but we can't graph infinity anyways, and making a graph go to 10^38 seems excessive
   //- if I asked a human where they'd put a cut if they had data going from 1 to 10 and then +inf and -inf, I think a
   // reasonable answer is that if a new value poped in that was above 100 or 1000 times larger than the previous max it
   // would be ambiguous if that should be binned with the + inf / -inf values or with the min / max values.So, let's
   // say we go with 100 times larger, that would mean we'd have a cut point at 10 * 100 = 1000 and -1000 (because our
   // range is
   //

   if(nullptr == lowGraphBoundOut) {
      LOG_0(Trace_Error, "ERROR SuggestGraphBounds nullptr == lowGraphBoundOut");
      return Error_IllegalParamVal;
   }
   if(nullptr == highGraphBoundOut) {
      LOG_0(Trace_Error, "ERROR SuggestGraphBounds nullptr == highGraphBoundOut");
      return Error_IllegalParamVal;
   }
   if(maxFeatureVal < minFeatureVal) {
      // silly caller, these should be reversed.  If either or both are NaN this won't execute, which is good
      LOG_0(Trace_Error, "ERROR SuggestGraphBounds maxFeatureVal < minFeatureVal");
      *lowGraphBoundOut = std::numeric_limits<double>::quiet_NaN();
      *highGraphBoundOut = std::numeric_limits<double>::quiet_NaN();
      return Error_IllegalParamVal;
   }

   if(countCuts <= IntEbm{0}) {
      if(countCuts < IntEbm{0}) {
         LOG_0(Trace_Error, "ERROR SuggestGraphBounds countCuts < IntEbm { 0 }");
         *lowGraphBoundOut = std::numeric_limits<double>::quiet_NaN();
         *highGraphBoundOut = std::numeric_limits<double>::quiet_NaN();
         return Error_IllegalParamVal;
      }

      // countCuts was zero, so the only information we have to go on are the minFeatureVal and maxFeatureVal..
      if(std::isnan(minFeatureVal)) {
         if(std::isnan(maxFeatureVal)) {
            // we can't avoid the scenario where min = float.lowest() and max = float.max.  In that case
            // the range of max - min would overflow to +inf, so the graphing code needs to handle this.
            // So, we might as well allow +inf and -inf as legal min/max returns.  If we have no
            // information at all, we might as well return those to cover the entire range
            // and force the caller to handle these weird conditions.  The alternative is to return NaNs
            // but that has it's own issues, and returning +-inf at least removes the requirement to handle
            // NaN returns

            *lowGraphBoundOut = -std::numeric_limits<double>::infinity();
            *highGraphBoundOut = +std::numeric_limits<double>::infinity();
            return Error_None;
         } else {
            // no min, but we do have a max?!  Ok..
            *lowGraphBoundOut = maxFeatureVal;
            *highGraphBoundOut = maxFeatureVal;
            return Error_None;
         }
      } else {
         if(std::isnan(maxFeatureVal)) {
            // no max, but we do have a min?!  Ok..
            *lowGraphBoundOut = minFeatureVal;
            *highGraphBoundOut = minFeatureVal;
            return Error_None;
         } else {
            // the danger here is that this allows both low & high to be either +-infinity, which makes sense
            // if all the data is +inf or -inf, but then when we go to subtract it to get a range you'd get NaN
            *lowGraphBoundOut = minFeatureVal;
            *highGraphBoundOut = maxFeatureVal;
            return Error_None;
         }
      }
   }

   if(std::isnan(lowestCut) || std::isinf(lowestCut) || std::isnan(highestCut) || std::isinf(highestCut)) {
      LOG_0(Trace_Error,
            "ERROR SuggestGraphBounds std::isnan(lowestCut) || std::isinf(lowestCut) || std::isnan(highestCut) || "
            "std::isinf(highestCut)");
      *lowGraphBoundOut = std::numeric_limits<double>::quiet_NaN();
      *highGraphBoundOut = std::numeric_limits<double>::quiet_NaN();
      return Error_IllegalParamVal;
   }

   // we're going to be checking lowestCut and highestCut, so we should check that they have valid values
   if(IntEbm{1} == countCuts) {
      if(lowestCut != highestCut) {
         LOG_0(Trace_Error,
               "ERROR SuggestGraphBounds when 1 == countCuts, then lowestCut and highestCut should be identical");
         *lowGraphBoundOut = std::numeric_limits<double>::quiet_NaN();
         *highGraphBoundOut = std::numeric_limits<double>::quiet_NaN();
         return Error_IllegalParamVal;
      }
   } else {
      if(highestCut <= lowestCut) {
         LOG_0(Trace_Error, "ERROR SuggestGraphBounds highestCut <= lowestCut");
         *lowGraphBoundOut = std::numeric_limits<double>::quiet_NaN();
         *highGraphBoundOut = std::numeric_limits<double>::quiet_NaN();
         return Error_IllegalParamVal;
      }
   }

   bool bExpandLower = false;
   if(std::isnan(minFeatureVal)) {
      // the user removed the min value from the model so we need to use the available info, which is the lowestCut
      minFeatureVal = lowestCut;
      bExpandLower = true;
   } else if(lowestCut < minFeatureVal) {
      // the model has been edited or supplied with non-data derived cut points OR lowestCut == minFeatureVal which is
      // legal our automatic binning code will never put a cut on the exact min value even if the two lowest cuts are
      // separated by a floating point epsilon, but we'll accept it here since then we're symetric with the upper bound
      // handling which can have a max on the highest cut
      minFeatureVal = lowestCut;
      bExpandLower = true;
   }

   bool bExpandHigher = false;
   if(std::isnan(maxFeatureVal)) {
      // the user removed the max value from the model so we need to use the available info, which is the highestCut
      maxFeatureVal = highestCut;
      bExpandHigher = true;
   } else if(maxFeatureVal < highestCut) {
      // the model has been edited or supplied with non-data derived cut points
      maxFeatureVal = highestCut;
      bExpandHigher = true;
   }

   if(minFeatureVal == maxFeatureVal) {
      // we handled zero cuts above, and if there were two cuts they'd have to have unique increasing values
      // so the only way we can have the low and high graph bounds the same is if we have one cut and both the
      // minFeatureVal and maxFeatureVal are the same as that cut, or they are illegal, or they are missing (NaN)
      EBM_ASSERT(IntEbm{1} == countCuts);

      // if the regular binning code was kept and the min/max value wasn't removed from the model, then we should
      // not be able to get here, since minFeatureVal == maxFeatureVal can only happen if there is only one value, and
      // if there is only one value we would never create cut points, so the cut points or min/max have been user edited

      *lowGraphBoundOut = minFeatureVal;
      *highGraphBoundOut = maxFeatureVal;
      return Error_None;
   }

   // limit the amount of dillution allowed for the tails by capping the relevant cCutPointRet value
   // to 1/32, which means we leave about 3% of the visible area to tail bounds (1.5% on the left and
   // 1.5% on the right)
   const size_t cCutsLimited = static_cast<size_t>(EbmMin(IntEbm{32}, countCuts));
   EBM_ASSERT(size_t{1} <= cCutsLimited);
   const size_t denominator = cCutsLimited << 1;
   EBM_ASSERT(size_t{2} <= denominator);
   const double denominatorFloat = static_cast<double>(denominator);

   EBM_ASSERT(minFeatureVal < maxFeatureVal);
   double movementFromEnds = maxFeatureVal - minFeatureVal;
   EBM_ASSERT(!std::isnan(
         movementFromEnds)); // since maxFeatureVal != minFeatureVal, they can't be both be +inf or both be -inf
   // IEEE 754 (which we static_assert) won't allow the subtraction of two unequal numbers to be non-zero
   EBM_ASSERT(double{0} < movementFromEnds);
   if(std::isinf(movementFromEnds)) {
      // movementFromEnds can be +infinity if highestCut is max and lowestCut is lowest or either is +-inf.  Try again
      // but divide first since we might find that we're in the legal range by dividing first
      movementFromEnds = maxFeatureVal / denominatorFloat - minFeatureVal / denominatorFloat;
   } else {
      movementFromEnds /= denominatorFloat;
   }

   // movementFromEnds can be +infinity.  It could also underflow to zero
   EBM_ASSERT(!std::isnan(movementFromEnds));
   EBM_ASSERT(double{0} <= movementFromEnds);

   if(bExpandLower) {
      // minFeatureVal must be lower than maxFeatureVal so it can't be +inf
      EBM_ASSERT(minFeatureVal <= std::numeric_limits<double>::max());

      minFeatureVal -= movementFromEnds;
      EBM_ASSERT(!std::isnan(minFeatureVal));
      EBM_ASSERT(minFeatureVal <= std::numeric_limits<double>::max());
      // minFeatureVal can be -inf
   }

   if(bExpandHigher) {
      // maxFeatureVal must be higher than minFeatureVal so it can't be -inf
      EBM_ASSERT(std::numeric_limits<double>::lowest() <= maxFeatureVal);

      maxFeatureVal += movementFromEnds;
      EBM_ASSERT(!std::isnan(maxFeatureVal));
      EBM_ASSERT(std::numeric_limits<double>::lowest() <= maxFeatureVal);
      // maxFeatureVal can be +inf
   }

   *lowGraphBoundOut = minFeatureVal;
   *highGraphBoundOut = maxFeatureVal;
   return Error_None;
}

static double Stddev(const size_t cSamples,
      const size_t cStride,
      const double* const aFeatureVals,
      const double* const aWeights,
      size_t* const pcNaN,
      size_t* const pcInf) {
   EBM_ASSERT(1 <= cSamples);
   EBM_ASSERT(1 <= cStride);
   EBM_ASSERT(nullptr != aFeatureVals);
   EBM_ASSERT(nullptr != pcNaN);
   EBM_ASSERT(nullptr != pcInf);

   const size_t iEnd = cSamples * cStride;

   // use Welford's method to calculate stddev
   // https://stackoverflow.com/questions/895929/how-do-i-determine-the-standard-deviation-stddev-of-a-set-of-values
   // https://www.johndcook.com/blog/standard_deviation/

   double factor = 1.0;
   double s;
   size_t cNaN;
   size_t cInf;
   double k;
   size_t cNormal;

   goto skip;
   do {
      factor *= 0.5;
      // there should be some factor that gives us a non-overflowing stddev
      EBM_ASSERT(std::numeric_limits<double>::min() <= factor);

   skip:;

      cNaN = 0;
      cInf = 0;

      size_t i = 0;
      s = 0;
      double m = 0;
      k = 0;
      const double* pWeight = aWeights;
      size_t cWeightInf = 0;
      cNormal = 0;
      do {
         double val = aFeatureVals[i];
         if(std::isnan(val)) {
            ++cNaN;
         } else if(std::isinf(val)) {
            ++cInf;
         } else {
            ++cNormal;
            double weight = 1.0;
            if(nullptr != pWeight) {
               weight = *pWeight;
               if(std::numeric_limits<double>::infinity() == weight) {
                  k = static_cast<double>(cWeightInf);
                  ++cWeightInf;
                  weight = 1.0;
               } else {
                  weight *= factor;
                  if(0 != cWeightInf) {
                     weight = 0.0;
                  }
               }
            }
            k += weight;
            val *= factor; // to handle overflows we scale back when necessary
            const double numerator = val - m;
            double ratio = weight / k;
            if(k < std::numeric_limits<double>::min()) {
               // if all the weights are zero, then weigh them all equally
               weight = double{1};
               ratio = double{1} / static_cast<double>(cNormal);
            }
            m += numerator * ratio;
            s += weight * numerator * (val - m);
         }
         if(nullptr != pWeight) {
            ++pWeight;
         }
         i += cStride;
      } while(iEnd != i);
   } while(std::isnan(s) || std::isinf(s) || std::numeric_limits<double>::infinity() == k);

   EBM_ASSERT(!std::isnan(s));
   EBM_ASSERT(!std::isinf(s));
   EBM_ASSERT(cNaN + cInf <= cSamples);
   EBM_ASSERT(cNormal == cSamples - cNaN - cInf);

   *pcNaN = cNaN;
   *pcInf = cInf;

   if(cNormal <= 1) {
      return 0.0;
   }

   if(k < std::numeric_limits<double>::min()) {
      // if all the weights are zero, then weigh them all equally
      k = static_cast<double>(cNormal);
   }

   s /= k;
   if(s < std::numeric_limits<double>::min()) {
      // clean up subnormal numbers and handle the case where floating point noise creates a slightly negative s
      return 0.0;
   }

   s = std::sqrt(s) / factor;
   if(s < std::numeric_limits<double>::min()) {
      // clean up subnormal numbers
      return 0.0;
   }

   if(std::numeric_limits<double>::infinity() == s) {
      // if +inf is observed, it's due to floating point noise, so change to max float
      return std::numeric_limits<double>::max();
   }

   return s;
}

static double Mean(const size_t cSamples,
      const size_t cStride,
      const double* const aFeatureVals,
      const double* const aWeights,
      size_t* const pcNaN,
      size_t* const pcPosInf,
      size_t* const pcNegInf) {
   EBM_ASSERT(1 <= cSamples);
   EBM_ASSERT(1 <= cStride);
   EBM_ASSERT(nullptr != aFeatureVals);
   EBM_ASSERT(nullptr != pcNaN);
   EBM_ASSERT(nullptr != pcPosInf);
   EBM_ASSERT(nullptr != pcNegInf);

   const size_t iEnd = cSamples * cStride;

   // use Welford's method to calculate mean (which is part of stddev)
   // https://stackoverflow.com/questions/895929/how-do-i-determine-the-standard-deviation-stddev-of-a-set-of-values
   // https://www.johndcook.com/blog/standard_deviation/

   double factor = 1.0;
   double mean;
   size_t cNaN;
   size_t cPosInf;
   size_t cNegInf;
   double k;
   size_t cNormal;

   goto skip;
   do {
      factor *= 0.5;
      // there should be some factor that gives us a non-overflowing stddev
      EBM_ASSERT(std::numeric_limits<double>::min() <= factor);

   skip:;

      cNaN = 0;
      cPosInf = 0;
      cNegInf = 0;

      size_t i = 0;
      mean = 0;
      k = 0;
      const double* pWeight = aWeights;
      size_t cWeightInf = 0;
      cNormal = 0;
      do {
         double val = aFeatureVals[i];
         if(std::isnan(val)) {
            ++cNaN;
         } else if(std::isinf(val)) {
            if(std::numeric_limits<double>::infinity() == val) {
               ++cPosInf;
            } else {
               EBM_ASSERT(-std::numeric_limits<double>::infinity() == val);
               ++cNegInf;
            }
         } else {
            ++cNormal;
            double weight = 1.0;
            if(nullptr != pWeight) {
               weight = *pWeight;
               if(std::numeric_limits<double>::infinity() == weight) {
                  k = static_cast<double>(cWeightInf);
                  ++cWeightInf;
                  weight = 1.0;
               } else {
                  weight *= factor;
                  if(0 != cWeightInf) {
                     weight = 0.0;
                  }
               }
            }
            k += weight;
            val *= factor; // to handle overflows we scale back when necessary
            const double numerator = val - mean;
            double ratio = weight / k;
            if(k < std::numeric_limits<double>::min()) {
               // if all the weights are zero, then weigh them all equally
               ratio = double{1} / static_cast<double>(cNormal);
            }
            mean += numerator * ratio;
         }
         if(nullptr != pWeight) {
            ++pWeight;
         }
         i += cStride;
      } while(iEnd != i);
   } while(std::isnan(mean) || std::isinf(mean) || std::numeric_limits<double>::infinity() == k);

   EBM_ASSERT(!std::isnan(mean));
   EBM_ASSERT(!std::isinf(mean));
   EBM_ASSERT(cNaN + cPosInf + cNegInf <= cSamples);
   EBM_ASSERT(cNormal == cSamples - cNaN - cPosInf - cNegInf);

   *pcNaN = cNaN;
   *pcPosInf = cPosInf;
   *pcNegInf = cNegInf;

   mean /= factor;

   if(std::isinf(mean)) {
      // if +inf or -inf is observed, it's due to floating point noise, so change to max/-max floats
      if(std::numeric_limits<double>::infinity() == mean) {
         mean = std::numeric_limits<double>::max();
      } else {
         EBM_ASSERT(-std::numeric_limits<double>::infinity() == mean);
         mean = std::numeric_limits<double>::lowest();
      }
   } else if(-std::numeric_limits<double>::min() < mean && mean < std::numeric_limits<double>::min()) {
      // clean up subnormal numbers
      mean = 0.0;
   }

   return mean;
}

// we don't care if an extra log message is outputted due to the non-atomic nature of the decrement to this value
static int g_cLogEnterSafeMeanCount = 25;
static int g_cLogExitSafeMeanCount = 25;

EBM_API_BODY ErrorEbm EBM_CALLING_CONVENTION SafeMean(
      IntEbm countBags, IntEbm countTensorBins, const double* vals, const double* weights, double* tensorOut) {

   LOG_COUNTED_N(&g_cLogEnterSafeMeanCount,
         Trace_Info,
         Trace_Verbose,
         "Entered SafeMean: "
         "countBags=%" IntEbmPrintf ", "
         "countTensorBins=%" IntEbmPrintf ", "
         "vals=%p, "
         "weights=%p, "
         "tensorOut=%p",
         countBags,
         countTensorBins,
         static_cast<const void*>(vals),
         static_cast<const void*>(weights),
         static_cast<const void*>(tensorOut));

   if(countBags <= IntEbm{0}) {
      if(countBags < IntEbm{0}) {
         LOG_0(Trace_Error, "ERROR SafeMean countBags < IntEbm{0}");
         return Error_IllegalParamVal;
      }
      return Error_None;
   }
   if(IsConvertError<size_t>(countBags)) {
      LOG_0(Trace_Error, "ERROR SafeMean IsConvertError<size_t>(countBags)");
      return Error_IllegalParamVal;
   }
   const size_t cBags = static_cast<size_t>(countBags);

   if(countTensorBins <= IntEbm{0}) {
      if(countTensorBins < IntEbm{0}) {
         LOG_0(Trace_Error, "ERROR SafeMean countTensorBins < IntEbm{0}");
         return Error_IllegalParamVal;
      }
      return Error_None;
   }
   if(IsConvertError<size_t>(countTensorBins)) {
      LOG_0(Trace_Error, "ERROR SafeMean IsConvertError<size_t>(countTensorBins)");
      return Error_IllegalParamVal;
   }
   const size_t cTensorBins = static_cast<size_t>(countTensorBins);

   if(nullptr == vals) {
      LOG_0(Trace_Error, "ERROR SafeMean nullptr == vals");
      return Error_IllegalParamVal;
   }

   if(nullptr == tensorOut) {
      LOG_0(Trace_Error, "ERROR SafeMean nullptr == tensorOut");
      return Error_IllegalParamVal;
   }

   const double* pVal = vals;
   double* pOut = tensorOut;
   const double* const pOutEnd = tensorOut + cTensorBins;
   do {
      size_t cNaN;
      size_t cPosInf;
      size_t cNegInf;
      double mean = Mean(cBags, cTensorBins, pVal, weights, &cNaN, &cPosInf, &cNegInf);
      // we use NaN as a user indicator that the prediction is illegal, so do everything possible
      // to avoid creating new NaN values. If we have more +inf than -inf then it's +inf, else -inf
      // If we have equal numbers of +inf and -inf, use +inf because that's more likely to be noticed.
      if(0 != cNaN) {
         mean = std::numeric_limits<double>::quiet_NaN();
      } else if(0 != cPosInf) {
         if(cNegInf <= cPosInf) {
            mean = std::numeric_limits<double>::infinity();
         } else {
            mean = -std::numeric_limits<double>::infinity();
         }
      } else if(0 != cNegInf) {
         mean = -std::numeric_limits<double>::infinity();
      }
      *pOut = mean;

      ++pVal;
      ++pOut;
   } while(pOutEnd != pOut);

   LOG_COUNTED_0(&g_cLogExitSafeMeanCount, Trace_Info, Trace_Verbose, "Exited SafeMean");

   return Error_None;
}

// we don't care if an extra log message is outputted due to the non-atomic nature of the decrement to this value
static int g_cLogEnterSafeStandardDeviationCount = 25;
static int g_cLogExitSafeStandardDeviationCount = 25;

EBM_API_BODY ErrorEbm EBM_CALLING_CONVENTION SafeStandardDeviation(
      IntEbm countBags, IntEbm countTensorBins, const double* vals, const double* weights, double* tensorOut) {

   LOG_COUNTED_N(&g_cLogEnterSafeStandardDeviationCount,
         Trace_Info,
         Trace_Verbose,
         "Entered SafeStandardDeviation: "
         "countBags=%" IntEbmPrintf ", "
         "countTensorBins=%" IntEbmPrintf ", "
         "vals=%p, "
         "weights=%p, "
         "tensorOut=%p",
         countBags,
         countTensorBins,
         static_cast<const void*>(vals),
         static_cast<const void*>(weights),
         static_cast<const void*>(tensorOut));

   if(countBags <= IntEbm{0}) {
      if(countBags < IntEbm{0}) {
         LOG_0(Trace_Error, "ERROR SafeStandardDeviation countBags < IntEbm{0}");
         return Error_IllegalParamVal;
      }
      return Error_None;
   }
   if(IsConvertError<size_t>(countBags)) {
      LOG_0(Trace_Error, "ERROR SafeStandardDeviation IsConvertError<size_t>(countBags)");
      return Error_IllegalParamVal;
   }
   const size_t cBags = static_cast<size_t>(countBags);

   if(countTensorBins <= IntEbm{0}) {
      if(countTensorBins < IntEbm{0}) {
         LOG_0(Trace_Error, "ERROR SafeStandardDeviation countTensorBins < IntEbm{0}");
         return Error_IllegalParamVal;
      }
      return Error_None;
   }
   if(IsConvertError<size_t>(countTensorBins)) {
      LOG_0(Trace_Error, "ERROR SafeStandardDeviation IsConvertError<size_t>(countTensorBins)");
      return Error_IllegalParamVal;
   }
   const size_t cTensorBins = static_cast<size_t>(countTensorBins);

   if(nullptr == vals) {
      LOG_0(Trace_Error, "ERROR SafeStandardDeviation nullptr == vals");
      return Error_IllegalParamVal;
   }

   if(nullptr == tensorOut) {
      LOG_0(Trace_Error, "ERROR SafeStandardDeviation nullptr == tensorOut");
      return Error_IllegalParamVal;
   }

   const double* pVal = vals;
   double* pOut = tensorOut;
   const double* const pOutEnd = tensorOut + cTensorBins;
   do {
      size_t cNaN;
      size_t cInf;
      double stddev = Stddev(cBags, cTensorBins, pVal, weights, &cNaN, &cInf);
      // we use NaN as a user indicator that the prediction is illegal, so do everything possible
      // to avoid creating new NaN values.
      if(0 != cNaN) {
         stddev = std::numeric_limits<double>::quiet_NaN();
      } else if(0 != cInf) {
         // Even if they are all +inf or -inf make standard deviation +inf since inf is not well defined.
         stddev = std::numeric_limits<double>::infinity();
      }
      *pOut = stddev;

      ++pVal;
      ++pOut;
   } while(pOutEnd != pOut);

   LOG_COUNTED_0(&g_cLogExitSafeStandardDeviationCount, Trace_Info, Trace_Verbose, "Exited SafeStandardDeviation");

   return Error_None;
}

// we don't care if an extra log message is outputted due to the non-atomic nature of the decrement to this value
static int g_cLogEnterGetHistogramCutCount = 25;
static int g_cLogExitGetHistogramCutCount = 25;

EBM_API_BODY IntEbm EBM_CALLING_CONVENTION GetHistogramCutCount(IntEbm countSamples, const double* featureVals) {
   LOG_COUNTED_N(&g_cLogEnterGetHistogramCutCount,
         Trace_Info,
         Trace_Verbose,
         "Entered GetHistogramCutCount: "
         "countSamples=%" IntEbmPrintf ", "
         "featureVals=%p",
         countSamples,
         static_cast<const void*>(featureVals));

   if(UNLIKELY(countSamples <= 0)) {
      if(UNLIKELY(countSamples < 0)) {
         LOG_0(Trace_Warning, "WARNING GetHistogramCutCount countSamples < 0");
      }
      return 0;
   }
   if(UNLIKELY(IsConvertError<size_t>(countSamples))) {
      LOG_0(Trace_Warning, "WARNING GetHistogramCutCount IsConvertError<size_t>(countSamples)");
      return 0;
   }
   const size_t cSamples = static_cast<size_t>(countSamples);

   IntEbm ret = 0;
   size_t cNaN;
   size_t cInf;
   const double stddev = Stddev(cSamples, 1, featureVals, nullptr, &cNaN, &cInf);
   if(double{0} < stddev) {
      const size_t cNormal = cSamples - cNaN - cInf;
      if(size_t{3} <= cNormal) {
         const double mean = Mean(cSamples, 1, featureVals, nullptr, &cNaN, &cInf, &cInf);
         const double cNormalDouble = static_cast<double>(cNormal);
         const double cNormalCubicRootDouble = std::cbrt(cNormalDouble);
         const double multFactor = double{1} / cNormalCubicRootDouble / stddev;

         double g1 = 0;
         const double* pFeatureVal = featureVals;
         const double* const pFeatureValsEnd = featureVals + cSamples;
         do {
            const double val = *pFeatureVal;
            if(!std::isnan(val) && !std::isinf(val)) {
               const double interior = (val - mean) * multFactor;
               g1 += interior * interior * interior;
            }
            ++pFeatureVal;
         } while(pFeatureValsEnd != pFeatureVal);
         g1 = std::abs(g1);

         const double denom = std::sqrt(
               double{6} * (cNormalDouble - double{2}) / ((cNormalDouble + double{1}) * (cNormalDouble + double{3})));
         const double countSturgesBins = double{1} + std::log2(cNormalDouble);
         double countBins = countSturgesBins + std::log2(double{1} + g1 / denom);
         countBins = std::ceil(countBins);
         if(std::isnan(countBins) || std::isinf(countBins)) {
            // use Sturges' formula if we have a numeracy issue with our data. countSturgesBins pretty much can't fail
            countBins = std::ceil(countSturgesBins);
         }
         ret = double{FLOAT64_TO_INT64_MAX} < countBins ? IntEbm{FLOAT64_TO_INT64_MAX} : static_cast<IntEbm>(countBins);
         EBM_ASSERT(1 <= ret); // since our formula started from 1 and added
         --ret; // # of cuts is one less than the number of bins
      }
   }

   LOG_COUNTED_N(&g_cLogExitGetHistogramCutCount,
         Trace_Info,
         Trace_Verbose,
         "Exited GetHistogramCutCount: "
         "return=%" IntEbmPrintf,
         ret);

   return ret;
}

} // namespace DEFINED_ZONE_NAME
