% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kmBlock.r
\name{kmBlockORPC}
\alias{kmBlockORPC}
\title{A function for optimizing multiple random partitions using k-means one-mode and linked blockmodeling. Calls \code{kmBlockC} for optimizing individual random partitions.}
\usage{
kmBlockORPC(
  M,
  k,
  rep,
  save.initial.param = TRUE,
  deleteMs = TRUE,
  max.iden = 10,
  return.all = FALSE,
  return.err = TRUE,
  seed = NULL,
  parGenFun = blockmodeling::genRandomPar,
  mingr = NULL,
  maxgr = NULL,
  addParam = list(genPajekPar = TRUE, probGenMech = NULL),
  maxTriesToFindNewPar = rep * 10,
  skip.par = NULL,
  printRep = ifelse(rep <= 10, 1, round(rep/10)),
  n = NULL,
  nCores = 1,
  useParLapply = TRUE,
  cl = NULL,
  stopcl = is.null(cl),
  ...
)
}
\arguments{
\item{M}{A square matrix giving the adjaciency relationg between the network's nodes (aka vertexes)}

\item{k}{The number of clusters used in the generation of partitions.}

\item{rep}{The number of repetitions/different starting partitions to check.}

\item{save.initial.param}{Should the inital parameters(\code{approaches}, ...) of using \code{kmBlockC} be saved. The default value is \code{TRUE}.}

\item{deleteMs}{Delete networks/matrices from the results of to save space. Defaults to \code{TRUE}.}

\item{max.iden}{Maximum number of results that should be saved (in case there are more than \code{max.iden} results with minimal error, only the first \code{max.iden} will be saved).}

\item{return.all}{If \code{FALSE}, solution for only the best (one or more) partition/s is/are returned.}

\item{return.err}{Should the error for each optimized partition be returned. Defaults to \code{TRUE}.}

\item{seed}{Optional. The seed for random generation of partitions.}

\item{parGenFun}{The function (object) that will generate random partitions. The default function is   \code{\link{genRandomPar}}. The function has to accept the following parameters: \code{k} (number o of partitions by modes, \code{n} (number of units by modes), \code{seed} (seed value for random generation of partition), \code{addParam} (a list of additional parameters).}

\item{mingr}{Minimal allowed group size.}

\item{maxgr}{Maximal allowed group size.}

\item{addParam}{A list of additional parameters for function specified above. In the usage section they are specified for the default function \code{\link{genRandomPar}}.}

\item{maxTriesToFindNewPar}{The maximum number of partition try when trying to find a new partition to optimize that was not yet checked before - the default value is \code{rep * 1000}.}

\item{skip.par}{The partitions that are not allowed or were already checked and should therefore be skipped.}

\item{printRep}{Should some information about each optimization be printed.}

\item{n}{The number of units by "modes". It is used only for generating random partitions. It has to be set only if there are more than two modes or if there are two modes, but the matrix representing the network is one mode (both modes are in rows and columns).}

\item{nCores}{Number of cores to be used. Value \code{0} means all available cores. It can also be a cluster object.}

\item{useParLapply}{Should \code{parLapplyLB} be used (otherwise \code{foreach} is used). Defaults to true as it needs less dependencies. It might be removed in future releases and only allow the use of parLapplyLB.}

\item{cl}{The cluster to use (if formed beforehand). Defaults to \code{NULL}.}

\item{stopcl}{Should the cluster be stopped after the function finishes. Defaults to \code{is.null(cl)}.}

\item{\dots}{Arguments passed to other functions, see \code{\link{kmBlockC}}.}
}
\value{
A list of class "opt.more.par" containing:
 \item{M}{The one- or multi-mode matrix of the network analyzed}
  \item{res}{If \code{return.all = TRUE} - A list of results the same as \code{best} - one \code{best} for each partition optimized.}
  \item{best}{A list of results from \code{kmBlockC}, only without \code{M}.}
  \item{err}{If \code{return.err = TRUE} - The vector of errors or inconsistencies = -log-likelihoods.}
  \item{ICL}{Integrated classification likelihood for the best partition.}
  \item{checked.par}{If selected - A list of checked partitions. If \code{merge.save.skip.par} is \code{TRUE}, this list also includes the partitions in \code{skip.par}.}
  \item{call}{The call to this function.}
  \item{initial.param}{If selected - The initial parameters are used.}
  \item{Random.seed}{.Random.seed at the end of the function.}
  \item{cl}{Cluster used for parallel computations if supplied as an input parameter.}
}
\description{
A function for optimizing multiple random partitions using k-means one-mode and linked blockmodeling. Calls \code{kmBlockC} for optimizing individual random partitions.
}
\section{Warning}{

It should be noted that the time needed to optimize the partition depends on the number of units (aka nodes) in the networks as well as the number of clusters
due to the underlying algorithm. Hence, partitioning networks with several hundred units and large number of blocks (e.g., >5) can take a long time (from 20 minutes to a few hours or even days).
}

\examples{
# Simple one-mode network
library(blockmodeling)
k<-2
blockSizes<-rep(20,k)
IM<-matrix(c(0.8,.4,0.2,0.8), nrow=2)
if(any(dim(IM)!=c(k,k))) stop("invalid dimensions")

set.seed(2021)
clu<-rep(1:k, times=blockSizes)
n<-length(clu)
M<-matrix(rbinom(n*n,1,IM[clu,clu]),ncol=n, nrow=n)
diag(M)<-0
plotMat(M)

resORP<-kmBlockORPC(M,k=2, rep=10, return.all = TRUE)
plot(resORP)
clu(resORP)


# Linked network
library(blockmodeling)
set.seed(2021)
IM<-matrix(c(0.8,.4,0.2,0.8), nrow=2)
clu<-rep(1:2, each=20)
n<-length(clu)
nClu<-length(unique(clu))
M1<-matrix(rbinom(n^2,1,IM[clu,clu]),ncol=n, nrow=n)
M2<-matrix(rbinom(n^2,1,IM[clu,clu]),ncol=n, nrow=n)
M12<-diag(n)
nn<-c(n,n)
k<-c(2,2)
Ml<-matrix(0, nrow=sum(nn),ncol=sum(nn))
Ml[1:n,1:n]<-M1
Ml[n+1:n,n+1:n]<-M2
Ml[n+1:n, 1:n]<-M12
plotMat(Ml)

resMl<-kmBlockORPC(M=Ml, k=k, n=nn, rep=10)
plot(resMl)
clu(resMl)

}
\references{
Žiberna, Aleš (2020). k-means-based algorithm for blockmodeling linked networks. Social Networks 32(1), 105-126, \doi{10.1016/j.socnet.2019.10.006}.
}
\seealso{
\code{\link{kmBlockC}}
}
\author{
\enc{Aleš, Žiberna}{Ales Ziberna}
}
