% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lfmm.R
\name{lfmm_lasso}
\alias{lfmm_lasso}
\title{LFMM least-squares estimates with lasso penalty (Sparse LFMM)}
\usage{
lfmm_lasso(Y, X, K, nozero.prop = 0.01, mu.num = 100,
  mu.min.ratio = 0.01, mu = NULL, it.max = 100,
  relative.err.epsilon = 1e-06)
}
\arguments{
\item{Y}{a response variable matrix with n rows and p columns.
Each column is a response variable (e.g., SNP genotype,
gene expression level, beta-normalized methylation profile, etc).
Response variables must be encoded as numeric.}

\item{X}{an explanatory variable matrix with n rows and d columns.
Each column corresponds to a distinct explanatory variable (eg. phenotype, exposure, outcome).
Explanatory variables must be encoded as numeric.}

\item{K}{an integer for the number of latent factors in the regression model.}

\item{nozero.prop}{a numeric value for the expected proportion of non-zero effect sizes.}

\item{mu.num}{a numeric value for the number of 'mu' values (advance parameter).}

\item{mu.min.ratio}{(advance parameter) A fraction of \code{mu.max}, the data derived entry value
(i.e. the smallest value for which all coefficients are zero).}

\item{mu}{(advance parameter) Smallest value of \code{mu}. Null value by default.}

\item{it.max}{an integer value for the number of iterations of the algorithm.}

\item{relative.err.epsilon}{a numeric value for a relative convergence error. Determine
whether the algorithm converges or not.}
}
\value{
an object of class \code{lfmm} with the following attributes:
\itemize{
\item U the latent variable score matrix with dimensions n x K,
\item V the latent variable axes matrix with dimensions p x K,
\item B the effect size matrix with dimensions p x d.
}
}
\description{
This function computes regularized least squares estimates
for latent factor mixed models using a lasso penalty.
}
\details{
The algorithm minimizes the following penalized least-squares criterion

The response variable matrix Y and the explanatory variable are centered.
}
\examples{

library(lfmm)

## An EWAS example with Y = methylation data 
## and X = exposure

## Simulate the data 

dat <- lfmm_sampler(n = 100, 
                    p = 1000,
                    K = 3,
                    outlier.prop = 0.02,
                    cs = 0.1,
                    sigma = 0.2,
                    B.sd = 5,
                    B.mean = 0,
                    U.sd = 1.0,
                    V.sd = 1.0)

Y <- scale(dat$Y)
X <- scale(dat$X)

## Fit an LFMM with 2 latent factors
mod.lfmm <- lfmm_lasso(Y = Y,
                       X = X, 
                       K = 3,
                       nozero.prop = 0.02)
                       
## Manhattan plot of sparse effect sizes
effect <- mod.lfmm$B
causal <- dat$outlier

plot(effect, 
     pch = 19, 
     cex = .3,
     xlab = "Probe",
     col = "grey")
     
points(causal, 
       effect[causal], 
       col = "blue")
}
\references{
B. Jumentier, Caye, K., J. Lepeule, and O. François, 2019 Sparse latent factor regression models for genome-wide and epigenome-wide association studies (in prep)
}
\author{
Kevin Caye, Basile Jumentier, Olivier Francois
}
