% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kernel.R
\name{calCov}
\alias{calCov}
\title{Calculate stationary Gaussian process kernel}
\usage{
calCov(
  phi,
  rInput,
  signrInput,
  bandsize = NULL,
  complexity = 3,
  kerneltype = "matern",
  df,
  noiseInjection = 1e-07
)
}
\arguments{
\item{phi}{the kernel hyper-parameters. See details for hyper-parameter specification for each \code{kerneltype}.}

\item{rInput}{the distance matrix between all time points s and t, i.e., |s - t|}

\item{signrInput}{the sign matrix of the time differences, i.e., sign(s - t)}

\item{bandsize}{size for band matrix approximation. See details.}

\item{complexity}{integer value for the complexity of the kernel calculations desired:
\itemize{
  \item 0 includes C only
  \item 1 additionally includes Cprime, Cdoubleprime, dCdphi
  \item 2 or above additionally includes Ceigen1over, CeigenVec, Cinv, mphi, Kphi, Keigen1over, KeigenVec, Kinv, mphiLeftHalf, dCdphiCube
}
See details for their definitions.}

\item{kerneltype}{must be one of \code{matern}, \code{rbf}, \code{compact1}, \code{periodicMatern}, \code{generalMatern}, \code{rationalQuadratic}. See details for the kernel formulae.}

\item{df}{degrees of freedom, for \code{generalMatern} and \code{rationalQuadratic} kernels only.  Default is \code{df=2.01} for \code{generalMatern} and \code{df=0.01} for \code{rationalQuadratic}.}

\item{noiseInjection}{a small value added to the diagonal elements of C and Kphi for numerical stability}
}
\value{
A list containing the kernel calculations included by the value of \code{complexity}.
}
\description{
Covariance calculations for Gaussian process kernels.
Currently supports matern, rbf, compact1, periodicMatern, generalMatern, and rationalQuadratic kernels.
Can also return m_phi and other additional quantities useful for ODE inference.
}
\details{
The covariance formulae and the hyper-parameters \code{phi} for the supported kernels are as follows.  Stationary kernels have \eqn{C(s,t) = C(r)} where \eqn{r = |s-t|} is the distance between the two time points.   Generally, the hyper-parameter \code{phi[1]} controls the overall variance level while \code{phi[2]} controls the bandwidth.
\describe{
  \item{\code{matern}}{ This is the simplified Matern covariance with \code{df = 5/2}:
  \deqn{C(r) = phi[1] * (1 + \sqrt 5 r/phi[2] + 5r^2/(3 phi[2]^2)) * \exp(-\sqrt 5 r/phi[2])}
  }
  \item{\code{rbf}}{
  \deqn{C(r) = phi[1] * \exp(-r^2/(2 phi[2]^2))}
  }
  \item{\code{compact1}}{
  \deqn{C(r) = phi[1] * \max(1-r/phi[2],0)^4 * (4r/phi[2]+1) }
  }
  \item{\code{periodicMatern}}{
  Define \eqn{r' =  | \sin(r \pi/phi[3])*2 |}.  Then the covariance is given by \eqn{C(r')} using the Matern formula.
  }
  \item{\code{generalMatern}}{
  \deqn{C(r) = phi[1] * 2^(1-df) / \Gamma(df) * ( \sqrt(2.0 * df) * r / phi[2] )^df * besselK( \sqrt(2.0 * df) * r / phi[2] , df)}
  where \code{besselK} is the modified Bessel function of the second kind.
  }
  \item{\code{rationalQuadratic}}{
  \deqn{C(r) = phi[1] * (1 + r^2/(2 df phi[2]^2))^(-df)}
  }
}

The kernel calculations available and their definitions are as follows: 
\describe{
  \item{C}{The covariance matrix corresponding to the distance matrix \code{rInput}.}
  \item{Cprime}{The cross-covariance matrix  \eqn{d C(s,t) / ds}.}
  \item{Cdoubleprime}{The cross-covariance matrix  \eqn{d^2 C(s,t) / ds dt}.}
  \item{dCdphi}{A list with the matrices \eqn{dC / dphi} for each element of phi.}
  \item{Ceigen1over}{The reciprocals of the eigenvalues of C.}
  \item{CeigenVec}{Matrix of eigenvectors of C.}
  \item{Cinv}{The inverse of C.}
  \item{mphi}{The matrix \code{Cprime * Cinv}.}
  \item{Kphi}{The matrix \code{Cdoubleprime - Cprime * Kinv * t(Cprime)}.}
  \item{Keigen1over}{The reciprocals of the eigenvalues of Kphi.}
  \item{Kinv}{The inverse of Kphi.}
  \item{mphiLeftHalf}{The matrix \code{Cprime * CeigenVec}.}
  \item{dCdphiCube}{\eqn{dC / dphi} as a 3-D array, with the third dimension corresponding to the elements of phi.}
}

If \code{bandsize} is a positive integer, additionally CinvBand, mphiBand, and KinvBand are provided in the return list, which are
band matrix approximations to Cinv, mphi, and Kinv with the specified \code{bandsize}.
}
\examples{
foo  <- outer(0:40, t(0:40),'-')[,1,]
r <- abs(foo)
signr <- -sign(foo)
calCov(c(0.2, 2), r, signr, bandsize=20, kerneltype="generalMatern", df=2.01)

}
