% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/esp_get_capimun.R
\name{esp_get_capimun}
\alias{esp_get_capimun}
\title{Get the location of municipalities of Spain}
\source{
IGN data via a custom CDN (see
\url{https://github.com/rOpenSpain/mapSpain/tree/sianedata}.
}
\usage{
esp_get_capimun(
  year = Sys.Date(),
  epsg = "4258",
  cache = TRUE,
  update_cache = FALSE,
  cache_dir = NULL,
  verbose = FALSE,
  region = NULL,
  munic = NULL,
  moveCAN = TRUE,
  rawcols = FALSE
)
}
\arguments{
\item{year}{Release year. See Details for years available.}

\item{epsg}{projection of the map: 4-digit \href{https://epsg.io/}{EPSG code}.
One of:
\itemize{
\item "4258": ETRS89
\item "4326": WGS84
\item "3035": ETRS89 / ETRS-LAEA
\item "3857": Pseudo-Mercator
}}

\item{cache}{A logical whether to do caching. Default is \code{TRUE}.}

\item{update_cache}{A logical whether to update cache. Default is \code{FALSE}.
When set to \code{TRUE} it would force a fresh download of the source
\code{.geojson} file.}

\item{cache_dir}{A path to a cache directory. The directory can also be
set globally with:
\itemize{
\item \code{options(mapSpain_cache_dir = "path/to/dir")}.
}

See Details on \code{\link[=esp_get_nuts]{esp_get_nuts()}}.}

\item{verbose}{Display information. Useful for debugging,
default is \code{FALSE}.}

\item{region}{Optional. A vector of region names, NUTS or ISO codes
(see \code{\link[=esp_dict_region_code]{esp_dict_region_code()}}.}

\item{munic}{A name or regex expression with the names of the required
municipalities. \code{NULL} would not produce any filtering.}

\item{moveCAN}{A logical \code{TRUE/FALSE} or a vector of coordinates
\code{c(lat, lon)}. It places the Canary Islands close to Spain's mainland.
Initial position can be adjusted using the vector of coordinates.}

\item{rawcols}{Logical. Setting this to \code{TRUE} would add the raw columns of
the dataset provided by IGN.}
}
\value{
A \code{POINT} object.
}
\description{
Get the location of the political powers for each municipality (possibly
the center of the municipality).

Note that this differs of the centroid of the boundaries of the
municipallity, returned by \code{\link[=esp_get_munic]{esp_get_munic()}}.
}
\details{
\code{year} could be passed as a single year ("YYYY" format, as end of year) or
as a specific date ("YYYY-MM-DD" format).
Historical information starts as of 2005.

When using \code{region} you can use and mix names and NUTS codes (levels 1,
2 or 3), ISO codes (corresponding to level 2 or 3) or \code{cpro}.

When calling a superior level (Province, Autonomous Community or NUTS1) ,
all the municipalities of that level would be added.
}
\note{
While \code{moveCAN} is useful for visualization, it would alter the
actual geographical position of the Canary Islands.
}
\examples{
\dontrun{
# This code compares centroid of municipalities against esp_get_capimun
# It also download tiles, make sure you are online

library(sf)

# Get shape
area <- esp_get_munic_siane(munic = "Valladolid", epsg = 3857)

# Area in km2
print(paste0(round(as.double(st_area(area)) / 1000000, 2), " km2"))

# Extract centroid
centroid <- st_centroid(area)

# Compare with capimun
capimun <- esp_get_capimun(munic = "Valladolid", epsg = 3857)


# Get a tile to check
tile <- esp_getTiles(area)

# Check on plot
library(tmap)

tm_shape(tile, raster.downsample = FALSE) +
  tm_rgb() +
  tm_shape(area) +
  tm_borders(col = "grey40") +
  tm_shape(centroid) +
  tm_symbols(col = "red", alpha = 0.4, shape = 19) +
  tm_shape(capimun) +
  tm_symbols(col = "blue", alpha = 0.4, shape = 19)


# Blue dot is located onto the actual city while red dot is located
# in the centroid of the boundaries
}
}
\seealso{
\code{\link[=esp_get_munic]{esp_get_munic()}}, \code{\link{esp_munic.sf}}, \code{\link{esp_codelist}}
}
\author{
dieghernan, \url{https://github.com/dieghernan/}.
}
\concept{political}
