% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cycle_time.R
\name{cycle_time}
\alias{cycle_time}
\title{Cycle time objects}
\usage{
cycle_time(time, cycle, reverse = TRUE)
}
\arguments{
\item{time}{An object belonging to one of the following classes:
\code{\link[base:numeric]{numeric}}, \code{\link[lubridate:duration]{Duration}},
\code{\link[base:difftime]{difftime}}, or \code{\link[hms:hms]{hms}}.}

\item{cycle}{A \code{\link[base:numeric]{numeric}} or
\code{\link[lubridate:duration]{Duration}} object of length \code{1}, equal or greater
than \code{0}, indicating the cycle length in seconds. See the Details section
to learn more.}

\item{reverse}{(optional) a \code{\link[=logical]{logical}} value indicating if the
function must use a reverse cycle for negative values in \code{time}. See
the Details section to learn more (default: \code{TRUE}).}
}
\value{
The same type of object of \code{time} cycled with the \code{cycle} parameter.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#maturing}{\figure{lifecycle-maturing.svg}{options: alt='[Maturing]'}}}{\strong{[Maturing]}}

\code{cycle_time()} cycles time span objects in a predetermined cycle length,
adapting linear time objects into a circular time frame.
}
\details{
\subsection{Linear versus circular time}{

Time can have different "shapes".

If the objective is to measure the duration (time span) of an event, time is
usually measured considering a linear frame, with a fixed point of
\href{https://en.wikipedia.org/wiki/Origin_(mathematics)}{origin}. In this
context, the time value distance itself to infinity in relation to the
origin.

\if{html}{\out{<div class="sourceCode">}}\preformatted{                                   B
                             |----------|
                                        A
                             |---------------------|
 - inf                                                inf +
<----------------------------|----------|----------|------->
 s                           0          5          10     s
                           origin

A + B = 10 + 5 = 15s
}\if{html}{\out{</div>}}

But that's not the only possible "shape" of time, as it can also be measured
in other contexts.

In a "time of day" context, the time will be linked to the rotation of the
earth, "resetting" when a new rotation cycle starts. That brings a different
kind of shape to time: a circular shape. With this shape the time value
encounters the origin at the beginning and end of each cycle.

\if{html}{\out{<div class="sourceCode">}}\preformatted{               - <--- h ---> +
                    origin
                . . . 0 . . .
             .                 .
            .                   .
           .                     .
          .                       .
         .                         .
         18                        6
         .                         .
          .                       .
           .                     .
            .                   .
             .                 .
                . . . 12 . . .

18 + 6 = 0h
}\if{html}{\out{</div>}}

If we transpose this circular time frame to a linear one, it would look like
this:

\if{html}{\out{<div class="sourceCode">}}\preformatted{<----|---------------|---------------|---------------|----->
    0h              12h              0h             12h
  origin                           origin
}\if{html}{\out{</div>}}

Note that now the origin is not fix, but cyclical.

\code{cycle_time()} operates by converting linear time objects using a circular
approach relative to the cycle length (e.g, \code{cycle = 86400} (1 day)).
}

\subsection{Fractional time}{

\code{cycle_time()} uses the \code{\link[base:Arithmetic]{\%\%}} operator to cycle values.
Hence, it can be subject to catastrophic loss of accuracy if \code{time} is
fractional and much larger than \code{cycle}. A warning is given if this is
detected.

\code{\link[base:Arithmetic]{\%\%}} is a \code{builtin} R function that operates like this:

\if{html}{\out{<div class="sourceCode">}}\preformatted{function(a, b) \{
    a - floor(a / b) * b
\}
}\if{html}{\out{</div>}}
}

\subsection{Negative time cycling}{

If \code{time} have a negative value and \code{reverse == FALSE}, \code{cycle_time()} will
perform the cycle considering the absolute value of \code{time} and return the
result with a negative signal.

However, If \code{time} have a negative value and \code{reverse == TRUE} (default),
\code{cycle_time()} will perform the cycle in reverse, relative to its origin.

Example: If you have a -30h time span in a reversed cycle of 24h, the result
will be 18h. By removing the full cycles of -30h you will get -6h (-30 + 24),
and -6h relative to the origin will be 18h.

\if{html}{\out{<div class="sourceCode">}}\preformatted{               - <--- h ---> +
                    origin
                . . . 0 . . .
              .                 .
            .                   .
           .                     .
          .                       .
         .                         .
    (-6) 18                        6 (-18)
         .                         .
          .                       .
           .                     .
            .                   .
             .                 .
                . . . 12 . . .
                    (-12)
}\if{html}{\out{</div>}}
}

\subsection{\code{Period} objects}{

\code{\link[lubridate:period]{Period}} objects are a special type of object
developed by the \link[lubridate:lubridate-package]{lubridate} team that
represents "human units", ignoring possible timeline irregularities. That is
to say that 1 day as \code{\link[lubridate:period]{Period}} can have different time
spans, when looking to a timeline after a irregularity event.

Since the time span of a \code{\link[lubridate:period]{Period}} object can
fluctuate, \code{cycle_time()} don't accept this kind of object. You can transform
it to a \code{\link[lubridate:duration]{Duration}} object and still use the
function, but beware that this can produce errors.

Learn more about \code{\link[lubridate:period]{Period}} objects in the \href{https://r4ds.had.co.nz/dates-and-times.html#periods}{Dates and times} chapter of
Wickham & Grolemund book (n.d.).
}
}
\examples{
## Scalar example

time <- lubridate::dhours(25)
cycle <- lubridate::ddays(1)
cycle_time(time, cycle)
#> [1] "3600s (~1 hours)" # Expected

time <- lubridate::dhours(-25)
cycle <- lubridate::ddays(1)
reverse <- FALSE
cycle_time(time, cycle, reverse)
#> [1] "-3600s (~-1 hours)" # Expected

time <- lubridate::dhours(-25)
cycle <- lubridate::ddays(1)
reverse <- TRUE
cycle_time(time, cycle, reverse)
#> [1] "82800s (~23 hours)" # Expected

## Vector example

time <- c(lubridate::dmonths(24), lubridate::dmonths(13))
cycle <- lubridate::dyears(1)
cycle_time(time, cycle)
#> [1] "0s"                     "2629800s (~4.35 weeks)" # Expected

time <- c(lubridate::dmonths(24), lubridate::dmonths(-13))
cycle <- lubridate::dyears(1)
reverse <- FALSE
cycle_time(time, cycle, reverse)
#> [1] "0s"                       "-2629800s (~-4.35 weeks)" # Expected

time <- c(lubridate::dmonths(24), lubridate::dmonths(-13))
cycle <- lubridate::dyears(1)
reverse <- TRUE
cycle_time(time, cycle, reverse)
#> [1] "0s"                       "28927800s (~47.83 weeks)" # Expected
}
\references{
Wickham, H., & Grolemund, G. (n.d.). \emph{R for data science}. Sebastopol, CA:
O'Reilly Media. \url{https://r4ds.had.co.nz}
}
\seealso{
Other utility functions: 
\code{\link{assign_date}()},
\code{\link{pretty_mctq}()},
\code{\link{qplot_walk}()},
\code{\link{random_mctq}()},
\code{\link{raw_data}()},
\code{\link{round_time}()},
\code{\link{shorter_interval}()},
\code{\link{sum_time}()}
}
\concept{utility functions}
