% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mcrpure.R
\name{mcrpure}
\alias{mcrpure}
\title{Multivariate curve resolution based on pure variables}
\usage{
mcrpure(
  x,
  ncomp,
  purevars = NULL,
  offset = 0.05,
  exclrows = NULL,
  exclcols = NULL,
  info = ""
)
}
\arguments{
\item{x}{spectra of mixtures (matrix or data frame).}

\item{ncomp}{maximum number of components to calculate.}

\item{purevars}{vector with indices for pure variables (optional, if you want to provide the variables directly).}

\item{offset}{offset for correcting noise in computing maximum angles (should be value within [0, 1)).}

\item{exclrows}{rows to be excluded from calculations (numbers, names or vector with logical values).}

\item{exclcols}{columns to be excluded from calculations (numbers, names or vector with logical values).}

\item{info}{a short text with description of the case (optional).}
}
\value{
Returns an object of \code{\link{mcrpure}} class with the following fields:
\item{resspec}{matrix with resolved spectra.}
\item{rescont}{matrix with resolved contributions.}
\item{purevars}{indices of the selected pure variables.}
\item{purevals}{purity values for the selected pure variables.}
\item{purityspec}{purity spectra (matrix with purity values for each variable and component).}
\item{expvar }{vector with explained variance for each component (in percent).}
\item{cumexpvar }{vector with cumulative explained variance for each component (in percent).}
\item{offset}{offset value used to compute the purity}
\item{ncomp}{number of resolved components}
\item{info }{information about the model, provided by user when build the model.}


More details and examples can be found in the Bookdown tutorial.
}
\description{
\code{mcrpure} allows to resolve spectroscopic data to linear combination of individual spectra
and contributions using the pure variables approach.
}
\details{
The method estimates purity of each variable and then uses the purest ones to decompose the
spectral data into spectra (`resspec`) and contributions (`rescont`) of individual
chemical components by ordinary least squares.

The pure variabes are identified using stepwise maximum angle calculations and described in
detail in [1]. So the purity of a spectral variable (wavelength, wavenumber) is actually an
angle (measured in degrees) between the variable and vector of ones for the first component;
and between the variable and space formed by previously found pure variables for the other
components.
}
\examples{
library(mdatools)

# resolve mixture of carbonhydrates Raman spectra

data(carbs)
m = mcrpure(carbs$D, ncomp = 3)

# examples for purity spectra plot (you can select which components to show)
par(mfrow = c(2, 1))
plotPuritySpectra(m)
plotPuritySpectra(m, comp = 2:3)

# you can do it manually and combine e.g. with original spectra
par(mfrow = c(1, 1))
mdaplotg(
   list(
      "spectra" = prep.norm(carbs$D, "area"),
      "purity" = prep.norm(mda.subset(mda.t(m$resspec), 1), "area")
   ), col = c("gray", "red"), type = "l"
)

# show the maximum purity for each component
par(mfrow = c(1, 1))
plotPurity(m)

# plot cumulative and individual explained variance
par(mfrow = c(1, 2))
plotVariance(m)
plotCumVariance(m)

# plot resolved spectra (all of them or individually)
par(mfrow = c(2, 1))
plotSpectra(m)
plotSpectra(m, comp = 2:3)

# plot resolved contributions (all of them or individually)
par(mfrow = c(2, 1))
plotContributions(m)
plotContributions(m, comp = 2:3)

# of course you can do this manually as well, e.g. show original
# and resolved spectra
par(mfrow = c(1, 1))
mdaplotg(
   list(
      "original" = prep.norm(carbs$D, "area"),
      "resolved" = prep.norm(mda.subset(mda.t(m$resspec), 1), "area")
   ), col = c("gray", "red"), type = "l"
)

# in case if you have reference spectra of components you can compare them with
# the resolved ones:
par(mfrow = c(3, 1))
for (i in 1:3) {
   mdaplotg(
      list(
         "pure" = prep.norm(mda.subset(mda.t(carbs$S), 1), "area"),
         "resolved" = prep.norm(mda.subset(mda.t(m$resspec), 1), "area")
      ), col = c("gray", "red"), type = "l", lwd = c(3, 1)
   )
}

# See bookdown tutorial for more details.

}
\references{
1. Willem Windig, Neal B. Gallagher, Jeremy M. Shaver, Barry M. Wise. A new approach for
interactive self-modeling mixture analysis. Chemometrics and Intelligent Laboratory Systems,
77 (2005) 85–96. DOI: 10.1016/j.chemolab.2004.06.009
}
\seealso{
Methods for \code{mcrpure} objects:
\tabular{ll}{
   \code{summary.mcrpure} \tab shows some statistics for the case.\cr
   \code{\link{unmix.mcrpure}} \tab makes unmixing of new set of spectra.\cr
   \code{\link{predict.mcrpure}} \tab computes contributions by projection of new spectra to
   the resolved ones.\cr
}

Plotting methods for \code{mcrpure} objects:
\tabular{ll}{
   \code{\link{plotPurity.mcrpure}} \tab shows plot with maximum purity of each component.\cr
   \code{\link{plotPuritySpectra.mcrpure}} \tab shows plot with purity spectra.\cr
   \code{\link{plotSpectra.mcr}} \tab shows plot with resolved spectra.\cr
   \code{\link{plotContributions.mcr}} \tab shows plot with resolved contributions.\cr
   \code{\link{plotVariance.mcr}} \tab shows plot with explained variance.\cr
   \code{\link{plotCumVariance.mcr}} \tab shows plot with cumulative explained variance.\cr
}
}
\author{
Sergey Kucheryavskiy (svkucheryavski@gmail.com)
}
