medsens <- function(x, rho.by=.1, sims=1000, eps=.Machine$double.eps)
{
    model.y <- eval(x$call.y, envir=x$env.y)
    model.m <- eval(x$call.m, envir=x$env.m)
    class.y <- class(model.y)[1]
    class.m <- class(model.m)[1]
    treat <- x$treat
    mediator <- x$mediator
    INT <- x$INT
    
    #########################################################
    ## Setting Up Sensitivity Parameters
    #########################################################
    rho <- seq(-1+rho.by, 1-rho.by, rho.by)
    R2star.prod <- rho^2
    
    #########################################################
    ## CASE 1: Continuous Outcome + Continuous Mediator
    #########################################################
    if(class.y=="lm" & class.m=="lm") {
        d0 <- matrix(NA, length(rho), 1)
        d1 <- matrix(NA, length(rho), 1)
        d0.var <- matrix(NA, length(rho), 1)
        d1.var <- matrix(NA, length(rho), 1)
        
        y.t.data <- model.frame(model.y)    
        if(is.factor(y.t.data[,paste(treat)])==TRUE){
            cat.c <- levels(y.t.data[,treat])[1] 
            cat.t <- levels(y.t.data[,treat])[2]
            T.cat <- paste(treat,cat.t, sep="") 
            } else {
            cat.c <- NULL
            cat.t <- NULL
            T.cat <- paste(treat,cat.t, sep="")
            }
            
        if(INT==TRUE){
        int.lab <- paste(T.cat,mediator, sep=":")
        t.m <- paste(treat,mediator, sep=":")
            }
                
        #Estimate Error Correlation
        if(INT==TRUE){
            mod.y <- update(model.y,as.formula(paste(". ~ . -", t.m, "-", mediator)))
            } else {
            mod.y <- update(model.y,as.formula(paste(". ~ . -", mediator)))
                }
        err.cr <- cor(model.m$resid, mod.y$resid)
                    
        
        for(i in 1:length(rho)){
        
        e.cor <- rho[i]
        
        b.dif <- 1
        
        #Stacked Equations
        m.mat <- model.matrix(model.m)
        y.mat <- model.matrix(model.y)
        m.k <- ncol(m.mat)
        m.n <- nrow(m.mat)
        y.k <- ncol(y.mat)
        y.n <- nrow(y.mat)
        n <- y.n
        m.zero <- matrix(0, m.n, y.k)
        y.zero <- matrix(0, y.n, m.k)
        X.1 <- cbind(m.mat, m.zero)
        X.2 <- cbind(y.zero, y.mat)
        X <- rbind(X.1, X.2)
        
        m.frame <- model.frame(model.m)
        y.frame <- model.frame(model.y)
        Y.c <- rbind(as.matrix(m.frame[,1]), as.matrix(y.frame[,1]))
        
        #Estimates of OLS Start Values
        inxx <- solve(crossprod(X))
        b.ols <- inxx %*% crossprod(X,Y.c)
        b.tmp <- b.ols
        
        while(abs(b.dif) > eps){
        
        e.hat <- as.matrix(Y.c - (X %*% b.tmp))
        
        e.1 <- e.hat[1:n]
        e.2 <- e.hat[(n+1): (2*n)]
        
        sd.1 <- sd(e.1)
        sd.2 <- sd(e.2)
        
        omega <- matrix(NA, 2,2)
        
        omega[1,1] <- crossprod(e.1)/(n-1)
        omega[2,2] <- crossprod(e.2)/(n-1) 
        omega[2,1] <- e.cor*sd.1*sd.2
        omega[1,2] <- e.cor*sd.1*sd.2
        
        I <- Diagonal(n)
        omega.i <- solve(omega)
        v.i <-  kronecker(omega.i, I) 
        Xv.i <- t(X) %*% v.i
        X.sur <- Xv.i %*% X 
        b.sur <- solve(X.sur) %*% Xv.i %*% Y.c
        
        #Variance-Covariance Matrix
        v.cov <- solve(X.sur)
        
        b.old <- b.tmp
        b.dif <- sum((b.sur - b.old)^2)
        b.tmp <- b.sur
        
        }
        
        #Name Elements - Extract Quantities
        m.names <- names(model.m$coef)
        y.names <- names(model.y$coef)
        b.names <- c(m.names, y.names)
        row.names(b.sur) <- b.names
        m.coefs <- as.matrix(b.sur[1:m.k])
        y.coefs <- as.matrix(b.sur[(m.k+1):(m.k+y.k)])
        row.names(m.coefs) <- m.names
        row.names(y.coefs) <- y.names
        rownames(v.cov) <- b.names
        colnames(v.cov) <- b.names
        v.m <- v.cov[1:m.k,1:m.k]
        v.y <- v.cov[(m.k+1):(m.k+y.k),(m.k+1):(m.k+y.k)]
        
        #Save Estimates
        if(INT==TRUE){
            d0[i,] <- m.coefs[paste(T.cat),]*y.coefs[paste(mediator),] 
            d1[i,] <- m.coefs[paste(T.cat),]*(y.coefs[paste(mediator),] + y.coefs[paste(int.lab),])
            } else {
                d0[i,] <- m.coefs[paste(T.cat),]*y.coefs[paste(mediator),]
                d1[i,] <- m.coefs[paste(T.cat),]*y.coefs[paste(mediator),] 
                }
        
        #Save Variance Estimates
        if(INT==TRUE){
            d0.var[i,] <- (y.coefs[paste(mediator),] + 0*y.coefs[paste(int.lab),])^2*v.m[T.cat,T.cat] + m.coefs[paste(T.cat),]^2*(v.y[mediator,mediator] + 0*v.y[int.lab, int.lab] + 0*2*v.y[mediator, int.lab])
            d1.var[i,] <- (y.coefs[paste(mediator),] + y.coefs[paste(int.lab),])^2*v.m[T.cat,T.cat] + m.coefs[paste(T.cat),]^2*(v.y[mediator,mediator] + v.y[int.lab, int.lab] + 2*v.y[mediator, int.lab])
            } else {
            d0.var[i,] <- (m.coefs[paste(T.cat),]^2*v.y[mediator,mediator]) + (y.coefs[paste(mediator),]^2*v.m[T.cat,T.cat])
            d1.var[i,] <- (m.coefs[paste(T.cat),]^2*v.y[mediator,mediator]) + (y.coefs[paste(mediator),]^2*v.m[T.cat,T.cat])
                }
                
        rm(b.sur, m.coefs, y.coefs, v.cov, v.m, v.y)
        
        }
        
        if(INT==FALSE){
        upper.d0 <- d0 + qnorm(0.975) * sqrt(d0.var)
        lower.d0 <- d0 - qnorm(0.975) * sqrt(d0.var)
        upper.d1 <- NULL
        lower.d1 <- NULL
        ind.d0 <- as.numeric(lower.d0 < 0 & upper.d0 > 0)
        ind.d1 <- NULL  
            } else {
        upper.d0 <- d0 + qnorm(0.975) * sqrt(d0.var)
        lower.d0 <- d0 - qnorm(0.975) * sqrt(d0.var)
        upper.d1 <- d1 + qnorm(0.975) * sqrt(d1.var)
        lower.d1 <- d1 - qnorm(0.975) * sqrt(d1.var)    
        ind.d0 <- as.numeric(lower.d0 < 0 & upper.d0 > 0)
        ind.d1 <- as.numeric(lower.d1 < 0 & upper.d1 > 0)
                }
                
        # Save R2 tilde values
        r.sq.m <- summary(model.m)$r.squared
        r.sq.y <- summary(model.y)$r.squared
        R2tilde.prod <- rho^2*(1-r.sq.m)*(1-r.sq.y)
        
        # Calculate rho at which ACME=0
        if(INT==TRUE){
        ii <- which(abs(d0-0)==min(abs(d0-0)))
        kk <- which(abs(d1-0)==min(abs(d1-0)))
        err.cr.1 <- rho[ii]
        err.cr.2 <- rho[kk]
        err.cr <- c(err.cr.1, err.cr.2)
        }
        R2star.thresh <- err.cr^2
        R2tilde.thresh <- err.cr^2*(1-r.sq.m)*(1-r.sq.y)

        type <- "ct"
        out <- list(rho = rho, err.cr=err.cr, d0=d0, d1=d1, upper.d0=upper.d0,
        lower.d0=lower.d0, upper.d1=upper.d1, lower.d1=lower.d1, ind.d0=ind.d0,
        ind.d1=ind.d1, R2star.prod=R2star.prod, R2tilde.prod=R2tilde.prod,
        R2star.thresh=R2star.thresh, R2tilde.thresh=R2tilde.thresh,
        r.square.y=r.sq.y, r.square.m=r.sq.m,
        rho.by=rho.by, INT=INT,
        tau=NULL, upper.tau=NULL, lower.tau=NULL, nu=NULL, upper.nu=NULL, lower.nu=NULL, type=type)
        class(out) <- "medsens"
        out

    ## END OF CASE 1: Continuous Outcome + Continuous Mediator    
    } else

    #########################################################
    ## CASE 2: Continuous Outcome + Binary Mediator
    #########################################################
    if (class.y == "lm" & class.m == "glm") {

        # Step 0: Setting Variable labels
        ## Uppercase letters (e.g. T) = labels in the input matrix
        ## Uppercase letters + ".out" (e.g. T.out) = labels in the regression output
        
        y.t.data <- model.frame(model.y)
        Y <- colnames(y.t.data)[1]
        if(is.factor(y.t.data[,treat])==TRUE){
            cat.c <- levels(y.t.data[,treat])[1] 
            cat.t <- levels(y.t.data[,treat])[2]
            T.out <- paste(treat,cat.t, sep="") 
            } else {
            cat.c <- NULL
            cat.t <- NULL
            T.out <- paste(treat,cat.t, sep="")
            }
        
        if(is.factor(y.t.data[,mediator])==TRUE){
            cat.m0 <- levels(y.t.data[,mediator])[1] 
            cat.m1 <- levels(y.t.data[,mediator])[2]
            M.out <- paste(mediator,cat.m1, sep="") 
            } else {
            cat.m0 <- NULL
            cat.m1 <- NULL
            M.out <- paste(mediator,cat.m1, sep="")
            }
        
        if(INT==TRUE){
        TM <- paste(treat,mediator, sep=":")
        TM.out <- paste(T.out,M.out, sep=":")
            }
        
        ## Variable values (LABEL.value)
        Y.value <- y.t.data[,1]
        y.k <- length(model.y$coef)
        
        # Step 1: Pre-loop computations
        # Step 1-1: Bootstrap M model parameters
        Mmodel.coef <- model.m$coef
        Mmodel.var.cov <- vcov(model.m)
        Mmodel.coef.boot <- mvrnorm(sims, mu=Mmodel.coef, Sigma=Mmodel.var.cov) # bootstrap M-model parameters
        
        # Step 1-2: Bootstrap lambda_0 and lambda_1; lambdas are (n x sims) matrix
        m.mat <- model.matrix(model.m)
        m.mat.1 <- model.matrix(model.m)
        m.mat.1[,T.out] <- 1 # M-model matrix with t=1
        m.mat.0 <- model.matrix(model.m)
        m.mat.0[,T.out] <- 0 # M-model matrix with t=0
        mu.boot <- m.mat %*% t(Mmodel.coef.boot) # E(M|T,X)
        mu.1.boot <- m.mat.1 %*% t(Mmodel.coef.boot) # E(M|T=1,X)
        mu.0.boot <- m.mat.0 %*% t(Mmodel.coef.boot) # E(M|T=0,X)
        lambda11 <- dnorm(-mu.1.boot) / pnorm(mu.1.boot) #lambda for m=1,t=1
        lambda10 <- dnorm(-mu.0.boot) / pnorm(mu.0.boot) #lambda for m=1,t=0
        lambda01 <- -dnorm(-mu.1.boot) / pnorm(-mu.1.boot) #lambda for m=0,t=1
        lambda00 <- -dnorm(-mu.0.boot) / pnorm(-mu.0.boot) #lambda for m=0,t=0
        
        # Step 1-3: Define lambda function
        lambda <- function(mmodel, mcoef) {
            mu <- model.matrix(mmodel) %*% mcoef
            m <- mmodel$y #this is M
            return((m*dnorm(-mu)-(1-m)*dnorm(-mu))/(m*pnorm(mu)+(1-m)*pnorm(-mu)))
        }

        # Step 2: Rho loop
        ## Step 2-0: Initialize containers
        d0 <- d1 <- matrix(NA, length(rho), 1)
        upper.d0 <- upper.d1 <- lower.d0 <- lower.d1 <- matrix(NA, length(rho), 1)
        ind.d0 <- ind.d1 <- matrix(NA, length(rho), 1)
        Ymodel.coef.boot <- matrix(NA, sims, y.k)
        colnames(Ymodel.coef.boot) <- names(model.y$coef)
        sigma.3.boot <- rep(NA, sims)
        d0.boot <- d1.boot <- rep(NA, sims)
        ## START OF RHO LOOP
        for(i in 1:length(rho)){
            
            ## START OF BOOTSTRAP LOOP
            for(k in 1:sims){
            ## Step 2-1: Obtain the initial Y model with the correction term
            adj <- lambda(model.m, Mmodel.coef.boot[k,]) * rho[i] # the adjustment term
            w <- 1 - rho[i]^2*lambda(model.m, Mmodel.coef.boot[k,])*(lambda(model.m, Mmodel.coef.boot[k,]) + mu.boot[,k])
            y.t.data.adj <- data.frame(y.t.data, w, adj)
            model.y.adj <- update(model.y, as.formula(paste(". ~ . + adj")), weights=w, data=y.t.data.adj)
            sigma.3 <- summary(model.y.adj)$sigma
            
            ## Step 2-2: Update the Y model via Iterative FGLS
            sigma.dif <- 1
            while(abs(sigma.dif) > eps){
                Y.star <- Y.value - sigma.3 * adj
                y.t.data.star <- data.frame(Y.star, y.t.data.adj)
                model.y.update <- update(model.y, as.formula(paste("Y.star ~ .")), weights=w, data=y.t.data.star)
                sigma.3.temp <- summary(model.y.update)$sigma
                sigma.dif <- sigma.3.temp - sigma.3
                sigma.3 <- sigma.3.temp
            }
            
            ## Step 2-3: Bootstrap Y model parameters
            Ymodel.coef <- model.y.update$coef
            Ymodel.var.cov <- vcov(model.y.update)
            Ymodel.coef.boot[k,] <- mvrnorm(1, mu=Ymodel.coef, Sigma=Ymodel.var.cov) #draw one bootstrap sample of Y-model parameters for each k
            
            ## Step 2-4: Bootstrap ACMEs; means are over observations
            d0.boot[k] <- mean( (Ymodel.coef.boot[k,M.out]) * (pnorm(mu.1.boot[,k]) - pnorm(mu.0.boot[,k])) )
            if(INT==TRUE){
                d1.boot[k] <- mean( (Ymodel.coef.boot[k,M.out] + Ymodel.coef.boot[k,TM.out]) * 
                    (pnorm(mu.1.boot[,k]) - pnorm(mu.0.boot[,k])) )
                } else {
                d1.boot[k] <- mean((Ymodel.coef.boot[k,M.out]) * 
                    (pnorm(mu.1.boot[,k]) - pnorm(mu.0.boot[,k])) )
                }

            ## END OF BOOTSTAP LOOP
            }
            
        ## Step 2-5: Compute Outputs
        d0[i] <- mean(d0.boot)
        d1[i] <- mean(d1.boot)
        upper.d0[i] <- quantile(d0.boot, 0.975)
        upper.d1[i] <- quantile(d1.boot, 0.975)
        lower.d0[i] <- quantile(d0.boot, 0.025)
        lower.d1[i] <- quantile(d1.boot, 0.025)
        ind.d0[i] <- as.numeric(lower.d0[i] < 0 & upper.d0[i] > 0)
        ind.d1[i] <- as.numeric(lower.d1[i] < 0 & upper.d1[i] > 0)
        
        ## END OF RHO LOOP
        }
        
        # Save R2 tilde values
        fitted<- m.mat %*% Mmodel.coef
        var.mstar <- var(fitted)
        r.sq.m <- var.mstar/(1+var.mstar)
        r.sq.y <- summary(model.y)$r.squared
        R2tilde.prod <- rho^2*(1-r.sq.m)*(1-r.sq.y)
        
        # Calculate rho at which ACME=0
        if(INT==TRUE){
        ii <- which(abs(d0-0)==min(abs(d0-0)))
        kk <- which(abs(d1-0)==min(abs(d1-0)))
        err.cr.1 <- rho[ii]
        err.cr.2 <- rho[kk]
        err.cr <- c(err.cr.1, err.cr.2)
        } else {
        ii <- which(abs(d0-0)==min(abs(d0-0)))
        err.cr <- rho[ii]   
            }
        R2star.thresh <- err.cr^2
        R2tilde.thresh <- err.cr^2*(1-r.sq.m)*(1-r.sq.y)
        
        # Step 3: Output
        type <- "bm"
        out <- list(rho = rho, d0=d0, d1=d1, upper.d0=upper.d0,
        lower.d0=lower.d0, upper.d1=upper.d1, lower.d1=lower.d1, ind.d0=ind.d0,
        ind.d1=ind.d1, R2star.prod=R2star.prod, R2tilde.prod=R2tilde.prod,
        R2star.thresh=R2star.thresh, R2tilde.thresh=R2tilde.thresh,
        r.square.y=r.sq.y, r.square.m=r.sq.m,
        rho.by=rho.by, INT=INT, tau=NULL, upper.tau=NULL,
        lower.tau=NULL, nu=NULL, upper.nu=NULL, lower.nu=NULL,type=type,
        err.cr=err.cr)
        class(out) <- "medsens"
        out

    ## END OF CASE 2: Continuous Outcome + Binary Mediator
    } else
    
    #########################################################
    ## CASE 3: Binary Outcome + Continuous Mediator
    #########################################################
    if(class.y=="glm" & class.m=="lm") {

        if(INT==TRUE){
        stop("Sensitivity Analysis Not Available Binary Outcome With Interactions \n")
        }
        # Step 0: Setting Variable labels
        ## Uppercase letters (e.g. T) = labels in the input matrix
        ## Uppercase letters + ".out" (e.g. T.out) = labels in the regression output
        
        y.t.data <- model.frame(model.y)
        Y <- colnames(y.t.data)[1]
        if(is.factor(y.t.data[,treat])==TRUE){
            cat.c <- levels(y.t.data[,treat])[1] 
            cat.t <- levels(y.t.data[,treat])[2]
            T.out <- paste(treat,cat.t, sep="") 
            } else {
            cat.c <- NULL
            cat.t <- NULL
            T.out <- paste(treat,cat.t, sep="")
            }
        
        if(is.factor(y.t.data[,mediator])==TRUE){
            cat.m0 <- levels(y.t.data[,mediator])[1] 
            cat.m1 <- levels(y.t.data[,mediator])[2]
            M.out <- paste(mediator,cat.m1, sep="") 
            } else {
            cat.m0 <- NULL
            cat.m1 <- NULL
            M.out <- paste(mediator,cat.m1, sep="")
            }
        
        if(INT==TRUE){
        TM <- paste(treat,mediator, sep=":")
        TM.out <- paste(T.out,M.out, sep=":")
            }
        
        # Step 1: Obtain Model Parameters
        ## Step 1-1: Bootstrap M model parameters
        Mmodel.coef <- model.m$coef
        m.k <- length(model.m$coef)
        Mmodel.var.cov <- vcov(model.m)
        Mmodel.coef.boot <- mvrnorm(sims, mu=Mmodel.coef, Sigma=Mmodel.var.cov)
        if(is.factor(y.t.data[,treat])==TRUE){
         beta2.boot <- Mmodel.coef.boot[,T.out] 
            } else {
          beta2.boot <- Mmodel.coef.boot[,treat]    
                }

        sigma.2 <- summary(model.m)$sigma
        sig2.shape <- model.m$df/2
        sig2.invscale <- (model.m$df/2) * sigma.2^2
        sigma.2.boot <- sqrt(1 / rgamma(sims, shape = sig2.shape, scale = 1/sig2.invscale))
        
        ## Step 1-2: Bootstrap Y model parameters
        Ymodel.coef <- model.y$coef
        Ymodel.var.cov <- vcov(model.y)
        y.k <- length(Ymodel.coef)
        Ymodel.coef.boot <- mvrnorm(sims, mu=Ymodel.coef, Sigma=Ymodel.var.cov)
        colnames(Ymodel.coef.boot) <- names(Ymodel.coef)
        gamma.tilde <- Ymodel.coef.boot[,M.out]

        # Step 2: Compute ACME via the procedure in IKT
        ## Step 2-1: Estimate Error Correlation from inconsistent estimate of Y on M
        rho12.boot <- (sigma.2.boot * gamma.tilde) / (1 + sqrt(sigma.2.boot^2*gamma.tilde^2))
        
        ## Step 2-2: Calculate alpha_1, beta_1 and xi_1
        YTmodel.coef.boot <- Ymodel.coef.boot[,!colnames(Ymodel.coef.boot)%in%M.out] * sqrt(1-rho12.boot^2) %x% t(rep(1,y.k-1)) + Mmodel.coef.boot * (rho12.boot/sigma.2.boot) %x% t(rep(1,y.k-1))
        
        ## Step 2-3: Calculate Gamma
        ## Data matrices for the Y model less M
        y.mat.1 <- model.matrix(model.y)[,!colnames(model.matrix(model.y))%in%M.out]
        y.mat.1[,T.out] <- 1
        y.mat.0 <- model.matrix(model.y)[,!colnames(model.matrix(model.y))%in%M.out]
        y.mat.0[,T.out] <- 0
        
        ## Initialize objects before the rho loop
        d0 <- d1 <- rep(NA, length(rho))
        upper.d0 <- upper.d1 <- lower.d0 <- lower.d1 <- rep(NA, length(rho))
        ind.d0 <- ind.d1 <- rep(NA, length(rho))
        tau <- nu <- rep(NA, length(rho))
        upper.tau <- upper.nu <- lower.tau <- lower.nu <- rep(NA, length(rho))
        d0.boot <- d1.boot <- rep(NA, sims)
        tau.boot <- nu.boot <- rep(NA, sims)
        
        ## START OF RHO LOOP
        for(i in 1:length(rho)){
            gamma.boot <- (-rho[i] + rho12.boot*sqrt((1-rho[i]^2)/(1-rho12.boot^2)))/sigma.2.boot
            for(k in 1:sims){
            d0.boot[k] <- mean( pnorm(y.mat.0 %*% YTmodel.coef.boot[k,] + 
                gamma.boot[k]*beta2.boot[k]/sqrt(gamma.boot[k]^2*sigma.2.boot[k]^2+2*gamma.boot[k]*rho[i]*sigma.2.boot[k]+1)) -
                pnorm(y.mat.0 %*% YTmodel.coef.boot[k,]) )
            d1.boot[k] <- mean( pnorm(y.mat.1 %*% YTmodel.coef.boot[k,]) - pnorm(y.mat.1 %*% YTmodel.coef.boot[k,] - 
                gamma.boot[k]*beta2.boot[k]/sqrt(gamma.boot[k]^2*sigma.2.boot[k]^2+2*gamma.boot[k]*rho[i]*sigma.2.boot[k]+1)) )
            tau.boot[k] <- mean( pnorm(y.mat.1 %*% YTmodel.coef.boot[k,]) - pnorm(y.mat.0 %*% YTmodel.coef.boot[k,]) )
            nu.boot[k] <- (d0.boot[k] + d1.boot[k])/(2*tau.boot[k])
            }
            
        ## Step 2-4: Compute Outputs
        d0[i] <- mean(d0.boot) # ACME(t=0)
        d1[i] <- mean(d1.boot) # ACME(t=1)
        upper.d0[i] <- quantile(d0.boot, 0.975)
        upper.d1[i] <- quantile(d1.boot, 0.975)
        lower.d0[i] <- quantile(d0.boot, 0.025)
        lower.d1[i] <- quantile(d1.boot, 0.025)
        tau[i] <- mean(tau.boot) # ATE
        nu[i] <- mean(nu.boot) # Proportion Mediated
        upper.tau[i] <- quantile(tau.boot, 0.975)
        upper.nu[i] <- quantile(nu.boot, 0.975)
        lower.tau[i] <- quantile(tau.boot, 0.025)
        lower.nu[i] <- quantile(nu.boot, 0.025)
        ind.d0[i] <- as.numeric(lower.d0[i] < 0 & upper.d0[i] > 0)
        ind.d1[i] <- as.numeric(lower.d1[i] < 0 & upper.d1[i] > 0)
        
        ## END OF RHO LOOP
        }
        
        # Save R2 tilde values
        r.sq.m <- summary(model.m)$r.squared
        y.mat <- model.matrix(model.y)
        fitted<- y.mat %*% Ymodel.coef
        var.ystar <- var(fitted)
        r.sq.y <-var.ystar/(1+var.ystar)
        R2tilde.prod <- rho^2*(1-r.sq.m)*(1-r.sq.y)
        
        # Calculate rho at which ACME=0
        if(INT==TRUE){
        ii <- which(abs(d0-0)==min(abs(d0-0)))
        kk <- which(abs(d1-0)==min(abs(d1-0)))
        err.cr.1 <- rho[ii]
        err.cr.2 <- rho[kk]
        err.cr <- c(err.cr.1, err.cr.2)
        } else {
        ii <- which(abs(d0-0)==min(abs(d0-0)))
        err.cr <- rho[ii]   
            }
        
        ## Step 3: Output
        type <- "bo"
        err.cr <- mean(rho12.boot) # Rho_12 estimate
        R2star.thresh <- err.cr^2
        R2tilde.thresh <- err.cr^2*(1-r.sq.m)*(1-r.sq.y)
        out <- list(rho = rho, err.cr=err.cr, d0=d0, d1=d1, upper.d0=upper.d0,
        lower.d0=lower.d0, upper.d1=upper.d1, lower.d1=lower.d1, ind.d0=ind.d0,
        ind.d1=ind.d1, R2star.prod=R2star.prod, R2tilde.prod=R2tilde.prod,
        R2star.thresh=R2star.thresh, R2tilde.thresh=R2tilde.thresh,
        r.square.y=r.sq.y, r.square.m=r.sq.m,
        tau=tau, upper.tau=upper.tau, lower.tau=lower.tau, nu=nu, upper.nu=upper.nu,
        lower.nu=lower.nu, INT=INT, rho.by=rho.by, type=type)
        class(out) <- "medsens"
        out

    ## END OF CASE 3: Binary Outcome + Continuous Mediator    
    }

## END OF SENSITIVITY FUNCTION
}



print.medsens <- function(x, ...){
    print(unlist(x[1:16]))
    invisible(x)
    }

summary.medsens <- function(object, ...)
    structure(object, class = c("summary.medsens", class(object)))
 
print.summary.medsens <- function(x, ...){
    if(x$type=="ct"){
        if(x$INT==FALSE){
            tab <- cbind(x$rho, round(x$d0,4), round(x$lower.d0,4), round(x$upper.d0, 4), x$ind.d0, x$R2star.prod, round(x$R2tilde.prod,4))
            if(sum(x$ind.d0)==1){
                tab <- as.matrix(tab[x$ind.d0==1, -5])
                tab <- t(tab)
            } else {
                tab <- tab[x$ind.d0==1, -5] 
            }
            colnames(tab) <-  c("Rho", "Med. Eff.", "95% CI Lower", "95% CI Upper", "R^2_M*R^2_Y*", "R^2_M~R^2_Y~")
            rownames(tab) <- NULL
            cat("\nMediation Sensitivity Analysis\n")
            cat("\nSensitivity Region\n\n")
            print(tab)
            cat("\nRho at which ACME = 0:", round(x$err.cr, 4), "\n\n")
            cat("\nR^2_M*R^2_Y* at which ACME = 0:", round(x$R2star.thresh, 4), "\n\n")
            cat("\nR^2_M~R^2_Y~ at which ACME = 0:", round(x$R2tilde.thresh, 4), "\n\n")
            invisible(x)    
        } else {
            tab.d0 <- cbind(x$rho, round(x$d0,4), round(x$lower.d0,4), round(x$upper.d0, 4), x$ind.d0, x$R2star.prod, round(x$R2tilde.prod, 4))
            if(sum(x$ind.d0)==1){
                tab.d0 <- as.matrix(tab.d0[x$ind.d0==1, -5])
                tab.d0 <- t(tab.d0)
            } else {
                tab.d0 <- tab.d0[x$ind.d0==1, -5] 
            }
            colnames(tab.d0) <-  c("Rho", "Med. Eff.", "95% CI Lower", "95% CI Upper", "R^2_M*R^2_Y*", "R^2_M~R^2_Y~")
            rownames(tab.d0) <- NULL
            tab.d1 <- cbind(x$rho, round(x$d1,4), round(x$lower.d1,4), round(x$upper.d1, 4), x$ind.d1, x$R2star.prod, round(x$R2tilde.prod, 4))
            if(sum(x$ind.d1)==1){
                tab.d1 <- as.matrix(tab.d1[x$ind.d1==1, -5])
                tab.d1 <- t(tab.d1)
            } else {
                tab.d1 <- tab.d1[x$ind.d1==1, -5] 
            }
            colnames(tab.d1) <-  c("Rho", "Med. Eff.", "95% CI Lower", "95% CI Upper", "R^2_M*R^2_Y*", "R^2_M~R^2_Y~")
            rownames(tab.d1) <- NULL
            cat("\nMediation Sensitivity Analysis\n")
            cat("\nSensitivity Region: ACME for Control Group\n\n")
            print(tab.d0)
            cat("\nRho at which Delta_0 = 0:", round(x$err.cr[1], 4), "\n\n")
            cat("\nR^2_M*R^2_Y* at which ACME for Control Group = 0:", round(x$R2star.thresh[1], 4), "\n\n")
            cat("\nR^2_M~R^2_Y~ at which ACME for Control Group = 0:", round(x$R2tilde.thresh[1], 4), "\n\n")
            cat("\nSensitivity Region: ACME for Treatment Group\n\n")
            print(tab.d1)
            cat("\nRho at which ACME for Treatment Group = 0:", round(x$err.cr[2], 4), "\n\n")
            cat("\nR^2_M*R^2_Y* at which ACME for Treatment Group = 0:", round(x$R2star.thresh[2], 4), "\n\n")
            cat("\nR^2_M~R^2_Y~ at which ACME for Treatment Group = 0:", round(x$R2tilde.thresh[2], 4), "\n\n")
            invisible(x)        
            }
    } else if(x$type=="bm") {
        if(x$INT==FALSE){
            tab <- cbind(x$rho, round(x$d0,4), round(x$lower.d0,4), round(x$upper.d0, 4), x$ind.d0, x$R2star.prod, round(x$R2tilde.prod, 4))
            if(sum(x$ind.d0)==1){
                tab <- as.matrix(tab[x$ind.d0==1, -5])
                tab <- t(tab)
            } else {
                tab <- tab[x$ind.d0==1, -5] 
            }
            colnames(tab) <-  c("Rho", "Med. Eff.", "95% CI Lower", "95% CI Upper", "R^2_M*R^2_Y*", "R^2_M~R^2_Y~")
            rownames(tab) <- NULL
            cat("\nMediation Sensitivity Analysis\n")
            cat("\nSensitivity Region\n\n")
            print(tab)
            cat("\nRho at which ACME = 0:", round(x$err.cr, 4), "\n\n")
            cat("\nR^2_M*R^2_Y* at which ACME = 0:", round(x$R2star.thresh, 4), "\n\n")
            cat("\nR^2_M~R^2_Y~ at which ACME = 0:", round(x$R2tilde.thresh, 4), "\n\n")
            invisible(x)    
        } else {
            tab.d0 <- cbind(x$rho, round(x$d0,4), round(x$lower.d0,4), round(x$upper.d0, 4), x$ind.d0, x$R2star.prod, round(x$R2tilde.prod, 4))
            if(sum(x$ind.d0)==1){
                tab.d0 <- as.matrix(tab.d0[x$ind.d0==1, -5])
                tab.d0 <- t(tab.d0)
            } else {
                tab.d0 <- tab.d0[x$ind.d0==1, -5] 
            }
            colnames(tab.d0) <-  c("Rho","Med. Eff.", "95% CI Lower", "95% CI Upper", "R^2_M*R^2_Y*", "R^2_M~R^2_Y~")
            rownames(tab.d0) <- NULL
            tab.d1 <- cbind(x$rho, round(x$d1,4), round(x$lower.d1,4), round(x$upper.d1, 4), x$ind.d1, x$R2star.prod, round(x$R2tilde.prod, 4))
            if(sum(x$ind.d1)==1){
                tab.d1 <- as.matrix(tab.d1[x$ind.d1==1, -5])
                tab.d1 <- t(tab.d1)
            } else {
                tab.d1 <- tab.d1[x$ind.d1==1, -5] 
            }
            colnames(tab.d1) <-  c("Rho","Med. Eff.", "95% CI Lower", "95% CI Upper", "R^2_M*R^2_Y*", "R^2_M~R^2_Y~")
            rownames(tab.d1) <- NULL
            cat("\nMediation Sensitivity Analysis\n")
            cat("\nSensitivity Region: ACME for Control Group\n\n")
            print(tab.d0)
            cat("\nRho at which ACME for Control Group = 0:", round(x$err.cr[1], 4), "\n\n")
            cat("\nR^2_M*R^2_Y* at which Delta_0 = 0:", round(x$R2star.thresh[1], 4), "\n\n")
            cat("\nR^2_M~R^2_Y~ at which Delta_0 = 0:", round(x$R2tilde.thresh[1], 4), "\n\n")
            cat("\nSensitivity Region: ACME for Treatment Group\n\n")
            print(tab.d1)
            cat("\nRho at which Delta_1 = 0:", round(x$err.cr[2], 4), "\n\n")
            cat("\nR^2_M*R^2_Y* at which ACME for Treatment Group = 0:", round(x$R2star.thresh[2], 4), "\n\n")
            cat("\nR^2_M~R^2_Y~ at which ACME for Treatment Group = 0:", round(x$R2tilde.thresh[2], 4), "\n\n")
            invisible(x)        
            }
    } else if(x$type=="bo") {
        if(x$INT==FALSE){
            tab <- cbind(x$rho, round(x$d0,4), round(x$lower.d0,4), round(x$upper.d0, 4), x$ind.d0, x$R2star.prod, round(x$R2tilde.prod, 4))
            if(sum(x$ind.d0)==1){
                tab <- as.matrix(tab[x$ind.d0==1, -5])
                tab <- t(tab)
            } else {
                tab <- tab[x$ind.d0==1, -5] 
            }
            colnames(tab) <-  c("Rho", "Med. Eff.", "95% CI Lower", "95% CI Upper", "R^2_M*R^2_Y*", "R^2_M~R^2_Y~")
            rownames(tab) <- NULL
            cat("\nMediation Sensitivity Analysis\n")
            cat("\nSensitivity Region\n\n")
            print(tab)
            cat("\nRho at which Delta = 0:", round(x$err.cr, 4), "\n\n")
            cat("\nR^2_M*R^2_Y* at which ACME = 0:", round(x$R2star.thresh, 4), "\n\n")
            cat("\nR^2_M~R^2_Y~ at which ACME = 0:", round(x$R2tilde.thresh, 4), "\n\n")
            invisible(x)    
        } else {
            tab.d0 <- cbind(x$rho, round(x$d0,4), round(x$lower.d0,4), round(x$upper.d0, 4), x$ind.d0, x$R2star.prod, round(x$R2tilde.prod, 4))
            if(sum(x$ind.d0)==1){
                tab <- as.matrix(tab[x$ind.d0==1, -5])
                tab <- t(tab)
            } else {
                tab <- tab[x$ind.d0==1, -5] 
            }
            colnames(tab.d0) <-  c("Rho","Med. Eff.", "95% CI Lower", "95% CI Upper", "R^2_M*R^2_Y*", "R^2_M~R^2_Y~")
            rownames(tab.d0) <- NULL
            tab.d1 <- cbind(x$rho, round(x$d1,4), round(x$lower.d1,4), round(x$upper.d1, 4), x$ind.d1, x$R2star.prod, round(x$R2tilde.prod, 4))
            if(sum(x$ind.d1)==1){
                tab <- as.matrix(tab[x$ind.d1==1, -5])
                tab <- t(tab)
            } else {
                tab <- tab[x$ind.d1==1, -5] 
            }
            colnames(tab.d1) <-  c("Rho","Med. Eff.", "95% CI Lower", "95% CI Upper", "R^2_M*R^2_Y*", "R^2_M~R^2_Y~")
            rownames(tab.d1) <- NULL
            cat("\nMediation Sensitivity Analysis\n")
            cat("\nSensitivity Region: ACME for Control Group\n\n")
            print(tab.d0)
            cat("\nRho at which ACME for Control Group = 0:", round(x$err.cr[1], 4), "\n\n")
            cat("\nR^2_M*R^2_Y* at which ACME for Control Group = 0:", round(x$R2star.thresh[1], 4), "\n\n")
            cat("\nR^2_M~R^2_Y~ at which ACME for Control Group = 0:", round(x$R2tilde.thresh[1], 4), "\n\n")
            cat("\nSensitivity Region: ACME for Treatment Group\n\n")
            print(tab.d1)
            cat("\nRho at which ACME for Treatment Group = 0:", round(x$err.cr[2], 4), "\n\n")
            cat("\nR^2_M*R^2_Y* at which ACME for Treatment Group  = 0:", round(x$R2star.thresh[2], 4), "\n\n")
            cat("\nR^2_M~R^2_Y~ at which ACME for Treatment Group  = 0:", round(x$R2tilde.thresh[2], 4), "\n\n")
            invisible(x)        
            }
     }
}

plot.medsens <- function(x, sens.par="rho", r.type=1, sign.prod=1, pr.plot=FALSE, smooth.effect=FALSE, smooth.ci=FALSE, levels=NULL, xlab=NULL, ylab=NULL, xlim=NULL, ylim=NULL, main=NULL, ...){
  if(sens.par=="rho"){
    if(pr.plot==TRUE){
        if(x$type!="bm"){
            stop("Proportion mediated is only implemented for binary mediator \n")
        }
        if(is.null(ylim)) 
            ylim <- c(min(x$nu), max(x$nu))
        if(is.null(main))
            main <- expression(paste("Proportion Mediated ", (rho)))
        if(smooth.effect==TRUE)
            nu <- lowess(x$rho, x$nu)$y
            else nu <- x$nu
        if(smooth.ci==TRUE){
            lower <- lowess(x$rho, x$lower.nu)$y
            upper <- lowess(x$rho, x$upper.nu)$y
        } else {
            lower <- x$lower.nu
            upper <- x$upper.nu
        }
        plot.default(x$rho, nu, type="n", xlab="", ylab="", main=main, xlim=xlim, ylim=ylim, ...)
        polygon(x=c(x$rho, rev(x$rho)), y=c(lower, rev(upper)), border=FALSE, col=8, lty=2)
        lines(x$rho, nu, lty=1)
        abline(h=0)
        abline(v=0)
        if(is.null(xlab)) 
            title(xlab = expression(paste("Sensitivity Parameter: ", rho)), line=2.5, cex.lab=.9)
            else title(xlab = xlab)
        if(is.null(ylab)) 
            title(ylab = expression(paste("Proportion Mediated: ", bar(nu))), line=2.5, cex.lab=.9)
            else title(ylab = ylab)
    } else {
    if(x$INT==FALSE){
        if(is.null(ylim))
            ylim <- c(min(x$d0), max(x$d0))
        if(is.null(main))
            main <- expression(paste("ACME(", rho, ")"))
        if(smooth.effect==TRUE)
            d0 <- lowess(x$rho, x$d0)$y
            else d0 <- x$d0
        if(smooth.ci==TRUE){
            lower <- lowess(x$rho, x$lower.d0)$y
            upper <- lowess(x$rho, x$upper.d0)$y
        } else {
            lower <- x$lower.d0
            upper <- x$upper.d0
        }
        plot.default(x$rho, d0, type="n", xlab="", ylab="", main=main, xlim=xlim, ylim=ylim, ...)
        polygon(x=c(x$rho, rev(x$rho)), y=c(lower, rev(upper)), border=FALSE, col=8, lty=2)
        lines(x$rho, d0, lty=1)
        abline(h=0)
        abline(v=0)
        abline(h=weighted.mean(c(d0[floor(1/x$rho.by)],d0[ceiling(1/x$rho.by)]), 
            c(1-1/x$rho.by+floor(1/x$rho.by), 1/x$rho.by-floor(1/x$rho.by))), lty=2)
        if(is.null(xlab)) 
            title(xlab = expression(paste("Sensitivity Parameter: ", rho)), line=2.5, cex.lab=.9)
            else title(xlab = xlab)
        if(is.null(ylab)) 
            title(ylab = expression(paste("Average Mediation Effect: ", bar(delta)(t))), line=2.5, cex.lab=.9)
            else title(ylab = ylab)
    } else {
        if(prod(par("mfrow")==1) && dev.interactive()){
            oask <- devAskNewPage(TRUE)
            on.exit(devAskNewPage(oask))
        }
        if(is.null(ylim))
            ylim <- c(min(x$d0), max(x$d0))
        if(is.null(main))
            main0 <- expression(paste("ACME"[0], "(", rho, ")"))
            else main0 <- main
        if(smooth.effect==TRUE)
            d0 <- lowess(x$rho, x$d0)$y
            else d0 <- x$d0
        if(smooth.ci==TRUE){
            lower <- lowess(x$rho, x$lower.d0)$y
            upper <- lowess(x$rho, x$upper.d0)$y
        } else {
            lower <- x$lower.d0
            upper <- x$upper.d0
        }
        plot.default(x$rho, d0, type="n", xlab="", ylab="", main=main0, xlim=xlim, ylim=ylim, ...)
        polygon(x=c(x$rho, rev(x$rho)), y=c(lower, rev(upper)), border=FALSE, col=8, lty=2)
        lines(x$rho, d0, lty=1)
        abline(h=0)
        abline(v=0)
        abline(h=weighted.mean(c(d0[floor(1/x$rho.by)],d0[ceiling(1/x$rho.by)]), 
            c(1-1/x$rho.by+floor(1/x$rho.by), 1/x$rho.by-floor(1/x$rho.by))), lty=2)
        if(is.null(xlab)) 
            title(xlab = expression(paste("Sensitivity Parameter: ", rho)), line=2.5, cex.lab=.9)
            else title(xlab = xlab)
        if(is.null(ylab)) 
            title(ylab = expression(paste("Average Mediation Effect: ", bar(delta)(0))), line=2.5, cex.lab=.9)
            else title(ylab = ylab)

        #Delta_1
        if(is.null(ylim))
            ylim <- c(min(x$d1), max(x$d1))
        if(is.null(main))
            main1 <- expression(paste("ACME"[1], "(", rho, ")"))
            else main1 <- main
        if(smooth.effect==TRUE)
            d1 <- lowess(x$rho, x$d1)$y
            else d1 <- x$d1
        if(smooth.ci==TRUE){
            lower <- lowess(x$rho, x$lower.d1)$y
            upper <- lowess(x$rho, x$upper.d1)$y
        } else {
            lower <- x$lower.d1
            upper <- x$upper.d1
        }
        plot.default(x$rho, d1, type="n", xlab="", ylab="", main=main1, xlim=xlim, ylim=ylim,...)
        polygon(x=c(x$rho, rev(x$rho)), y=c(lower, rev(upper)), border=FALSE, col=8, lty=2)
        lines(x$rho, d1, lty=1)
        abline(h=0)
        abline(v=0)
        abline(h=weighted.mean(c(d1[floor(1/x$rho.by)],d1[ceiling(1/x$rho.by)]), 
            c(1-1/x$rho.by+floor(1/x$rho.by), 1/x$rho.by-floor(1/x$rho.by))), lty=2)
        if(is.null(xlab)) 
            title(xlab = expression(paste("Sensitivity Parameter: ", rho)), line=2.5, cex.lab=.9)
            else title(xlab = xlab)
        if(is.null(ylab)) 
            title(ylab = expression(paste("Average Mediation Effect: ", bar(delta)(1))), line=2.5, cex.lab=.9)
            else title(ylab = ylab)
            }
        }
  } else if (sens.par=="R2"){
    if(pr.plot==TRUE)
        stop("Proportion mediated is only plotted in terms of rho\n")
        
    R2Mstar <- seq(0, 1-x$rho.by, 0.01)
    R2Ystar <- seq(0, 1-x$rho.by, 0.01)
    R2Mtilde <- (1-x$r.square.m)*seq(0, 1-x$rho.by, 0.01)
    R2Ytilde <- (1-x$r.square.y)*seq(0, 1-x$rho.by, 0.01)
  
    if(r.type == 1){
        R2M <- R2Mstar; R2Y <- R2Ystar
    } else if(r.type == 2) {
        R2M <- R2Mtilde; R2Y <- R2Ytilde
    } else stop("r.type must be either 1 or 2\n")
    
    dlength <- length(seq(0, (1-x$rho.by)^2, 0.0001))
    R2prod.mat <- outer(R2Mstar, R2Ystar)
    Rprod.mat <- round(sqrt(R2prod.mat), digits=4)
    
    if(is.null(xlim))
        xlim <- c(0,1)
    if(is.null(ylim))
        ylim <- c(0,1)
    
    if(x$INT==FALSE){
        if(is.null(levels))
            levels <- pretty(quantile(x$d0, probs=c(0.1,0.9)), 10)
        if(sign.prod == 1){
            d0.p <- approx(x$d0[((length(x$d0)+1)/2):length(x$d0)], n=dlength)$y
            d0mat.p <- matrix(d0.p[Rprod.mat/0.0001+1], nrow=length(R2M))
            if(is.null(main) & r.type == 1)
                main <- expression(paste("ACME(", R[M]^{2},"*,", R[Y]^2,"*), sgn", (lambda[2]*lambda[3])==1))
            else if(is.null(main) & r.type == 2)
                main <- expression(paste("ACME(", tilde(R)[M]^{2}, "," , tilde(R)[Y]^2, "), sgn", (lambda[2]*lambda[3])==1))
            contour(R2M, R2Y, d0mat.p, levels=levels, main=main, xlab=xlab, ylab=ylab, ylim=ylim, xlim=xlim, ...)
        } else if(sign.prod == -1){
            d0.n <- rev(approx(x$d0[1:((length(x$d0)+1)/2)], n=dlength)$y)
            d0mat.n <- matrix(d0.n[Rprod.mat/0.0001+1], nrow=length(R2M))
            if(is.null(main) & r.type == 1)
                main <- expression(paste("ACME(", R[M]^{2},"*,", R[Y]^2,"*), sgn", (lambda[2]*lambda[3])==-1))
            else if(is.null(main) & r.type == 2)
                main <- expression(paste("ACME(", tilde(R)[M]^{2}, "," , tilde(R)[Y]^2, "), sgn", (lambda[2]*lambda[3])==-1))
            contour(R2M, R2Y, d0mat.n, levels=levels, main=main, xlab=xlab, ylab=ylab, ylim=ylim, xlim=xlim, ...)
        } else stop("'sign.prod' must be either -1 or 1\n")
        if(is.null(xlab) & r.type==1)
            title(xlab=expression(paste(R[M]^{2},"*")), line=2.5, cex.lab=.9)
            else if(is.null(xlab) & r.type==2)
            title(xlab=expression(paste(tilde(R)[M]^{2})), line=2.5, cex.lab=.9)
        if(is.null(ylab) & r.type==1)
            title(ylab=expression(paste(R[Y]^2,"*")), line=2.5, cex.lab=.9)
            else if(is.null(ylab) & r.type==2)
            title(ylab=expression(paste(tilde(R)[Y]^2)), line=2.5, cex.lab=.9)
        axis(2,at=seq(0,1,by=.1))
        axis(1,at=seq(0,1,by=.1))
    } else {
        if(prod(par("mfrow")==1) && dev.interactive()){
            oask <- devAskNewPage(TRUE)
            on.exit(devAskNewPage(oask))
        }
        if(is.null(levels))
            levels0 <- pretty(quantile(x$d0, probs=c(0.1,0.9)), 10)
            else levels0 <- levels
        if(sign.prod == 1){
            d0.p <- approx(x$d0[((length(x$d0)+1)/2):length(x$d0)], n=dlength)$y
            d0mat.p <- matrix(d0.p[Rprod.mat/0.0001+1], nrow=length(R2M))
            if(is.null(main) & r.type == 1)
                main0 <- expression(paste("ACME"[0], "(", R[M]^{2},"*,", R[Y]^2,"*), sgn", (lambda[2]*lambda[3])==1))
            else if(is.null(main) & r.type == 2)
                main0 <- expression(paste("ACME"[0], "(", tilde(R)[M]^{2}, "," , tilde(R)[Y]^2, "), sgn", (lambda[2]*lambda[3])==1))
            else main0 <- main
            contour(R2M, R2Y, d0mat.p, levels=levels0, main=main0, xlab=xlab, ylab=ylab, ylim=ylim, xlim=xlim)
        } else if(sign.prod == -1){
            d0.n <- rev(approx(x$d0[1:((length(x$d0)+1)/2)], n=dlength)$y)
            d0mat.n <- matrix(d0.n[Rprod.mat/0.0001+1], nrow=length(R2M))
            if(is.null(main) & r.type == 1)
                main0 <- expression(paste("ACME"[0],"(", R[M]^{2},"*,", R[Y]^2,"*), sgn", (lambda[2]*lambda[3])==-1))
            else if(is.null(main) & r.type == 2)
                main0 <- expression(paste("ACME"[0], "(", tilde(R)[M]^{2}, "," , tilde(R)[Y]^2, "), sgn", (lambda[2]*lambda[3])==-1))
            else main0 <- main
            contour(R2M, R2Y, d0mat.n, levels=levels0, main=main0, xlab=xlab, ylab=ylab, ylim=ylim, xlim=xlim)
        } else stop("sign.prod must be either -1 or 1\n")
        if(is.null(xlab) & r.type==1)
            title(xlab=expression(paste(R[M]^{2},"*")), line=2.5, cex.lab=.9)
            else if(is.null(xlab) & r.type==2)
            title(xlab=expression(paste(tilde(R)[M]^{2})), line=2.5, cex.lab=.9)
        if(is.null(ylab) & r.type==1)
            title(ylab=expression(paste(R[Y]^2,"*")), line=2.5, cex.lab=.9)
            else if(is.null(ylab) & r.type==2)
            title(ylab=expression(paste(tilde(R)[Y]^2)), line=2.5, cex.lab=.9)
        axis(2,at=seq(0,1,by=.1))
        axis(1,at=seq(0,1,by=.1))
    #Delta_1
        if(is.null(levels))
            levels1 <- pretty(quantile(x$d1, probs=c(0.1,0.9)), 10)
            else levels1 <- levels
        if(sign.prod == 1){
            d1.p <- approx(x$d1[((length(x$d1)+1)/2):length(x$d1)], n=dlength)$y
            d1mat.p <- matrix(d1.p[Rprod.mat/0.0001+1], nrow=length(R2M))
            if(is.null(main) & r.type == 1)
                main1 <- expression(paste("ACME"[1], "(", R[M]^{2},"*,", R[Y]^2,"*), sgn", (lambda[2]*lambda[3])==1))
            else if(is.null(main) & r.type == 2)
                main1 <- expression(paste("ACME"[1], "(", tilde(R)[M]^{2}, "," , tilde(R)[Y]^2, "), sgn", (lambda[2]*lambda[3])==1))
            else main1 <- main
            contour(R2M, R2Y, d1mat.p, levels=levels1, main=main1, xlab=xlab, ylab=ylab, ylim=ylim, xlim=xlim)
        } else if(sign.prod == -1){
            d1.n <- rev(approx(x$d1[1:((length(x$d1)+1)/2)], n=dlength)$y)
            d1mat.n <- matrix(d1.n[Rprod.mat/0.0001+1], nrow=length(R2M))
            if(is.null(main) & r.type == 1)
                main1 <- expression(paste("ACME"[1], "(", R[M]^{2},"*,", R[Y]^2,"*), sgn", (lambda[2]*lambda[3])==-1))
            else if(is.null(main) & r.type == 2)
                main1 <- expression(paste("ACME"[1], "(", tilde(R)[M]^{2}, "," , tilde(R)[Y]^2, "), sgn", (lambda[2]*lambda[3])==-1))
            else main1 <- main
            contour(R2M, R2Y, d1mat.n, levels=levels1, main=main1, xlab=xlab, ylab=ylab, ylim=ylim, xlim=xlim)
        }
        if(is.null(xlab) & r.type==1)
            title(xlab=expression(paste(R[M]^{2},"*")), line=2.5, cex.lab=.9)
            else if(is.null(xlab) & r.type==2)
            title(xlab=expression(paste(tilde(R)[M]^{2})), line=2.5, cex.lab=.9)
        if(is.null(ylab) & r.type==1)
            title(ylab=expression(paste(R[Y]^2,"*")), line=2.5, cex.lab=.9)
            else if(is.null(ylab) & r.type==2)
            title(ylab=expression(paste(tilde(R)[Y]^2)), line=2.5, cex.lab=.9)
        axis(2,at=seq(0,1,by=.1))
        axis(1,at=seq(0,1,by=.1))
      }
  } else stop("sens.par must be either 'rho' or 'R2'")
}

