\name{smooth.construct.fs.smooth.spec}
\alias{smooth.construct.fs.smooth.spec}
\alias{Predict.matrix.fs.interaction}
\alias{factor.smooth.interaction}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Factor smooth interactions in GAMs}

\description{Simple factor smooth interactions, which are efficient when used with \code{\link{gamm}}.
This smooth class allows a separate smooth for each level of a factor, with the same smoothing parameter for all 
smooths. It is an alternative to using factor \code{by} variables.

See the discussion of \code{by} variables in \code{\link{gam.models}} for more general alternatives 
for factor smooth interactions (including interactions of tensor product smooths with factors).
}

\usage{
\method{smooth.construct}{fs.smooth.spec}(object, data, knots)
\method{Predict.matrix}{fs.interaction}(object, data)
}

\arguments{ 
\item{object}{For the \code{smooth.construct} method a smooth specification object, 
usually generated by a term \code{s(x,...,bs="fs",)}. For the \code{predict.Matrix} method 
an object of class \code{"fs.interaction"} produced by the \code{smooth.construct} method.}

\item{data}{a list containing just the data (including any \code{by} variable) required by this term, 
            with names corresponding to \code{object$term}.} 

\item{knots}{ a list containing any knots supplied for smooth basis setup.}

}

\value{ An object of class \code{"fs.interaction"} or a matrix mapping the coefficients of the factor smooth interaction to the smooths themselves.
}

\details{This class produces a smooth for each level of a single factor variable. Within a \code{\link{gam}} 
formula this is done with something like \code{s(x,fac,bs="fs")}, which is almost equivalent to \code{s(x,by=fac,id=1)} 
(with the \code{gam} argument \code{select=TRUE}). The terms are fully penalized, with separate penalties on each null 
space component: for this reason they are not centred (no sum-to-zero constraint).

The class is particularly useful for use with \code{\link{gamm}}, where estimation efficiently exploits 
the nesting of the smooth within the factor. Note however that: i) \code{gamm} only allows one conditioning 
factor for smooths, so \code{s(x)+s(z,fac,bs="fs")+s(v,fac,bs="fs")} is OK, but \code{s(x)+s(z,fac1,bs="fs")+s(v,fac2,bs="fs")}
is not; ii) all aditional random effects and correlation structures will be treated as nested within the factor
of the smooth factor interaction.

Note that \code{gamm4} from the {\code{gamm4}} package suffers from none of the restrictions that apply to \code{gamm}, and
\code{"fs"} terms can be used without side-effects. 

Any singly penalized basis can be used to smooth at each factor level. The default is \code{"tp"}, but alternatives can 
be supplied in the \code{xt} argument of \code{s} (e.g. \code{s(x,fac,bs="fs",xt="cr")} or 
\code{s(x,fac,bs="fs",xt=list(bs="cr")}). The \code{k} argument to \code{s(...,bs="fs")} refers to the basis dimension to 
use for each level of the factor variable.

Note one computational bottleneck: currently \code{\link{gamm}} (or \code{gamm4}) will produce the full posterior covariance matrix for the 
smooths, including the smooths at each level of the factor. This matrix can get large and computationally costly if there 
are more than a few hundred levels of the factor. Even at one or two hundred levels, care should be taken to keep 
down \code{k}. 

The plot method for this class has two schemes. \code{scheme==0} is in colour, while \code{scheme==1} is black and white.
}


\author{ Simon N. Wood \email{simon.wood@r-project.org}}

\seealso{\code{\link{gam.models}}, \code{\link{gamm}}}

\examples{
library(mgcv)
set.seed(0)
## simulate data...
f0 <- function(x) 2 * sin(pi * x)
f1 <- function(x,a=2,b=-1) exp(a * x)+b
f2 <- function(x) 0.2 * x^11 * (10 * (1 - x))^6 + 10 * 
            (10 * x)^3 * (1 - x)^10
n <- 500;nf <- 25
fac <- sample(1:nf,n,replace=TRUE)
x0 <- runif(n);x1 <- runif(n);x2 <- runif(n)
a <- rnorm(nf)*.2 + 2;b <- rnorm(nf)*.5
f <- f0(x0) + f1(x1,a[fac],b[fac]) + f2(x2)
fac <- factor(fac)
y <- f + rnorm(n)*2
## so response depends on global smooths of x0 and 
## x2, and a smooth of x1 for each level of fac.

## fit model...
bm <- gamm(y~s(x0)+ s(x1,fac,bs="fs",k=5)+s(x2,k=20))
plot(bm$gam,pages=1)

## Could also use...
## b <- gam(y~s(x0)+ s(x1,fac,bs="fs",k=5)+s(x2,k=20),method="ML")
## ... but its slower (increasingly so with increasing nf)
## b <- gam(y~s(x0)+ t2(x1,fac,bs=c("tp","re"),k=5,full=TRUE)+
##        s(x2,k=20),method="ML"))
## ... is exactly equivalent. 
}
\keyword{models} \keyword{regression}%-- one or more ..


