% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mirai.R
\name{mirai}
\alias{mirai}
\title{mirai (Evaluate Async)}
\usage{
mirai(.expr, ..., .args = list(), .timeout = NULL, .compute = NULL)
}
\arguments{
\item{.expr}{an expression to evaluate asynchronously (of arbitrary length,
wrapped in \{ \} where necessary), \strong{or else} a pre-constructed language
object.}

\item{...}{(optional) \strong{either} named arguments (name = value pairs)
specifying objects referenced, but not defined, in \code{.expr}, \strong{or} an
environment containing such objects. See 'evaluation' section below.}

\item{.args}{(optional) \strong{either} a named list specifying objects
referenced, but not defined, in \code{.expr}, \strong{or} an environment containing
such objects. These objects will remain local to the evaluation environment
as opposed to those supplied in \code{...} above - see 'evaluation' section
below.}

\item{.timeout}{[default NULL] for no timeout, or an integer value in
milliseconds. A mirai will resolve to an 'errorValue' 5 (timed out) if
evaluation exceeds this limit.}

\item{.compute}{[default NULL] character value for the compute profile
to use (each has its own independent set of daemons), or NULL to use the
'default' profile.}
}
\value{
A 'mirai' object.
}
\description{
Evaluate an expression asynchronously in a new background R process or
persistent daemon (local or remote). This function will return immediately
with a 'mirai', which will resolve to the evaluated result once complete.
}
\details{
This function will return a 'mirai' object immediately.

The value of a mirai may be accessed at any time at \verb{$data}, and if yet
to resolve, an 'unresolved' logical NA will be returned instead.

\code{\link[=unresolved]{unresolved()}} may be used on a mirai, returning TRUE if a 'mirai' has yet to
resolve and FALSE otherwise. This is suitable for use in control flow
statements such as \code{while} or \code{if}.

Alternatively, to call (and wait for) the result, use \code{\link[=call_mirai]{call_mirai()}} on the
returned 'mirai'. This will block until the result is returned.

Specify \code{.compute} to send the mirai using a specific compute profile (if
previously created by \code{\link[=daemons]{daemons()}}), otherwise leave as \code{"default"}.
}
\section{Evaluation}{


The expression \code{.expr} will be evaluated in a separate R process in a clean
environment (not the global environment), consisting only of the objects
supplied to \code{.args}, with the objects passed as \code{...} assigned to the global
environment of that process.

As evaluation occurs in a clean environment, all undefined objects must be
supplied through \code{...} and/or \code{.args}, including self-defined functions.
Functions from a package should use namespaced calls such as
\code{mirai::mirai()}, or else the package should be loaded beforehand as part of
\code{.expr}.

For evaluation to occur \emph{as if} in your global environment, supply objects to
\code{...} rather than \code{.args}, e.g. for non-local variables or helper functions
required by other functions, as scoping rules may otherwise prevent them from
being found.
}

\section{Timeouts}{


Specifying the \code{.timeout} argument ensures that the mirai always resolves.
When using dispatcher, the mirai will be cancelled after it times out (as if
\code{\link[=stop_mirai]{stop_mirai()}} had been called). As in that case, there is no guarantee that
any cancellation will be successful, if the code cannot be interrupted for
instance. When not using dispatcher, the mirai task will continue to
completion in the daemon process, even if it times out in the host process.
}

\section{Errors}{


If an error occurs in evaluation, the error message is returned as a
character string of class 'miraiError' and 'errorValue'. \code{\link[=is_mirai_error]{is_mirai_error()}}
may be used to test for this. The elements of the original condition are
accessible via \code{$} on the error object. A stack trace comprising a list of
calls is also available at \verb{$stack.trace}.

If a daemon crashes or terminates unexpectedly during evaluation, an
'errorValue' 19 (Connection reset) is returned.

\code{\link[=is_error_value]{is_error_value()}} tests for all error conditions including 'mirai' errors,
interrupts, and timeouts.
}

\examples{
\dontshow{if (interactive()) withAutoprint(\{ # examplesIf}
# specifying objects via '...'
n <- 3
m <- mirai(x + y + 2, x = 2, y = n)
m
m$data
Sys.sleep(0.2)
m$data

# passing the calling environment to '...'
df1 <- data.frame(a = 1, b = 2)
df2 <- data.frame(a = 3, b = 1)
df_matrix <- function(x, y) {
  mirai(as.matrix(rbind(x, y)), environment(), .timeout = 1000)
}
m <- df_matrix(df1, df2)
m[]

# using unresolved()
m <- mirai(
  {
    res <- rnorm(n)
    res / rev(res)
  },
  n = 1e6
)
while (unresolved(m)) {
  cat("unresolved\n")
  Sys.sleep(0.1)
}
str(m$data)

# evaluating scripts using source() in '.expr'
n <- 10L
file <- tempfile()
cat("r <- rnorm(n)", file = file)
m <- mirai({source(file); r}, file = file, n = n)
call_mirai(m)$data
unlink(file)

# use source(local = TRUE) when passing in local variables via '.args'
n <- 10L
file <- tempfile()
cat("r <- rnorm(n)", file = file)
m <- mirai({source(file, local = TRUE); r}, .args = list(file = file, n = n))
call_mirai(m)$data
unlink(file)

# passing a language object to '.expr' and a named list to '.args'
expr <- quote(a + b + 2)
args <- list(a = 2, b = 3)
m <- mirai(.expr = expr, .args = args)
collect_mirai(m)
\dontshow{\}) # examplesIf}
}
