% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SIBTEST.R
\name{SIBTEST}
\alias{SIBTEST}
\title{Simultaneous Item Bias Test (SIBTEST)}
\usage{
SIBTEST(dat, group, suspect_set, match_set, focal_name, guess_correction = 0,
  Jmin = 2, pk_focal = FALSE, correction = TRUE, details = FALSE)
}
\arguments{
\item{dat}{integer dataset to be tested containing dichotomous or polytomous responses}

\item{group}{a vector indicating group membership}

\item{suspect_set}{an integer vector indicating which items to inspect with SIBTEST. Including only
one value will perform a DIF test, while including more than one will perform a simultaneous
bundle test (DBF); including all non-matched items will perform DTF.
If missing, a simultaneous test using all the items not listed in match_set
will be used (i.e., DTF)}

\item{match_set}{an integer vector indicating which items to use as the items which are matched
(i.e., contain no DIF). These are analogous to 'achor' items in the likelihood method to locate
DIF. If missing, all items other than the items found in the suspect_set will be used}

\item{focal_name}{name of the focal group; e.g., 'focal'. If not specified then one will be
selected automatically}

\item{guess_correction}{a vector of numbers from 0 to 1 indicating how much to correct the items
for guessing. It's length should be the same as ncol(dat)}

\item{Jmin}{the minimum number of observations required when splitting the data into focal and
reference groups conditioned on the matched set}

\item{pk_focal}{logical; using the group weights from the focal group instead of the total
sample? Default is FALSE as per Shealy and Stout's recommendation}

\item{correction}{logical; apply the composite correction for the difference between focal
composite scores using the true-score regression technique? Default is \code{TRUE},
reflecting Shealy and Stout's method}

\item{details}{logical; return a data.frame containing the details required to compute SIBTEST?}
}
\description{
Classical test theory approach to detecting unidirectional and bidirectional (with one
crossing location) DIF. This family of statistics is intended for unidimensional tests,
and applies a regression-corrected matched-total score approach to quantify the response
bias between two groups. Can be used for DIF, DBF, and DTF testing.
}
\details{
SIBTEST is similar to the Mantel-Haenszel approach for detecting DIF but uses a regression
correction based on the KR-20/coefficient alpha reliability index to correct the observed
differences when the latent trait distributions are not equal.
Function supports the standard SIBTEST for dichotomous and polytomous data (compensatory) and
supports crossing DIF testing (i.e., non-compensatory/non-uniform) using the asymtotic sampling
distribution version of the Crossing-SIBTEST (CSIBTEST) statistic described by
Chalmers (accepted).
}
\examples{
\dontrun{

set.seed(1234)
n <- 30
N <- 500
a <- matrix(1, n)
d <- matrix(rnorm(n), n)
group <- c(rep('reference', N), rep('focal', N*2))

## -------------
# groups completely equal
dat1 <- simdata(a, d, N, itemtype = 'dich')
dat2 <- simdata(a, d, N*2, itemtype = 'dich')
dat <- rbind(dat1, dat2)

#DIF (all other items as anchors)
SIBTEST(dat, group, suspect_set = 6)

#DIF (specific anchors)
SIBTEST(dat, group, match_set = 1:5, suspect_set = 6)

# DBF (all and specific anchors, respectively)
SIBTEST(dat, group, suspect_set = 11:30)
SIBTEST(dat, group, match_set = 1:5, suspect_set = 11:30)

#DTF
SIBTEST(dat, group, suspect_set = 11:30)
SIBTEST(dat, group, match_set = 1:10) #equivalent

# different hyper pars
dat1 <- simdata(a, d, N, itemtype = 'dich')
dat2 <- simdata(a, d, N*2, itemtype = 'dich', mu = .5, sigma = matrix(1.5))
dat <- rbind(dat1, dat2)
SIBTEST(dat, group, 6:30)
SIBTEST(dat, group, 11:30)

#DIF testing with anchors 1 through 5
SIBTEST(dat, group, 6, match_set = 1:5)
SIBTEST(dat, group, 7, match_set = 1:5)
SIBTEST(dat, group, 8, match_set = 1:5)

#DIF testing with all other items as anchors
SIBTEST(dat, group, 6)
SIBTEST(dat, group, 7)
SIBTEST(dat, group, 8)

## -------------
## systematic differing slopes and intercepts (clear DTF)
dat1 <- simdata(a, d, N, itemtype = 'dich')
dat2 <- simdata(a + c(numeric(15), rnorm(n-15, 1, .25)), d + c(numeric(15), rnorm(n-15, 1, 1)),
  N*2, itemtype = 'dich')
dat <- rbind(dat1, dat2)
SIBTEST(dat, group, 6:30)
SIBTEST(dat, group, 11:30)

#DIF testing using valid anchors
SIBTEST(dat, group, suspect_set = 6, match_set = 1:5)
SIBTEST(dat, group, suspect_set = 7, match_set = 1:5)
SIBTEST(dat, group, suspect_set = 30, match_set = 1:5)

}
}
\references{
Chalmers, R. P. (accepted). Improving the Crossing-SIBTEST statistic for
detecting non-uniform DIF. \emph{Psychometrika}.

Chalmers, R., P. (2012). mirt: A Multidimensional Item Response Theory
Package for the R Environment. \emph{Journal of Statistical Software, 48}(6), 1-29.
\doi{10.18637/jss.v048.i06}

Chang, H. H., Mazzeo, J. & Roussos, L. (1996). DIF for Polytomously Scored Items: An Adaptation of the
  SIBTEST Procedure. \emph{Journal of Educational Measurement, 33}, 333-353.

Li, H.-H. & Stout, W. (1996). A new procedure for detetion of crossing DIF.
\emph{Psychometrika, 61}, 647-677.

Shealy, R. & Stout, W. (1993). A model-based standardization approach that separates true
  bias/DIF from group ability differences and ddetect test bias/DTF as well as item bias/DIF.
  \emph{Psychometrika, 58}, 159-194.
}
\author{
Phil Chalmers \email{rphilip.chalmers@gmail.com}
}
\keyword{SIBTEST}
\keyword{SIBTEST,}
\keyword{crossing}
