% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ensemble_model_spec.R
\name{ensemble_model_spec}
\alias{ensemble_model_spec}
\title{Creates a Stacked Ensemble Model from a Model Spec}
\usage{
ensemble_model_spec(
  object,
  model_spec,
  kfolds = 5,
  param_info = NULL,
  grid = 6,
  control = control_grid()
)
}
\arguments{
\item{object}{A Modeltime Table. Used for ensemble sub-models.}

\item{model_spec}{A \code{model_spec} object defining the
meta-learner stacking model specification to be used.

Can be either:
\enumerate{
\item \strong{A non-tunable \code{model_spec}:} Parameters are specified and are not optimized via tuning.
\item \strong{A tunable \code{model_spec}:} Contains parameters identified for tuning with
\code{tune::tune()}
}}

\item{kfolds}{K-Fold Cross Validation for tuning the Meta-Learner.
Controls the number of folds used in the meta-learner's cross-validation.
Gets passed to \code{rsample::vfold_cv()}.}

\item{param_info}{A \code{dials::parameters()} object or \code{NULL}. If none is given, a
parameters set is derived from other arguments. Passing this argument
can be useful when parameter ranges need to be customized.}

\item{grid}{Grid specification or grid size for tuning the Meta Learner.
Gets passed to \code{tune::tune_grid()}.}

\item{control}{An object used to modify the tuning process.
Uses \code{tune::control_grid()} by default.
Use \code{control_grid(verbose = TRUE)} to follow the training process.}
}
\value{
A \code{mdl_time_ensemble} object.
}
\description{
A 2-stage stacking regressor that follows:
\enumerate{
\item Stage 1: Sub-Model's are Trained & Predicted using \code{\link[modeltime.resample:modeltime_fit_resamples]{modeltime.resample::modeltime_fit_resamples()}}.
\item Stage 2: A Meta-learner (\code{model_spec}) is trained on Out-of-Sample Sub-Model
Predictions using \code{ensemble_model_spec()}.
}
}
\details{
\strong{Stacked Ensemble Process}
\itemize{
\item Start with a \emph{Modeltime Table} to define your sub-models.
\item Step 1: Use \code{\link[=modeltime_fit_resamples]{modeltime_fit_resamples()}} to perform the submodel resampling procedure.
\item Step 2: Use \code{\link[=ensemble_model_spec]{ensemble_model_spec()}} to define and train the meta-learner.
}

\strong{What goes on inside the Meta Learner?}

The Meta-Learner Ensembling Process uses the following basic steps:
\enumerate{
\item \strong{Make Cross-Validation Predictions.}
Cross validation predictions are made for each sub-model with \code{\link[=modeltime_fit_resamples]{modeltime_fit_resamples()}}.
The out-of-sample sub-model predictions contained in \code{.resample_results}
are used as the input to the meta-learner.
\item \strong{Train a Stacked Regressor (Meta-Learner).}
The sub-model out-of-sample cross validation predictions are then
modeled using a \code{model_spec} with options:
\itemize{
\item \strong{Tuning:} If the \code{model_spec} does include tuning parameters via \code{tune::tune()}
then the meta-learner will be hypeparameter tuned using K-Fold Cross Validation. The
parameters and grid can adjusted using \code{kfolds}, \code{grid}, and \code{param_info}.
\item \strong{No-Tuning:} If the \code{model_spec} does \emph{not} include tuning parameters via \code{tune::tune()}
then the meta-learner will not be hypeparameter tuned and will have the model
fitted to the sub-model predictions.
}
\item \strong{Final Model Selection.}
\itemize{
\item \strong{If tuned}, the final model is selected based on RMSE, then
retrained on the full set of out of sample predictions.
\item \strong{If not-tuned}, the fitted model from Stage 2 is used.
}
}

\strong{Progress}

The best way to follow the training process and watch progress is to use
\code{control = control_grid(verbose = TRUE)} to see progress.

\strong{Parallelize}

Portions of the process can be parallelized. To parallelize, set
up parallelization using \code{tune} via one of the backends such as
\code{doFuture}. Then set \code{control = control_grid(allow_par = TRUE)}
}
\examples{
library(tidymodels)
library(modeltime)
library(modeltime.ensemble)
library(tidyverse)
library(timetk)

\donttest{
# Step 1: Make resample predictions for submodels
resamples_tscv <- training(m750_splits) \%>\%
    time_series_cv(
        assess  = "2 years",
        initial = "5 years",
        skip    = "2 years",
        slice_limit = 1
    )

submodel_predictions <- m750_models \%>\%
    modeltime_fit_resamples(
        resamples = resamples_tscv,
        control   = control_resamples(verbose = TRUE)
    )

# Step 2: Metalearner ----

# * No Metalearner Tuning
ensemble_fit_lm <- submodel_predictions \%>\%
    ensemble_model_spec(
        model_spec = linear_reg() \%>\% set_engine("lm"),
        control    = control_grid(verbose = TRUE)
    )

ensemble_fit_lm

# * With Metalearner Tuning ----
ensemble_fit_glmnet <- submodel_predictions \%>\%
    ensemble_model_spec(
        model_spec = linear_reg(
            penalty = tune(),
            mixture = tune()
        ) \%>\%
            set_engine("glmnet"),
        grid       = 2,
        control    = control_grid(verbose = TRUE)
    )

ensemble_fit_glmnet

}

}
