% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/analytical_functions.R
\name{life.table}
\alias{life.table}
\title{Calculates a life table}
\usage{
life.table(neclist, agecor = TRUE, agecorfac = c(), option_spline = NULL)
}
\arguments{
\item{neclist}{single data.frame or list of data.frames
             with the columns 'x', 'a', 'Dx'.
\itemize{
  \item \bold{x}:  age interval identifier, optional. -
                   Otherwise determined from \bold{a}.
  \item \bold{a}:  years within x.
  \item \bold{Dx}: number of deaths within x.
}}

\item{agecor}{logical, optional. If set TRUE, the average number of years lived within a
given age class of individuals having died in this class can be adjusted via agecorfac.
If set FALSE, it is assumed that they died in the middle of this class. Due to the higher
mortality rates of infants, this assumption is certainly inaccurate for individuals
<= 5 years.

Default setup is: TRUE.}

\item{agecorfac}{numeric vector, optional. Only applies if agecor == TRUE. Given values
replace the standard values from the first age interval onward.

Default setup is 1/3 for every age class <= 5 life years, and 1/2 for the others.}

\item{option_spline}{integer, optional. If > 0, values for adults will be
interpolated by a monotonic cubic spline. Usual options will by '10' or '20' which
will interpolate the values for individuals of an age of 20 or older by 10- or 20-
year cumulated values. To be used carefully, as diagnostic features of the life
table might be smoothed and essentially removed. Only available when the methods
'Standard' or 'Equal5' in prep.life.table have been chosen.}
}
\value{
An object of class mortaar_life_table or mortaar_life_table_list.
Each mortaar_life_table contains the following variables:

\itemize{
  \item \bold{x}:  age interval.
  \item \bold{a}:  years within x.

  \item \bold{Ax}: average number of years lived by an
                   individual that died within a specific
                   age class x :

                   \eqn{A_{x} = a_{x} * agecorfac_{x}}

  \item \bold{Dx}: number of deaths within x.
  \item \bold{dx}: proportion of deaths within x (percent) :

                   \eqn{d_{x} = \frac{D_{x}}{\sum_{i=1}^{n} D_{i}} * 100}

  \item \bold{lx}: survivorship within x (percent) :

                   \eqn{l_{x+1} = l_{x} - d_{x}} with \eqn{l_{0} = 100}

  \item \bold{qx}: probability of death within x (percent) :

                   \eqn{q_{x} = \frac{d_{x}}{l_{x}}* 100}

  \item \bold{Lx}: number of years lived within x by those that died within x and those
                   that reached the next age class :

                   \eqn{L_{x} = (a_{x} * l_{x}) - ((a_{x} - A_{x}) * d_{x})}

  \item \bold{Tx}: sum of years lived within
                   current and remaining x :

                   \eqn{T_{x+1} = T_{x} - L_{x}} with \eqn{T_{0} = \sum_{i=1}^{n}{L_{i}}}

  \item \bold{ex}: average years of life remaining
                   (average life expectancy at mean(x)) :

                   \eqn{e_{x} = \frac{T_{x}}{l_{x}}}

  \item \bold{rel_popx}: percentage of L(x) of the sum of L(x) :

                   \eqn{relpopx_{x} = \frac{L_{x}}{\sum_{i=1}^{n}{L_{i}}} * 100}
}
}
\description{
\code{life.table} calculates
\href{https://en.wikipedia.org/wiki/Life_table}{life table(s)}.
The algorithm is optimised for deceased populations
encountered in archaeological research.
See \emph{Chamberlain 2006}, 27ff., \emph{Herrmann et. al 1990}, 303ff.,
\emph{Kokkotidis/Richter 1991}, \emph{Keyfitz et al. 2005}
for selected literature. \cr
The function takes an individual data.frame or a list of
data.frames and returns an object of class mortaar_life_table
or mortaar_life_table_list, for which specialised summary,
print and plot functions exist.
}
\examples{
# Create a mortaar_life_table from a prepared dataset.
schleswig_1 <- life.table(schleswig_ma[c("a", "Dx")])
print(schleswig_1)
plot(schleswig_1, display = "lx")

# Create a mortaar_life_table_list from two datasets.
odagsen <- life.table(list(
  "corpus mandibulae" = odagsen_cm[c("a", "Dx")],
  "margo orbitalis" = odagsen_mo[c("a", "Dx")]
))
print(odagsen)
plot(odagsen, display = "ex")

# Prepare a real world dataset and create a mortaar_life_table.
library(magrittr)
magdalenenberg \%>\%
 replace(. == "60-x", "60-70") \%>\%
 tidyr::separate(a, c("from", "to")) \%>\%
 dplyr::mutate(from = as.numeric(from), to = as.numeric(to)) \%>\%
 prep.life.table(
  dec = "Dx",
  agebeg = "from",
  ageend = "to",
  method = "Standard",
  agerange = "excluded"
 ) \%>\%
 life.table()


}
\references{
\insertRef{chamberlain_demography_2006}{mortAAR}

\insertRef{herrmann_prahistorische_1990}{mortAAR}

\insertRef{keyfitz_applied_2005}{mortAAR}

\insertRef{kokkotidis_graberfeld-_1991}{mortAAR}
}
