% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/animate_move.R
\name{animate_move}
\alias{animate_move}
\title{Animate movement data}
\usage{
animate_move(m, out_dir, conv_dir = "", paths_mode = "true_data",
  paths_na.hold = TRUE, paths_col = "auto", paths_alpha = 1,
  indi_names = NA, layer = "basemap", layer_dt = "basemap",
  layer_int = FALSE, layer_type = "", layer_stretch = "none",
  layer_col = c("sandybrown", "white", "darkgreen"), layer_nacol = "white",
  map_type = "satellite", api_key = NULL, map_zoom = 0,
  map_crop = FALSE, stats_create = FALSE, static_data = NA,
  static_gg = NA, extent_factor = 1e-04, tail_elements = 10,
  tail_size = 4, img_title = "title", img_sub = "subtitle",
  img_caption = "caption", img_labs = "labs", legend_title = "",
  legend_limits = NA, legend_labels = "auto", map_elements = TRUE,
  time_bar_col = "grey", time_scale = TRUE, time_pos_x = 0.5,
  time_pos_y = 0.05, time_size = 3, scalebar_col = "white",
  scalebar_dist = "auto", north_col = "white", frames_layout = 0,
  frames_nmax = 0, frames_fps = 25, frames_nres = 1, frames_tres = 0,
  frames_width = NA, frames_height = NA, frames_pixres = 80,
  out_name = "moveVis", out_format = "gif", overwrite = FALSE,
  log_level = 1, log_logical = FALSE, ..., conv_cmd = "",
  conv_frames = 100)
}
\arguments{
\item{m}{list or \code{moveStack} class object. Needs to contain one or several \code{move} class objects (one for each individual path to be displayed) containing point coordinates, timestamps, projection and individual ID.}

\item{out_dir}{character. Output directory of the output file.}

\item{conv_dir}{character. Command of or path to required image/video converter tool. Depends on, what is specified for \code{out_format}. If \code{out_format = "gif"}, animate_move() works with the ImageMagick \code{convert} tool. In this case, specify command of or path to the \code{convert} tool. You can use \code{\link{get_libraries}} to find or download/install \code{convert}. If \code{out_format} is a video format (e.g. "mp4", "mov" ...), animate_move() works with either the FFmpeg \code{ffmepg} tool or the libav \code{avconv} tool. In this case, specify command of or path to the \code{ffmpeg} or \code{avconv} tool. See also \code{\link{get_libraries}}. If not specified, animate_move() trys to find libraries automatically.}

\item{paths_mode}{character vector. Mode to be used for dealing with time information when displaying multiple individual paths. If set to "true_data", paths are displayed based on true coverage times, showing only time periods that are covered. Time gaps will be skipped. Each frame is linked to a specific true time. If set to "true_time",  paths are displayed based on true coverage times. Time gaps will be filled with non-movement frames. This mode is only recommended, if the dataset has no time gaps. Each frame is linked to a specific, true time. If set to "simple", all movement paths are displayed individually with no regard to the true coverage times. Time gaps will be skipped. Each frame displays several times at once, since each individual path has its own time. Default is "true_data".}

\item{paths_na.hold}{logical. If TRUE, last path location is being hold on frame for NA path locations. If FALSE, path disappears until next path non-NA location. Default is TRUE.}

\item{paths_col}{character vector. Colours of the individual animation paths. If set to "auto", a predfined colour set will be used. If single colour, all paths will be displayed by the same colour. If more individuals then colours, the colours are repeated.}

\item{paths_alpha}{numeric. Set transparency of pathes. If set to 0, path is invisible. Default is 1.}

\item{indi_names}{character. Optional vector of individual names. Length has to be equal to number of individuals in \code{m}. If NA, individual names are tried to be extracted from \code{m} using \code{\link{idData}}. Default is NA.}

\item{layer}{raster, list or character "basemap". Single raster object or list of raster objects to be used as (dynamically changing) basemap layer. Default is \code{"basemap"} to download a static basemap layer. Use a rasterBrick class object and set layer_type to "\code{RGB}" to compute a RGB basemap.}

\item{layer_dt}{POSIXct vector or list. Single POSIXct date/time stamp or list of POSIXct date/time stamps representing the acquisition dates of the \code{layer} raster objects.}

\item{layer_int}{logical. Whether to interpolate the basemap layer objects over time, if several are provided (\code{TRUE}), or to display them one after another depending on the animation time frame that is displayed (\code{FALSE}). Default is \code{FALSE}.}

\item{layer_type}{charachter. Layer type. Either "\code{RGB}" (if layer is a rasterBrick class onejct), "\code{gradient}" or "\code{discrete}". Default is "\code{RGB}". Ignored, if \code{layer = "basemap"}.}

\item{layer_stretch}{character. Ignored, if \code{layer_type} is not "RGB". Either "none", "lin", "hist", "sqrt" or "log" for no stretch, linear, histogram, square-root or logarithmic stretch. Default is "none".}

\item{layer_col}{character vector.  Two or more colours to be used for displaying the background layer. If \code{layer_type = "gradient"}, a colour ramp between the colous is calcualted. If \code{layer_type = "discrete"}, the colours will be used per value range. Ignored, if \code{layer_type = "RGB"}.}

\item{layer_nacol}{character. Colour to be displayed for NA values. Default is "white".}

\item{map_type}{character. Static basemap type. Either Bing Maps options "satellite" (default), "hybrid" or OpenStreetMaps options "roadmap", "roadmap_dark", "roadmap_bw", "roadmap_watercolor".}

\item{api_key}{character. For \code{basemap="satellite"} and \code{basemap="hybrid"}, the Microsoft Bing Maps service is used. If you use this option often, please get your own (free) API key by registering at the \href{https://msdn.microsoft.com/en-us/library/ff428642.aspx}{Microsoft website}.}

\item{map_zoom}{numeric. Increase or decrease the degree of detail of a static basemap. -1 or smaller integers decrease zoom, +1 or greate integers increase zoom. Default is 0. Ignored, if custom basemap is used.}

\item{map_crop}{logical. Set to \code{TRUE} to rop the static basemap to the extent of \code{m}. Otherwise, use the static basemap with the best resolution for the given extent. Default is \code{FALSE}.}

\item{stats_create}{logical. \code{TRUE} to create statistic plots side by side with the spatial plot. Use the arguments explained for \code{\link{animate_stats}} to adjust the plotting behaviour. Default is \code{FALSE}.}

\item{static_data}{data.frame. Data (e.g. static points) to be displayed within the spatial plot of the output animation. At least, "x", "y" columns for the coordinates and "names" for the naming of the point have to be included. If "static_gg" remains unspecified, "static_data" is plottet as points to the output map, annotated with their namings. Points outside the frame extent are not displayed. See "static_gg" for further options.}

\item{static_gg}{character. One or several \code{ggplot2} functions, concatenated by "+" specifying how "static_data" should be displayed, e.g. using \code{geom_point} and \code{geom_text} for displaying points annotated with text. \code{ggplot2 data} and \code{aes, aes_} arguments etc. need to referr to the columns specified in "static_data". As default, "static_data" is plotted as \code{geom_point} and \code{geom_label}.}

\item{extent_factor}{numeric. Defines the distance between the spatial extents of the movement data set and the basemap as proportion of the axis distance. Default is 0.0001. The higher the value, the larger the basemap extent. Ignored, if \code{layer} = "basemap".}

\item{tail_elements}{numeric. Number of points to be displayed as path tail of the animation paths. Default is 10.}

\item{tail_size}{numeric. Size of the first tail element. Default is 4.}

\item{img_title}{character. Titel to be displayed above the animated plot. If not specified, no title will be displayed.}

\item{img_sub}{character. Subtitel to be displayed underneath the title. If not specified, no subtitle will be displayed.}

\item{img_caption}{character. Caption to be displayed underneath the plot. If not specified, no caption will be displayed.}

\item{img_labs}{character. Axis titles to be displayed at the x and y axis of the plot. If not specified, labs will be computed depending on the projection or will be "x" and "y".}

\item{legend_title}{character. Title to be displayed above the basemap layer legend (if layer_type is not \code{"RGB"}). Ignored, if \code{layer = "basemap"}.}

\item{legend_limits}{numeric vector. Fixed minimum and maximum limit values of the legend (gradient layer type). Default is NA for data-depending minimum and maximum values. Ignored, if \code{layer_type} is "discrete" or "RGB".}

\item{legend_labels}{character vectors. Label for each legend break class. If set to "auto", values are displayed. Default is "auto".}

\item{map_elements}{logical. If \code{FALSE}, map elements (north arrow and scale bar) are hidden. Default is \code{TRUE}.}

\item{time_bar_col}{character. Colour of the time progress bar on the top edge of the map. Default is "grey".}

\item{time_scale}{logical. If \code{FALSE}, time scale is hidden. Default is \code{TRUE}.}

\item{time_pos_x}{numeric between 0 and 1, defines the relative position of the time scale display in the x direction. Default is 0.5 (centered).}

\item{time_pos_y}{numeric between 0 and 1, defines the relative position of the time scale display in the y direction. Default is 0.06 (top).}

\item{time_size}{numeric. Defines the font size of the time scale display. Default is 3.}

\item{scalebar_col}{character. Colour of the scalebar text. Default is "white".}

\item{scalebar_dist}{numeric. Distance represented by the scalebar in kilometers.}

\item{north_col}{character. Colour of the north arrow. Default is "white".}

\item{frames_layout}{matrix. Optional layout. Define, which plots should be placed where using a matrix representing the GIF/video frame. Matrix elements can be the following plot identifiers: "map" for the spatial plot, "st_all", "st_per" for the overall and periodic stats plot or "st_allR", "st_perR", "st_allG", "st_perG", "st_allB", "st_perB" for the overall and periodic stats plots per band, when using \code{layer_type = "RGB"}, and 'st_leg' for a stats legend. Alternatively, integers from 1 to 8 corresponding to the described order can be used. Plots not mentioned using \code{frames_layout} identifiers are not displayed. If set to 0, layout is generated automatically. Default is 0.}

\item{frames_nmax}{numeric. Number of maximum frames. If set, the animation will be stopped, after the specified number of frames is reached. Default is 0 (displaying all frames).}

\item{frames_fps}{numeric. Frames to display per second (FPS). Note that the \code{gif} format only can handle FPS out of 100, e.g. 25. In that case, \code{frames_fps} input is rounded. Default is 25.}

\item{frames_nres}{numeric. Interval of which frames of all frames should be used (nth elements). Default is 1 (every frame is used). If set to 2, only every second frame is used.}

\item{frames_tres}{numeric. Defines temporal output resolution in seconds, 'm' should be interpolated to (linear interpolation). If 0, temporal resolution is detected automatically. Default is 0.}

\item{frames_width}{numeric. Number of pixels of frame width. Default is 600 (with stats plots 1000).}

\item{frames_height}{numeric. Number of pixels of frame height. Defualt is 600.}

\item{frames_pixres}{numeric. Resolution of output file in pixel in ppi. The higher the ppi, the higher frames_height and frames_width should be to avoid large fonts and overlaps. Default is 80.}

\item{out_name}{character. Name of the output file. Default is "moveVis".}

\item{out_format}{character. Output format, e.g. "gif", "avi", "3gp", "mov", "mpeg", "mp4". Use \code{\link{get_formats}} to get all supported file formats on your system. "gif" is recommended for short animations only (recommended max. frame number around 200 frames; GIF frames are unlimited, but compution time will be very long). Use a video format for long animations. Format "gif" requires ImageMagick, all other video formats require FFmpeg ('ffmpeg') or libav ('avconv') to be installed. For that, also see \code{\link{get_libraries}}.}

\item{overwrite}{logical. If TRUE, files with equal file names to \code{out_name} will be overwritten. Default is FALSE.}

\item{log_level}{numeric. Level of console output given by the function. There are three log levels. If set to 3, no messages will be displayed except erros that caused an abortion of the process. If set to 2, warnings and errors will be displayed. If set to 1, a log showing the process activity, wanrnings ans errors will be displayed.}

\item{log_logical}{logical. For large processing schemes. If \code{TRUE}, the function returns \code{TRUE} when finished processing succesfully.}

\item{...}{optional arguments. All arguments taken by \code{\link{animate_stats}} can be handed over to \code{\link{animate_move}} as well to create sidy-by-side spatial and statistic plot animations (see \code{\link{animate_stats}}).}

\item{conv_cmd}{character. Recommended for expert use only. Passes additional command line options to the conversion command, e.g. with a \code{convert} call adding '-limit' for memory ressource handling. For details, see check the documentations of ImageMagick \code{convert}, FFmpeg \code{ffmpeg} and libav \code{avconv}.}

\item{conv_frames}{numeric. Recommended for expert use only. Only used, if \code{out_format = "gif"}. Number of frames to be used for creating GIF segments that will be assembled to a final GIF file. Correct number depends on system performance and total frames number. Default is 100. Ignored, if \code{out_format} is not "gif".}
}
\value{
None or logical (see \code{log_logical}). The output file is written to the ouput directory.
}
\description{
\code{animate_move} animates movement data provided as \code{move} class objects or a list of them. The function creates an animated GIF or video file and saves it into the output directory. \code{animate_move} can be operated in different timing modes (see \code{paths_mode}) and with different background layer types (see \code{layer}, \code{layer_type} and \code{map_type}).
}
\details{
Make sure you have run \code{\link{get_libraries}} before you use moveVis for the first time: Depending on the selected output format (\code{out_format}, \code{animate_move} either needs the \code{convert} tool of the ImageMagick software package (.gif format) or either \code{ffmpeg} from FFmpeg or \code{avconv} from libav (video formats). The command or directory to the convert tool needs to be provided with \code{conv_dir}. Please use \code{\link{get_libraries}} to search for the needed libraries and command/tool directories on your system or to automatically download and install the required software. See \code{\link{get_libraries}} and \code{out_format} and \code{conv_dir} for details.

\code{animate_move} preprocesses your move data depending on the state of the data (see \code{paths_mode} and \code{frames_tres}). \code{animate_move} is based on \code{ggplot2}.
}
\examples{

\dontrun{
#Load move and moveVis packages
library(move)
library(moveVis)

#Get the sample data from the moveVis package (a data.frame)
data("move_data")
move_data$dt <- as.POSIXct(strptime(move_data$dt, "\%Y-\%m-\%d \%H:\%M:\%S", tz = "UTC"))

#Create moveStack object including multiple individuals from the data.frame
#alternatively, use the move package to download data directly from movebank.org
m <- move(move_data$lon, move_data$lat, proj=CRS("+proj=longlat +ellps=WGS84"),
                 time = move_data$dt, animal=move_data$individual, data=move_data)

#Find the command or directory to convert tool of ImageMagick
conv_dir <- get_libraries()

#Specify the output directory, e.g.
out_dir <- paste0(getwd(),"/test")

#Specify some optional appearance variables
img_title <- "Movement of the white stork population at Lake Constance, Germany"
img_sub <- paste0("including individuals ",paste(rownames(idData(m)), collapse=', '))
img_caption <- "Projection: Geographical, WGS84; Sources: Movebank 2013; Bing Maps"

#Call animate_move() with an automatic basemap from Bing, maximum frames at 50
#output format "gif"
animate_move(m, out_dir, conv_dir, tail_elements = 10,
             paths_mode = "true_data", frames_nmax = 50,
             img_caption = img_caption, img_title = img_title,
             img_sub = img_sub, log_level = 1, extent_factor = 0.0002,
             out_format = "gif")
 
#Improve your animation by adding a static points layer
static_data <- data.frame(x = c(8.94,8.943), y = c(47.75,47.753), names = c("Site 1","Site 2"))

#Call animate_move() with "static_data" added
#use another output format, e.g. "mov"
animate_move(m, out_dir, conv_dir, tail_elements = 10,
             paths_mode = "true_data", frames_nmax = 50,
             img_caption = img_caption, img_title = img_title,
             img_sub = img_sub, log_level = 1, extent_factor = 0.0002, 
             static_data=static_data, out_format = "mov")
             
#Try a different paths_mode: Instead of "true_data" use "simple"
#output format "mp4". Longer videos then 100-200 frames should not be GIFs
animate_move(m, out_dir, conv_dir, tail_elements = 10,
             paths_mode = "simple", frames_nmax = 50,
             img_caption = img_caption, img_title = img_title,
             img_sub = img_sub, log_level = 1, extent_factor = 0.0002,
             static_data=static_data, out_format = "mp4")
 
#Use your own basemap by adding lists of rasters and of timestamps
data("basemap_data")
layer = basemap_data[[1]] #this is a example MODIS NDVI dataset
layer_dt = basemap_data[[2]] #this is a corresponding date/time list
 
#Call animate_move with NDVI data as basemap
#layer_type is "gradient", since NDVI values are continuous
animate_move(m, out_dir, conv_dir, tail_elements = 10, layer_type = "gradient",
             paths_mode = "true_data", frames_nmax = 50, layer =layer, layer_dt = layer_dt,
             img_caption = img_caption, img_title = img_title,
             img_sub = img_sub, log_level = 1, extent_factor = 0.0002)
             
#How do your moving individuals interact with their environments?
#Use "stats_create" to create statistics plots
animate_move(m, out_dir, conv_dir, tail_elements = 10, layer_type = "gradient",
             paths_mode = "true_data", frames_nmax = 50, layer =layer, layer_dt = layer_dt,
             img_caption = img_caption, img_title = img_title,
             img_sub = img_sub, log_level = 1, extent_factor = 0.0002,
             stats_create = TRUE)

#If you just want those stats plots, use animate_stats()

#Use "frames_layout" to change the layout of your animation
#e.g. change the position of st_all and st_per
frames_layout <-  rbind(c("map","map","map","st_all","st_leg"),
                        c("map","map","map","st_per","st_leg"))

#or equalize the sizes of spatial map and stats plots
frames_layout <-  rbind(c("map","st_all","st_per","st_leg"))

animate_move(m, out_dir, conv_dir, tail_elements = 10, layer_type = "gradient",
             paths_mode = "true_data", frames_nmax = 50, layer =layer, layer_dt = layer_dt,
             img_caption = img_caption, img_title = img_title,
             img_sub = img_sub, log_level = 1, extent_factor = 0.0002,
             stats_create = TRUE, frames_layout=frames_layout)
}

}
\seealso{
\code{\link{get_libraries}}, \code{\link{animate_stats}}, \code{\link{animate_raster}}
}
\author{
Jakob Schwalb-Willmann
}
