% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/linking.R
\name{bin_data}
\alias{bin_data}
\title{Create bins in variable time series}
\usage{
bin_data(
  data,
  start_time,
  end_time,
  by = c("sec", "min", "hour", "day"),
  fixed = TRUE
)
}
\arguments{
\item{data}{A data frame or tibble containing the time series.}

\item{start_time}{The column name of the start time of the interval, a POSIXt.}

\item{end_time}{The column name of the end time of the interval, a POSIXt.}

\item{by}{A binning specification.}

\item{fixed}{Whether to create fixed bins. If \code{TRUE}, bins will be rounded to, for example,
whole hours or days (depending on \code{by}). If \code{FALSE}, bins will be created based on the
first timestamp.}
}
\value{
A tibble containing the group columns (if any), date, hour (if \code{by = "hour"}), and
the duration in seconds.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}

In time series with variable measurements, an often recurring task is calculating the total time
spent (i.e. the duration) in fixed bins, for example per hour or day. However, this may be
difficult when two subsequent measurements are in different bins or span over multiple bins.
}
\examples{
data <- tibble::tibble(
  participant_id = 1,
  datetime = c(
    "2022-06-21 15:00:00", "2022-06-21 15:55:00",
    "2022-06-21 17:05:00", "2022-06-21 17:10:00"
  ),
  confidence = 100,
  type = "WALKING"
)

# get bins per hour, even if the interval is longer than one hour
data \%>\%
  dplyr::mutate(datetime = as.POSIXct(datetime)) \%>\%
  dplyr::mutate(lead = dplyr::lead(datetime)) \%>\%
  bin_data(
    start_time = datetime,
    end_time = lead,
    by = "hour"
  )

# Alternatively, you can give an integer value to by to create custom-sized
# bins, but only if fixed = FALSE. Not that these bins are not rounded to,
# as in this example 30 minutes, but rather depends on the earliest time
# in the group.
data \%>\%
  dplyr::mutate(datetime = as.POSIXct(datetime)) \%>\%
    dplyr::mutate(lead = dplyr::lead(datetime)) \%>\%
    bin_data(
      start_time = datetime,
      end_time = lead,
      by = 1800L,
      fixed = FALSE
    )

# More complicated data for showcasing grouping:
data <- tibble::tibble(
  participant_id = 1,
  datetime = c(
    "2022-06-21 15:00:00", "2022-06-21 15:55:00",
    "2022-06-21 17:05:00", "2022-06-21 17:10:00"
  ),
  confidence = 100,
  type = c("STILL", "WALKING", "STILL", "WALKING")
)

# binned_intervals also takes into account the prior grouping structure
out <- data \%>\%
  dplyr::mutate(datetime = as.POSIXct(datetime)) \%>\%
  dplyr::mutate(lead = dplyr::lead(datetime)) \%>\%
  dplyr::group_by(participant_id, type) \%>\%
  bin_data(
    start_time = datetime,
    end_time = lead,
    by = "hour"
  )
print(out)

# To get the duration for each bin (note to change the variable names in sum):
purrr::map_dbl(out$bin_data, ~ sum(.x$lead - .x$datetime, na.rm = TRUE))

# Or:
out \%>\%
  tidyr::unnest(bin_data) \%>\%
  dplyr::mutate(duration = .data$lead - .data$datetime) \%>\%
  dplyr::group_by(date, .add = TRUE) \%>\%
  dplyr::summarise(duration = sum(.data$duration), .groups = "drop")
}
\seealso{
\code{\link[mpathsenser]{link_gaps}} for linking gaps to data.
}
