% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/glmer.mp.R
\name{glmer.mp}
\alias{glmer.mp}
\title{Build a multinomial-Poisson GLMM for nominal response data}
\usage{
glmer.mp(formula, data, ...)
}
\arguments{
\item{formula}{A formula object in the style of, e.g., \code{Y ~ X1*X2 + (1|PId)}, where \code{X1} and
\code{X2} are factors or predictors and \code{PId} is a factor serving as a subject identifier. The
response \code{Y} must be of type \code{factor}. See the \code{formula} entry for \code{\link[lme4]{glmer}}.}

\item{data}{A data frame in long-format. See the \code{data} entry for \code{\link[lme4]{glmer}}.}

\item{...}{Additional arguments to be passed to \code{\link[lme4]{glmer}}. Generally, these are
unnecessary but are provided for advanced users. They should not specify \code{formula}, \code{data},
or \code{family} arguments. See \code{\link[lme4]{glmer}} for valid arguments.}
}
\value{
A mixed-effects Poisson regression model of type \code{\link[lme4]{merMod}}, specifically
of \emph{subclass} \code{glmerMod}. See the return value for \code{\link[lme4]{glmer}}.
}
\description{
This function uses the multinomial-Poisson trick to analyze \strong{nominal response} data using a Poisson
generalized linear mixed model (GLMM). The nominal response should be a factor with two or more unordered
categories. The independent variables should have at least one within-subjects factor or numeric predictor.
There also must be a repeated subject identifier to be used as a random factor.
}
\details{
This function should be used for nominal response data with repeated measures. In essence, it provides
for the equivalent of \code{\link[lme4]{glmer}} with \code{family=multinomial}, were that option to
exist. (That option does not exist, which was a key motivation for developing this function.)

For polytomous response data with only between-subjects factors, use \code{\link{glm.mp}} or
\code{\link[nnet]{multinom}}.

Users wishing to verify the correctness of \code{glmer.mp} should compare its \code{\link{Anova.mp}}
results to \code{\link[car]{Anova}} results for models built with \code{\link[lme4]{glmer}} using
\code{family=binomial} for dichotomous responses. The results should generally match, or be very similar.
}
\note{
It is common to receive a \code{boundary (singular) fit} message. This generally can be ignored
provided the test output looks sensible. Less commonly, the procedure can fail to converge, which
can happen when counts of one or more categories are very small or zero in some conditions. In such
cases, any results should be regarded with caution.
}
\examples{
library(car)
library(lme4)
library(lmerTest)

## within-subjects factors (x1,X2) with dichotomous response (Y)
data(ws2, package="multpois")

ws2$PId = factor(ws2$PId)
ws2$Y = factor(ws2$Y)
ws2$X1 = factor(ws2$X1)
ws2$X2 = factor(ws2$X2)
contrasts(ws2$X1) <- "contr.sum"
contrasts(ws2$X2) <- "contr.sum"

m1 = glmer(Y ~ X1*X2 + (1|PId), data=ws2, family=binomial)
Anova(m1, type=3)

m2 = glmer.mp(Y ~ X1*X2 + (1|PId), data=ws2) # compare
Anova.mp(m2, type=3)

## within-subjects factors (x1,X2) with polytomous response (Y)
data(ws3, package="multpois")

ws3$PId = factor(ws3$PId)
ws3$Y = factor(ws3$Y)
ws3$X1 = factor(ws3$X1)
ws3$X2 = factor(ws3$X2)
contrasts(ws3$X1) <- "contr.sum"
contrasts(ws3$X2) <- "contr.sum"

m3 = glmer.mp(Y ~ X1*X2 + (1|PId), data=ws3)
Anova.mp(m3, type=3)

}
\references{
Baker, S.G. (1994). The multinomial-Poisson transformation.
\emph{The Statistician 43} (4), pp. 495-504. \doi{10.2307/2348134}

Chen, Z. and Kuo, L. (2001). A note on the estimation of the
multinomial logit model with random effects. \emph{The American Statistician
55} (2), pp. 89-95. \url{https://www.jstor.org/stable/2685993}

Guimaraes, P. (2004). Understanding the multinomial-Poisson
transformation. \emph{The Stata Journal 4} (3), pp. 265-273.
\url{https://www.stata-journal.com/article.html?article=st0069}

Lee, J.Y.L., Green, P.J.,and Ryan, L.M. (2017). On the “Poisson
trick” and its extensions for fitting multinomial regression models. \emph{arXiv
preprint} available at \doi{10.48550/arXiv.1707.08538}
}
\seealso{
\code{\link[=Anova.mp]{Anova.mp()}}, \code{\link[=glmer.mp.con]{glmer.mp.con()}}, \code{\link[=glm.mp]{glm.mp()}}, \code{\link[=glm.mp.con]{glm.mp.con()}}, \code{\link[lme4:glmer]{lme4::glmer()}}
}
\author{
Jacob O. Wobbrock
}
