#' Simulate Pareto random vectors
#'
#' \code{simulPareto} provides \code{n} replicates of a multivariate Pareto distribution
#' for the semi-variogram \code{vario}.
#'
#' The algorithm used here is based on the spectral representation of the Brown--Resnick
#' model as described in Dombry et al. (2015). It provides \code{n} replicates conditioned
#' that \code{mean(x) > 1} on the unit Frechet scale.
#'
#' @param n The number of replicates desired
#' @param loc A matrix of coordinates as given by \code{expand.grid()}.
#' @param vario A semi-variogram function.
#' @param nCores The number of cores used for the computation
#' @param cl A cluster instance as created by \code{makeCluster} of the \code{parallel} package. Make sure
#' the random number generator has been properly initialized with \code{clusterSetRNGStream()}.
#' @return A list of \code{n} random vectors drawn from a mutlivariate Pareto distribution with semi-variogram \code{vario}.
#' @examples
#' #Define variogram function
#' vario <- function(h){
#'    1 / 2 * norm(h,type = "2")^1.5
#' }
#'
#' #Define locations
#' loc <- expand.grid(1:4, 1:4)
#'
#' #Simulate data
#' obs <- simulPareto(100, loc, vario)
#' @export


simulPareto <- function(n, loc, vario, nCores = 1, cl = NULL){

  if(class(loc) != "data.frame") {
    stop('loc must be the data frame of coordinates as generated by expand.grid()')
  }

  dim <- nrow(loc)

  if(!is.numeric(nCores) || nCores < 1) {
    stop('nCores must a positive number of cores to use for parallel computing.')
  }
  if(nCores > 1 && length(grep("cluster",class(cl))) > 0) {
    stop('For parallel computation, cl must an cluster created by makeCluster of the package parallel.')
  }

  gamma <- tryCatch({
    dists <- lapply(1:ncol(loc), function(i) {
      outer(loc[,i],loc[,i], "-")
    })

    computeVarMat <- sapply(1:length(dists[[1]]), function(i){
      h <- rep(0,ncol(loc))
      for(j in 1:ncol(loc)){
        h[j] = dists[[j]][i]
      }
      vario(h)
    })
    matrix(computeVarMat, dim, dim)
  }, warning = function(war) {
    war
  }, error = function(err) {
    stop('The semi-variogram provided is not valide for the provided locations.')
  })

  k <- sample(1:dim, n, replace = TRUE)

  simFun <- function(i) {
    mean <- - gamma[-k[i],k[i]]
    cov  <- (outer(gamma[-k[i],k[i]],gamma[-k[i],k[i]], "+") - (gamma[-k[i],-k[i]]))


    paretoProcess <- rep(1,dim)
    paretoProcess[-k[i]] <- pmax(.Machine$double.xmin,exp(MASS::mvrnorm(n = 1, mu = mean, Sigma = cov)))
    intensity <- evd::rgpd(1, loc=1, scale=1, shape=1)
    (intensity * paretoProcess / mean(paretoProcess))
  }

  if(nCores > 1){
    sims <- parallel::parLapply(cl, 1:n, simFun)
  } else {
    sims <- lapply(1:n, simFun)
  }

  return(sims)
}
