% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/censored_likelihood.R
\name{censoredLikelihoodBR}
\alias{censoredLikelihoodBR}
\alias{censoredLikelihood}
\title{Censored log-likelihood function for the Brown--Resnick model.}
\usage{
censoredLikelihoodBR(
  obs,
  loc,
  vario,
  u,
  p = 499L,
  vec = NULL,
  nCores = 1L,
  cl = NULL,
  likelihood = "mgp",
  ntot = NULL,
  ...
)

censoredLikelihood(
  obs,
  loc,
  vario,
  u,
  p = 499L,
  vec = NULL,
  nCores = 1L,
  cl = NULL
)
}
\arguments{
\item{obs}{List of vectors for which at least one component exceeds a high threshold.}

\item{loc}{Matrix of coordinates as given by \code{expand.grid()}.}

\item{vario}{Semi-variogram function taking a vector of coordinates as input.}

\item{u}{Vector of threshold under which to censor components.}

\item{p}{Number of samples used for quasi-Monte Carlo estimation. Must be a prime number.}

\item{vec}{Generating vector for the quasi-Monte Carlo procedure. For a given prime \code{p} and dimension,
can be computed using \code{genVecQMC}.}

\item{nCores}{Number of cores used for the computation}

\item{cl}{Cluster instance as created by \code{makeCluster} of the \code{parallel} package.}

\item{likelihood}{vector of strings specifying the contribution. Either \code{"mgp"} for multivariate generalized Pareto,
\code{"poisson"} for a Poisson contribution for the observations falling below or \code{"binom"} for a binomial contribution.}

\item{ntot}{integer number of observations below and above the threshold, to be used with Poisson or binomial likelihood}

\item{...}{Additional arguments passed to Cpp routine.}
}
\value{
Negative censored log-likelihood for the set of observations \code{obs} and semi-variogram \code{vario} with \code{attributes}  \code{exponentMeasure} for all of the \code{likelihood} type selected, in the order \code{"mgp"}, \code{"poisson"}, \code{"binom"}.
}
\description{
Compute the peaks-over-threshold censored negative log-likelihood function for the Brown--Resnick model.
}
\details{
The function computes the censored negative log-likelihood function based on the representation
developed by Wadsworth et al. (2014) and Engelke et al. (2015). Margins must have been
standardized first, for instance to the unit Frechet scale.
}
\examples{
#Define semi-variogram function
vario <- function(h){
   0.5 * norm(h, type = "2")^1.5
}

#Define locations
loc <- expand.grid(1:4, 1:4)

#Simulate data
obs <- simulPareto(1000, loc, vario)

#Evaluate risk functional
maxima <- sapply(obs, max)
thres <- quantile(maxima, 0.9)

#Select exceedances
exceedances <- obs[maxima > thres]

#Compute generating vector
p <- 499
latticeRule <- genVecQMC(p, (nrow(loc) - 1))
primeP <- latticeRule$primeP
vec <- latticeRule$genVec


#Compute log-likelihood function
censoredLikelihoodBR(obs = exceedances, loc = loc, vario = vario,
 u = thres, p = primeP, vec = vec, ntot = 1000)
}
\references{
Wadsworth, J. L. and J. A. Tawn (2014). Efficient inference for spatial extreme value
 processes associated to log-Gaussian random functions. Biometrika, 101(1), 1-15.

Asadi, P., Davison A. C. and S. Engelke (2015). Extremes on River Networks.
 Annals of Applied Statistics, 9(4), 2023-2050.
}
\author{
Raphael de Fondeville
}
